import Resource from './resource';
import RENDERBUFFER_FORMATS from './renderbuffer-formats';
import { isWebGL2 } from '@luma.gl/gltools';
import { assert } from '../utils/assert';

function isFormatSupported(gl, format, formats) {
  const info = formats[format];

  if (!info) {
    return false;
  }

  const value = isWebGL2(gl) ? info.gl2 || info.gl1 : info.gl1;

  if (typeof value === 'string') {
    return gl.getExtension(value);
  }

  return value;
}

export default class Renderbuffer extends Resource {
  get [Symbol.toStringTag]() {
    return 'Renderbuffer';
  }

  static isSupported(gl) {
    let {
      format
    } = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {
      format: null
    };
    return !format || isFormatSupported(gl, format, RENDERBUFFER_FORMATS);
  }

  static getSamplesForFormat(gl, _ref) {
    let {
      format
    } = _ref;
    return gl.getInternalformatParameter(36161, format, 32937);
  }

  constructor(gl) {
    let opts = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    super(gl, opts);
    this.initialize(opts);
    Object.seal(this);
  }

  initialize(_ref2) {
    let {
      format,
      width = 1,
      height = 1,
      samples = 0
    } = _ref2;
    assert(format, 'Needs format');

    this._trackDeallocatedMemory();

    this.gl.bindRenderbuffer(36161, this.handle);

    if (samples !== 0 && isWebGL2(this.gl)) {
      this.gl.renderbufferStorageMultisample(36161, samples, format, width, height);
    } else {
      this.gl.renderbufferStorage(36161, format, width, height);
    }

    this.format = format;
    this.width = width;
    this.height = height;
    this.samples = samples;

    this._trackAllocatedMemory(this.width * this.height * (this.samples || 1) * RENDERBUFFER_FORMATS[this.format].bpp);

    return this;
  }

  resize(_ref3) {
    let {
      width,
      height
    } = _ref3;

    if (width !== this.width || height !== this.height) {
      return this.initialize({
        width,
        height,
        format: this.format,
        samples: this.samples
      });
    }

    return this;
  }

  _createHandle() {
    return this.gl.createRenderbuffer();
  }

  _deleteHandle() {
    this.gl.deleteRenderbuffer(this.handle);

    this._trackDeallocatedMemory();
  }

  _bindHandle(handle) {
    this.gl.bindRenderbuffer(36161, handle);
  }

  _syncHandle(handle) {
    this.format = this.getParameter(36164);
    this.width = this.getParameter(36162);
    this.height = this.getParameter(36163);
    this.samples = this.getParameter(36011);
  }

  _getParameter(pname) {
    this.gl.bindRenderbuffer(36161, this.handle);
    const value = this.gl.getRenderbufferParameter(36161, pname);
    return value;
  }

}
//# sourceMappingURL=renderbuffer.js.map