import { isWebGL2, assertWebGL2Context, withParameters, log } from '@luma.gl/gltools';
import Resource from './resource';
import Buffer from './buffer';
import { TEXTURE_FORMATS, DATA_FORMAT_CHANNELS, TYPE_SIZES, isFormatSupported, isLinearFilteringSupported } from './texture-formats';
import { assert } from '../utils/assert';
import { uid, isPowerOfTwo } from '../utils/utils';
const NPOT_MIN_FILTERS = [9729, 9728];

const WebGLBuffer = globalThis.WebGLBuffer || function WebGLBuffer() {};

export default class Texture extends Resource {
  get [Symbol.toStringTag]() {
    return 'Texture';
  }

  static isSupported(gl) {
    let opts = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
    const {
      format,
      linearFiltering
    } = opts;
    let supported = true;

    if (format) {
      supported = supported && isFormatSupported(gl, format);
      supported = supported && (!linearFiltering || isLinearFilteringSupported(gl, format));
    }

    return supported;
  }

  constructor(gl, props) {
    const {
      id = uid('texture'),
      handle,
      target
    } = props;
    super(gl, {
      id,
      handle
    });
    this.target = target;
    this.textureUnit = undefined;
    this.loaded = false;
    this.width = undefined;
    this.height = undefined;
    this.depth = undefined;
    this.format = undefined;
    this.type = undefined;
    this.dataFormat = undefined;
    this.border = undefined;
    this.textureUnit = undefined;
    this.mipmaps = undefined;
  }

  toString() {
    return "Texture(".concat(this.id, ",").concat(this.width, "x").concat(this.height, ")");
  }

  initialize() {
    let props = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    let data = props.data;

    if (data instanceof Promise) {
      data.then(resolvedImageData => this.initialize(Object.assign({}, props, {
        pixels: resolvedImageData,
        data: resolvedImageData
      })));
      return this;
    }

    const isVideo = typeof HTMLVideoElement !== 'undefined' && data instanceof HTMLVideoElement;

    if (isVideo && data.readyState < HTMLVideoElement.HAVE_METADATA) {
      this._video = null;
      data.addEventListener('loadeddata', () => this.initialize(props));
      return this;
    }

    const {
      pixels = null,
      format = 6408,
      border = 0,
      recreate = false,
      parameters = {},
      pixelStore = {},
      textureUnit = undefined
    } = props;

    if (!data) {
      data = pixels;
    }

    let {
      width,
      height,
      dataFormat,
      type,
      compressed = false,
      mipmaps = true
    } = props;
    const {
      depth = 0
    } = props;
    ({
      width,
      height,
      compressed,
      dataFormat,
      type
    } = this._deduceParameters({
      format,
      type,
      dataFormat,
      compressed,
      data,
      width,
      height
    }));
    this.width = width;
    this.height = height;
    this.depth = depth;
    this.format = format;
    this.type = type;
    this.dataFormat = dataFormat;
    this.border = border;
    this.textureUnit = textureUnit;

    if (Number.isFinite(this.textureUnit)) {
      this.gl.activeTexture(33984 + this.textureUnit);
      this.gl.bindTexture(this.target, this.handle);
    }

    if (mipmaps && this._isNPOT()) {
      log.warn("texture: ".concat(this, " is Non-Power-Of-Two, disabling mipmaping"))();
      mipmaps = false;

      this._updateForNPOT(parameters);
    }

    this.mipmaps = mipmaps;
    this.setImageData({
      data,
      width,
      height,
      depth,
      format,
      type,
      dataFormat,
      border,
      mipmaps,
      parameters: pixelStore,
      compressed
    });

    if (mipmaps) {
      this.generateMipmap();
    }

    this.setParameters(parameters);

    if (recreate) {
      this.data = data;
    }

    if (isVideo) {
      this._video = {
        video: data,
        parameters,
        lastTime: data.readyState >= HTMLVideoElement.HAVE_CURRENT_DATA ? data.currentTime : -1
      };
    }

    return this;
  }

  update() {
    if (this._video) {
      const {
        video,
        parameters,
        lastTime
      } = this._video;

      if (lastTime === video.currentTime || video.readyState < HTMLVideoElement.HAVE_CURRENT_DATA) {
        return;
      }

      this.setSubImageData({
        data: video,
        parameters
      });

      if (this.mipmaps) {
        this.generateMipmap();
      }

      this._video.lastTime = video.currentTime;
    }
  }

  resize(_ref) {
    let {
      height,
      width,
      mipmaps = false
    } = _ref;

    if (width !== this.width || height !== this.height) {
      return this.initialize({
        width,
        height,
        format: this.format,
        type: this.type,
        dataFormat: this.dataFormat,
        border: this.border,
        mipmaps
      });
    }

    return this;
  }

  generateMipmap() {
    let params = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

    if (this._isNPOT()) {
      log.warn("texture: ".concat(this, " is Non-Power-Of-Two, disabling mipmaping"))();
      return this;
    }

    this.mipmaps = true;
    this.gl.bindTexture(this.target, this.handle);
    withParameters(this.gl, params, () => {
      this.gl.generateMipmap(this.target);
    });
    this.gl.bindTexture(this.target, null);
    return this;
  }

  setImageData(options) {
    this._trackDeallocatedMemory('Texture');

    const {
      target = this.target,
      pixels = null,
      level = 0,
      format = this.format,
      border = this.border,
      offset = 0,
      parameters = {}
    } = options;
    let {
      data = null,
      type = this.type,
      width = this.width,
      height = this.height,
      dataFormat = this.dataFormat,
      compressed = false
    } = options;

    if (!data) {
      data = pixels;
    }

    ({
      type,
      dataFormat,
      compressed,
      width,
      height
    } = this._deduceParameters({
      format,
      type,
      dataFormat,
      compressed,
      data,
      width,
      height
    }));
    const {
      gl
    } = this;
    gl.bindTexture(this.target, this.handle);
    let dataType = null;
    ({
      data,
      dataType
    } = this._getDataType({
      data,
      compressed
    }));
    let gl2;
    let compressedTextureSize = 0;
    withParameters(this.gl, parameters, () => {
      switch (dataType) {
        case 'null':
          gl.texImage2D(target, level, format, width, height, border, dataFormat, type, data);
          break;

        case 'typed-array':
          gl.texImage2D(target, level, format, width, height, border, dataFormat, type, data, offset);
          break;

        case 'buffer':
          gl2 = assertWebGL2Context(gl);
          gl2.bindBuffer(35052, data.handle || data);
          gl2.texImage2D(target, level, format, width, height, border, dataFormat, type, offset);
          gl2.bindBuffer(35052, null);
          break;

        case 'browser-object':
          if (isWebGL2(gl)) {
            gl.texImage2D(target, level, format, width, height, border, dataFormat, type, data);
          } else {
            gl.texImage2D(target, level, format, dataFormat, type, data);
          }

          break;

        case 'compressed':
          for (const [levelIndex, levelData] of data.entries()) {
            gl.compressedTexImage2D(target, levelIndex, levelData.format, levelData.width, levelData.height, border, levelData.data);
            compressedTextureSize += levelData.levelSize;
          }

          break;

        default:
          assert(false, 'Unknown image data type');
      }
    });

    if (dataType === 'compressed') {
      this._trackAllocatedMemory(compressedTextureSize, 'Texture');
    } else if (data && data.byteLength) {
      this._trackAllocatedMemory(data.byteLength, 'Texture');
    } else {
      const channels = DATA_FORMAT_CHANNELS[this.dataFormat] || 4;
      const channelSize = TYPE_SIZES[this.type] || 1;

      this._trackAllocatedMemory(this.width * this.height * channels * channelSize, 'Texture');
    }

    this.loaded = true;
    return this;
  }

  setSubImageData(_ref2) {
    let {
      target = this.target,
      pixels = null,
      data = null,
      x = 0,
      y = 0,
      width = this.width,
      height = this.height,
      level = 0,
      format = this.format,
      type = this.type,
      dataFormat = this.dataFormat,
      compressed = false,
      offset = 0,
      border = this.border,
      parameters = {}
    } = _ref2;
    ({
      type,
      dataFormat,
      compressed,
      width,
      height
    } = this._deduceParameters({
      format,
      type,
      dataFormat,
      compressed,
      data,
      width,
      height
    }));
    assert(this.depth === 0, 'texSubImage not supported for 3D textures');

    if (!data) {
      data = pixels;
    }

    if (data && data.data) {
      const ndarray = data;
      data = ndarray.data;
      width = ndarray.shape[0];
      height = ndarray.shape[1];
    }

    if (data instanceof Buffer) {
      data = data.handle;
    }

    this.gl.bindTexture(this.target, this.handle);
    withParameters(this.gl, parameters, () => {
      if (compressed) {
        this.gl.compressedTexSubImage2D(target, level, x, y, width, height, format, data);
      } else if (data === null) {
        this.gl.texSubImage2D(target, level, x, y, width, height, dataFormat, type, null);
      } else if (ArrayBuffer.isView(data)) {
        this.gl.texSubImage2D(target, level, x, y, width, height, dataFormat, type, data, offset);
      } else if (data instanceof WebGLBuffer) {
        const gl2 = assertWebGL2Context(this.gl);
        gl2.bindBuffer(35052, data);
        gl2.texSubImage2D(target, level, x, y, width, height, dataFormat, type, offset);
        gl2.bindBuffer(35052, null);
      } else if (isWebGL2(this.gl)) {
        const gl2 = assertWebGL2Context(this.gl);
        gl2.texSubImage2D(target, level, x, y, width, height, dataFormat, type, data);
      } else {
        this.gl.texSubImage2D(target, level, x, y, dataFormat, type, data);
      }
    });
    this.gl.bindTexture(this.target, null);
  }

  copyFramebuffer() {
    let opts = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    log.error('Texture.copyFramebuffer({...}) is no logner supported, use copyToTexture(source, target, opts})')();
    return null;
  }

  getActiveUnit() {
    return this.gl.getParameter(34016) - 33984;
  }

  bind() {
    let textureUnit = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : this.textureUnit;
    const {
      gl
    } = this;

    if (textureUnit !== undefined) {
      this.textureUnit = textureUnit;
      gl.activeTexture(33984 + textureUnit);
    }

    gl.bindTexture(this.target, this.handle);
    return textureUnit;
  }

  unbind() {
    let textureUnit = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : this.textureUnit;
    const {
      gl
    } = this;

    if (textureUnit !== undefined) {
      this.textureUnit = textureUnit;
      gl.activeTexture(33984 + textureUnit);
    }

    gl.bindTexture(this.target, null);
    return textureUnit;
  }

  _getDataType(_ref3) {
    let {
      data,
      compressed = false
    } = _ref3;

    if (compressed) {
      return {
        data,
        dataType: 'compressed'
      };
    }

    if (data === null) {
      return {
        data,
        dataType: 'null'
      };
    }

    if (ArrayBuffer.isView(data)) {
      return {
        data,
        dataType: 'typed-array'
      };
    }

    if (data instanceof Buffer) {
      return {
        data: data.handle,
        dataType: 'buffer'
      };
    }

    if (data instanceof WebGLBuffer) {
      return {
        data,
        dataType: 'buffer'
      };
    }

    return {
      data,
      dataType: 'browser-object'
    };
  }

  _deduceParameters(opts) {
    const {
      format,
      data
    } = opts;
    let {
      width,
      height,
      dataFormat,
      type,
      compressed
    } = opts;
    const textureFormat = TEXTURE_FORMATS[format];
    dataFormat = dataFormat || textureFormat && textureFormat.dataFormat;
    type = type || textureFormat && textureFormat.types[0];
    compressed = compressed || textureFormat && textureFormat.compressed;
    ({
      width,
      height
    } = this._deduceImageSize(data, width, height));
    return {
      dataFormat,
      type,
      compressed,
      width,
      height,
      format,
      data
    };
  }

  _deduceImageSize(data, width, height) {
    let size;

    if (typeof ImageData !== 'undefined' && data instanceof ImageData) {
      size = {
        width: data.width,
        height: data.height
      };
    } else if (typeof HTMLImageElement !== 'undefined' && data instanceof HTMLImageElement) {
      size = {
        width: data.naturalWidth,
        height: data.naturalHeight
      };
    } else if (typeof HTMLCanvasElement !== 'undefined' && data instanceof HTMLCanvasElement) {
      size = {
        width: data.width,
        height: data.height
      };
    } else if (typeof ImageBitmap !== 'undefined' && data instanceof ImageBitmap) {
      size = {
        width: data.width,
        height: data.height
      };
    } else if (typeof HTMLVideoElement !== 'undefined' && data instanceof HTMLVideoElement) {
      size = {
        width: data.videoWidth,
        height: data.videoHeight
      };
    } else if (!data) {
      size = {
        width: width >= 0 ? width : 1,
        height: height >= 0 ? height : 1
      };
    } else {
      size = {
        width,
        height
      };
    }

    assert(size, 'Could not deduced texture size');
    assert(width === undefined || size.width === width, 'Deduced texture width does not match supplied width');
    assert(height === undefined || size.height === height, 'Deduced texture height does not match supplied height');
    return size;
  }

  _createHandle() {
    return this.gl.createTexture();
  }

  _deleteHandle() {
    this.gl.deleteTexture(this.handle);

    this._trackDeallocatedMemory('Texture');
  }

  _getParameter(pname) {
    switch (pname) {
      case 4096:
        return this.width;

      case 4097:
        return this.height;

      default:
        this.gl.bindTexture(this.target, this.handle);
        const value = this.gl.getTexParameter(this.target, pname);
        this.gl.bindTexture(this.target, null);
        return value;
    }
  }

  _setParameter(pname, param) {
    this.gl.bindTexture(this.target, this.handle);
    param = this._getNPOTParam(pname, param);

    switch (pname) {
      case 33082:
      case 33083:
        this.gl.texParameterf(this.handle, pname, param);
        break;

      case 4096:
      case 4097:
        assert(false);
        break;

      default:
        this.gl.texParameteri(this.target, pname, param);
        break;
    }

    this.gl.bindTexture(this.target, null);
    return this;
  }

  _isNPOT() {
    if (isWebGL2(this.gl)) {
      return false;
    }

    if (!this.width || !this.height) {
      return false;
    }

    return !isPowerOfTwo(this.width) || !isPowerOfTwo(this.height);
  }

  _updateForNPOT(parameters) {
    if (parameters[this.gl.TEXTURE_MIN_FILTER] === undefined) {
      parameters[this.gl.TEXTURE_MIN_FILTER] = this.gl.LINEAR;
    }

    if (parameters[this.gl.TEXTURE_WRAP_S] === undefined) {
      parameters[this.gl.TEXTURE_WRAP_S] = this.gl.CLAMP_TO_EDGE;
    }

    if (parameters[this.gl.TEXTURE_WRAP_T] === undefined) {
      parameters[this.gl.TEXTURE_WRAP_T] = this.gl.CLAMP_TO_EDGE;
    }
  }

  _getNPOTParam(pname, param) {
    if (this._isNPOT()) {
      switch (pname) {
        case 10241:
          if (NPOT_MIN_FILTERS.indexOf(param) === -1) {
            param = 9729;
          }

          break;

        case 10242:
        case 10243:
          if (param !== 33071) {
            param = 33071;
          }

          break;

        default:
          break;
      }
    }

    return param;
  }

}
//# sourceMappingURL=texture.js.map