import { log } from '@luma.gl/gltools';
import Framebuffer from './framebuffer';
import Renderbuffer from './renderbuffer';
import Texture from './texture';
import { assert } from '../utils/assert';
const UNIFORM_SETTERS = {
  [5126]: getArraySetter.bind(null, 'uniform1fv', toFloatArray, 1, setVectorUniform),
  [35664]: getArraySetter.bind(null, 'uniform2fv', toFloatArray, 2, setVectorUniform),
  [35665]: getArraySetter.bind(null, 'uniform3fv', toFloatArray, 3, setVectorUniform),
  [35666]: getArraySetter.bind(null, 'uniform4fv', toFloatArray, 4, setVectorUniform),
  [5124]: getArraySetter.bind(null, 'uniform1iv', toIntArray, 1, setVectorUniform),
  [35667]: getArraySetter.bind(null, 'uniform2iv', toIntArray, 2, setVectorUniform),
  [35668]: getArraySetter.bind(null, 'uniform3iv', toIntArray, 3, setVectorUniform),
  [35669]: getArraySetter.bind(null, 'uniform4iv', toIntArray, 4, setVectorUniform),
  [35670]: getArraySetter.bind(null, 'uniform1iv', toIntArray, 1, setVectorUniform),
  [35671]: getArraySetter.bind(null, 'uniform2iv', toIntArray, 2, setVectorUniform),
  [35672]: getArraySetter.bind(null, 'uniform3iv', toIntArray, 3, setVectorUniform),
  [35673]: getArraySetter.bind(null, 'uniform4iv', toIntArray, 4, setVectorUniform),
  [35674]: getArraySetter.bind(null, 'uniformMatrix2fv', toFloatArray, 4, setMatrixUniform),
  [35675]: getArraySetter.bind(null, 'uniformMatrix3fv', toFloatArray, 9, setMatrixUniform),
  [35676]: getArraySetter.bind(null, 'uniformMatrix4fv', toFloatArray, 16, setMatrixUniform),
  [35678]: getSamplerSetter,
  [35680]: getSamplerSetter,
  [5125]: getArraySetter.bind(null, 'uniform1uiv', toUIntArray, 1, setVectorUniform),
  [36294]: getArraySetter.bind(null, 'uniform2uiv', toUIntArray, 2, setVectorUniform),
  [36295]: getArraySetter.bind(null, 'uniform3uiv', toUIntArray, 3, setVectorUniform),
  [36296]: getArraySetter.bind(null, 'uniform4uiv', toUIntArray, 4, setVectorUniform),
  [35685]: getArraySetter.bind(null, 'uniformMatrix2x3fv', toFloatArray, 6, setMatrixUniform),
  [35686]: getArraySetter.bind(null, 'uniformMatrix2x4fv', toFloatArray, 8, setMatrixUniform),
  [35687]: getArraySetter.bind(null, 'uniformMatrix3x2fv', toFloatArray, 6, setMatrixUniform),
  [35688]: getArraySetter.bind(null, 'uniformMatrix3x4fv', toFloatArray, 12, setMatrixUniform),
  [35689]: getArraySetter.bind(null, 'uniformMatrix4x2fv', toFloatArray, 8, setMatrixUniform),
  [35690]: getArraySetter.bind(null, 'uniformMatrix4x3fv', toFloatArray, 12, setMatrixUniform),
  [35678]: getSamplerSetter,
  [35680]: getSamplerSetter,
  [35679]: getSamplerSetter,
  [35682]: getSamplerSetter,
  [36289]: getSamplerSetter,
  [36292]: getSamplerSetter,
  [36293]: getSamplerSetter,
  [36298]: getSamplerSetter,
  [36299]: getSamplerSetter,
  [36300]: getSamplerSetter,
  [36303]: getSamplerSetter,
  [36306]: getSamplerSetter,
  [36307]: getSamplerSetter,
  [36308]: getSamplerSetter,
  [36311]: getSamplerSetter
};
const FLOAT_ARRAY = {};
const INT_ARRAY = {};
const UINT_ARRAY = {};
const array1 = [0];

function toTypedArray(value, uniformLength, Type, cache) {
  if (uniformLength === 1 && typeof value === 'boolean') {
    value = value ? 1 : 0;
  }

  if (Number.isFinite(value)) {
    array1[0] = value;
    value = array1;
  }

  const length = value.length;

  if (length % uniformLength) {
    log.warn("Uniform size should be multiples of ".concat(uniformLength), value)();
  }

  if (value instanceof Type) {
    return value;
  }

  let result = cache[length];

  if (!result) {
    result = new Type(length);
    cache[length] = result;
  }

  for (let i = 0; i < length; i++) {
    result[i] = value[i];
  }

  return result;
}

function toFloatArray(value, uniformLength) {
  return toTypedArray(value, uniformLength, Float32Array, FLOAT_ARRAY);
}

function toIntArray(value, uniformLength) {
  return toTypedArray(value, uniformLength, Int32Array, INT_ARRAY);
}

function toUIntArray(value, uniformLength) {
  return toTypedArray(value, uniformLength, Uint32Array, UINT_ARRAY);
}

export function getUniformSetter(gl, location, info) {
  const setter = UNIFORM_SETTERS[info.type];

  if (!setter) {
    throw new Error("Unknown GLSL uniform type ".concat(info.type));
  }

  return setter().bind(null, gl, location);
}
export function parseUniformName(name) {
  if (name[name.length - 1] !== ']') {
    return {
      name,
      length: 1,
      isArray: false
    };
  }

  const UNIFORM_NAME_REGEXP = /([^[]*)(\[[0-9]+\])?/;
  const matches = name.match(UNIFORM_NAME_REGEXP);

  if (!matches || matches.length < 2) {
    throw new Error("Failed to parse GLSL uniform name ".concat(name));
  }

  return {
    name: matches[1],
    length: matches[2] || 1,
    isArray: Boolean(matches[2])
  };
}
export function checkUniformValues(uniforms, source, uniformMap) {
  for (const uniformName in uniforms) {
    const value = uniforms[uniformName];
    const shouldCheck = !uniformMap || Boolean(uniformMap[uniformName]);

    if (shouldCheck && !checkUniformValue(value)) {
      source = source ? "".concat(source, " ") : '';
      console.error("".concat(source, " Bad uniform ").concat(uniformName), value);
      throw new Error("".concat(source, " Bad uniform ").concat(uniformName));
    }
  }

  return true;
}

function checkUniformValue(value) {
  if (Array.isArray(value) || ArrayBuffer.isView(value)) {
    return checkUniformArray(value);
  }

  if (isFinite(value)) {
    return true;
  } else if (value === true || value === false) {
    return true;
  } else if (value instanceof Texture) {
    return true;
  } else if (value instanceof Renderbuffer) {
    return true;
  } else if (value instanceof Framebuffer) {
    return Boolean(value.texture);
  }

  return false;
}

export function copyUniform(uniforms, key, value) {
  if (Array.isArray(value) || ArrayBuffer.isView(value)) {
    if (uniforms[key]) {
      const dest = uniforms[key];

      for (let i = 0, len = value.length; i < len; ++i) {
        dest[i] = value[i];
      }
    } else {
      uniforms[key] = value.slice();
    }
  } else {
    uniforms[key] = value;
  }
}

function checkUniformArray(value) {
  if (value.length === 0) {
    return false;
  }

  const checkLength = Math.min(value.length, 16);

  for (let i = 0; i < checkLength; ++i) {
    if (!Number.isFinite(value[i])) {
      return false;
    }
  }

  return true;
}

function getSamplerSetter() {
  let cache = null;
  return (gl, location, value) => {
    const update = cache !== value;

    if (update) {
      gl.uniform1i(location, value);
      cache = value;
    }

    return update;
  };
}

function getArraySetter(functionName, toArray, size, uniformSetter) {
  let cache = null;
  let cacheLength = null;
  return (gl, location, value) => {
    const arrayValue = toArray(value, size);
    const length = arrayValue.length;
    let update = false;

    if (cache === null) {
      cache = new Float32Array(length);
      cacheLength = length;
      update = true;
    } else {
      assert(cacheLength === length, 'Uniform length cannot change.');

      for (let i = 0; i < length; ++i) {
        if (arrayValue[i] !== cache[i]) {
          update = true;
          break;
        }
      }
    }

    if (update) {
      uniformSetter(gl, functionName, location, arrayValue);
      cache.set(arrayValue);
    }

    return update;
  };
}

function setVectorUniform(gl, functionName, location, value) {
  gl[functionName](location, value);
}

function setMatrixUniform(gl, functionName, location, value) {
  gl[functionName](location, false, value);
}
//# sourceMappingURL=uniforms.js.map