// luma.gl
// SPDX-License-Identifier: MIT
// Copyright (c) vis.gl contributors
import { Framebuffer } from '@luma.gl/core';
import { GL } from '@luma.gl/constants';
import { getGLTypeFromTypedArray, getTypedArrayFromGLType } from "./typed-array-utils.js";
import { glFormatToComponents, glTypeToBytes } from "./format-utils.js";
import { WEBGLTexture } from "../resources/webgl-texture.js";
import { withGLParameters } from "../../context/state-tracker/with-parameters.js";
/** A "border" parameter is required in many WebGL texture APIs, but must always be 0... */
const BORDER = 0;
/**
 * Initializes a texture memory space
 * Clear all the textures and mip levels of a two-dimensional or array texture at the same time.
 * On some implementations faster than repeatedly setting levels
 *
 * @note From WebGL 2 spec section 3.7.6:
 * @see https://registry.khronos.org/webgl/specs/latest/2.0/
 * - The image contents are set as if a buffer of sufficient size initialized to 0 would be passed to each level's texImage2D/3D
 * - texStorage2D should be considered a preferred alternative to texImage2D. It may have lower memory costs than texImage2D in some implementations.
 * - Once texStorage*D has been called, the texture is immutable and can only be updated with texSubImage*(), not texImage()
 */
export function initializeTextureStorage(gl, levels, options) {
    const { dimension, width, height, depth = 0 } = options;
    const { glInternalFormat } = options;
    const glTarget = options.glTarget; // getWebGLCubeFaceTarget(options.glTarget, dimension, depth);
    switch (dimension) {
        case '2d-array':
        case '3d':
            gl.texStorage3D(glTarget, levels, glInternalFormat, width, height, depth);
            break;
        default:
            gl.texStorage2D(glTarget, levels, glInternalFormat, width, height);
    }
}
/**
 * Copy a region of compressed data from a GPU memory buffer into this texture.
 */
export function copyExternalImageToMipLevel(gl, handle, image, options) {
    const { width, height } = options;
    const { dimension, depth = 0, mipLevel = 0 } = options;
    const { x = 0, y = 0, z = 0 } = options;
    const { glFormat, glType } = options;
    const glTarget = getWebGLCubeFaceTarget(options.glTarget, dimension, depth);
    const glParameters = options.flipY ? { [37440]: true } : {};
    withGLParameters(gl, glParameters, () => {
        switch (dimension) {
            case '2d-array':
            case '3d':
                gl.bindTexture(glTarget, handle);
                // prettier-ignore
                gl.texSubImage3D(glTarget, mipLevel, x, y, z, width, height, depth, glFormat, glType, image);
                gl.bindTexture(glTarget, null);
                break;
            case '2d':
            case 'cube':
                gl.bindTexture(glTarget, handle);
                // prettier-ignore
                gl.texSubImage2D(glTarget, mipLevel, x, y, width, height, glFormat, glType, image);
                gl.bindTexture(glTarget, null);
                break;
            default:
                throw new Error(dimension);
        }
    });
}
/**
 * Copy a region of data from a CPU memory buffer into this texture.
 */
export function copyCPUDataToMipLevel(gl, typedArray, options) {
    const { dimension, width, height, depth = 0, mipLevel = 0, byteOffset = 0 } = options;
    const { x = 0, y = 0, z = 0 } = options;
    const { glFormat, glType, compressed } = options;
    const glTarget = getWebGLCubeFaceTarget(options.glTarget, dimension, depth);
    // gl.bindTexture(glTarget, null);
    switch (dimension) {
        case '2d-array':
        case '3d':
            if (compressed) {
                // prettier-ignore
                gl.compressedTexSubImage3D(glTarget, mipLevel, x, y, z, width, height, depth, glFormat, typedArray, byteOffset); // , byteLength
            }
            else {
                // prettier-ignore
                gl.texSubImage3D(glTarget, mipLevel, x, y, z, width, height, depth, glFormat, glType, typedArray, byteOffset); // , byteLength
            }
            break;
        case '2d':
        case 'cube':
            if (compressed) {
                // prettier-ignore
                gl.compressedTexSubImage2D(glTarget, mipLevel, x, y, width, height, glFormat, typedArray, byteOffset); // , byteLength
            }
            else {
                // prettier-ignore
                gl.texSubImage2D(glTarget, mipLevel, x, y, width, height, glFormat, glType, typedArray, byteOffset); // , byteLength
            }
            break;
        default:
            throw new Error(dimension);
    }
}
/**
 * Copy a region of compressed data from a GPU memory buffer into this texture.
 */
export function copyGPUBufferToMipLevel(gl, webglBuffer, byteLength, options) {
    const { dimension, width, height, depth = 0, mipLevel = 0, byteOffset = 0 } = options;
    const { x = 0, y = 0, z = 0 } = options;
    const { glFormat, glType, compressed } = options;
    const glTarget = getWebGLCubeFaceTarget(options.glTarget, dimension, depth);
    gl.bindBuffer(35052, webglBuffer);
    switch (dimension) {
        case '2d-array':
        case '3d':
            // 3 dimensional textures requires 3D texture functions
            if (compressed) {
                // TODO enable extension?
                // prettier-ignore
                gl.compressedTexSubImage3D(glTarget, mipLevel, x, y, z, width, height, depth, glFormat, byteLength, byteOffset);
            }
            else {
                // prettier-ignore
                gl.texSubImage3D(glTarget, mipLevel, x, y, z, width, height, depth, glFormat, glType, byteOffset);
            }
            break;
        case '2d':
        case 'cube':
            if (compressed) {
                // prettier-ignore
                gl.compressedTexSubImage2D(glTarget, mipLevel, x, y, width, height, glFormat, byteLength, byteOffset);
            }
            else {
                // prettier-ignore
                gl.texSubImage2D(glTarget, mipLevel, x, y, width, height, BORDER, glFormat, byteOffset);
            }
            break;
        default:
            throw new Error(dimension);
    }
}
// INTERNAL HELPERS
/** Convert a WebGPU style texture constant to a WebGL style texture constant */
export function getWebGLTextureTarget(dimension) {
    // prettier-ignore
    switch (dimension) {
        case '1d': break; // not supported in any WebGL version
        case '2d': return 3553; // supported in WebGL1
        case '3d': return 32879; // supported in WebGL2
        case 'cube': return 34067; // supported in WebGL1
        case '2d-array': return 35866; // supported in WebGL2
        case 'cube-array': break; // not supported in any WebGL version
    }
    throw new Error(dimension);
}
/**
 * In WebGL, cube maps specify faces by overriding target instead of using the depth parameter.
 * @note We still bind the texture using GL.TEXTURE_CUBE_MAP, but we need to use the face-specific target when setting mip levels.
 * @returns glTarget unchanged, if dimension !== 'cube'.
 */
export function getWebGLCubeFaceTarget(glTarget, dimension, level) {
    return dimension === 'cube' ? 34069 + level : glTarget;
}
/**
 * Copies data from a type  or a Texture object into ArrayBuffer object.
 * App can provide targetPixelArray or have it auto allocated by this method
 *  newly allocated by this method unless provided by app.
 * @deprecated Use CommandEncoder.copyTextureToBuffer and Buffer.read
 * @note Slow requires roundtrip to GPU
 *
 * @param source
 * @param options
 * @returns pixel array,
 */
export function readPixelsToArray(source, options) {
    const { sourceX = 0, sourceY = 0, sourceAttachment = 0 // TODO - support gl.readBuffer
     } = options || {};
    let { target = null, 
    // following parameters are auto deduced if not provided
    sourceWidth, sourceHeight, sourceDepth, sourceFormat, sourceType } = options || {};
    const { framebuffer, deleteFramebuffer } = getFramebuffer(source);
    // assert(framebuffer);
    const { gl, handle } = framebuffer;
    sourceWidth ||= framebuffer.width;
    sourceHeight ||= framebuffer.height;
    const texture = framebuffer.colorAttachments[sourceAttachment]?.texture;
    if (!texture) {
        throw new Error(`Invalid framebuffer attachment ${sourceAttachment}`);
    }
    sourceDepth = texture?.depth || 1;
    sourceFormat ||= texture?.glFormat || 6408;
    // Deduce the type from color attachment if not provided.
    sourceType ||= texture?.glType || 5121;
    // Deduce type and allocated pixelArray if needed
    target = getPixelArray(target, sourceType, sourceFormat, sourceWidth, sourceHeight, sourceDepth);
    // Pixel array available, if necessary, deduce type from it.
    sourceType = sourceType || getGLTypeFromTypedArray(target);
    // Note: luma.gl overrides bindFramebuffer so that we can reliably restore the previous framebuffer (this is the only function for which we do that)
    const prevHandle = gl.bindFramebuffer(36160, handle);
    // Select the color attachment to read from
    gl.readBuffer(36064 + sourceAttachment);
    // There is a lot of hedging in the WebGL2 spec about what formats are guaranteed to be readable
    // (It should always be possible to read RGBA/UNSIGNED_BYTE, but most other combinations are not guaranteed)
    // Querying is possible but expensive:
    // const {device} = framebuffer;
    // texture.glReadFormat ||= gl.getParameter(gl.IMPLEMENTATION_COLOR_READ_FORMAT);
    // texture.glReadType ||= gl.getParameter(gl.IMPLEMENTATION_COLOR_READ_TYPE);
    // console.log('params', device.getGLKey(texture.glReadFormat), device.getGLKey(texture.glReadType));
    gl.readPixels(sourceX, sourceY, sourceWidth, sourceHeight, sourceFormat, sourceType, target);
    gl.readBuffer(36064);
    gl.bindFramebuffer(36160, prevHandle || null);
    if (deleteFramebuffer) {
        framebuffer.destroy();
    }
    return target;
}
/**
 * Copies data from a Framebuffer or a Texture object into a Buffer object.
 * NOTE: doesn't wait for copy to be complete, it programs GPU to perform a DMA transffer.
 * @deprecated Use CommandEncoder
 * @param source
 * @param options
 */
export function readPixelsToBuffer(source, options) {
    const { target, sourceX = 0, sourceY = 0, sourceFormat = 6408, targetByteOffset = 0 } = options || {};
    // following parameters are auto deduced if not provided
    let { sourceWidth, sourceHeight, sourceType } = options || {};
    const { framebuffer, deleteFramebuffer } = getFramebuffer(source);
    // assert(framebuffer);
    sourceWidth = sourceWidth || framebuffer.width;
    sourceHeight = sourceHeight || framebuffer.height;
    // Asynchronous read (PIXEL_PACK_BUFFER) is WebGL2 only feature
    const webglFramebuffer = framebuffer;
    // deduce type if not available.
    sourceType = sourceType || 5121;
    let webglBufferTarget = target;
    if (!webglBufferTarget) {
        // Create new buffer with enough size
        const components = glFormatToComponents(sourceFormat);
        const byteCount = glTypeToBytes(sourceType);
        const byteLength = targetByteOffset + sourceWidth * sourceHeight * components * byteCount;
        webglBufferTarget = webglFramebuffer.device.createBuffer({ byteLength });
    }
    // TODO(donmccurdy): Do we have tests to confirm this is working?
    const commandEncoder = source.device.createCommandEncoder();
    commandEncoder.copyTextureToBuffer({
        sourceTexture: source,
        width: sourceWidth,
        height: sourceHeight,
        origin: [sourceX, sourceY],
        destinationBuffer: webglBufferTarget,
        byteOffset: targetByteOffset
    });
    commandEncoder.destroy();
    if (deleteFramebuffer) {
        framebuffer.destroy();
    }
    return webglBufferTarget;
}
/**
 * Copy a rectangle from a Framebuffer or Texture object into a texture (at an offset)
 * @deprecated Use CommandEncoder
 */
// eslint-disable-next-line complexity, max-statements
export function copyToTexture(sourceTexture, destinationTexture, options) {
    const { sourceX = 0, sourceY = 0, 
    // attachment = GL.COLOR_ATTACHMENT0, // TODO - support gl.readBuffer
    targetMipmaplevel = 0, targetInternalFormat = 6408 } = options || {};
    let { targetX, targetY, targetZ, width, // defaults to target width
    height // defaults to target height
     } = options || {};
    const { framebuffer, deleteFramebuffer } = getFramebuffer(sourceTexture);
    // assert(framebuffer);
    const webglFramebuffer = framebuffer;
    const { device, handle } = webglFramebuffer;
    const isSubCopy = typeof targetX !== 'undefined' ||
        typeof targetY !== 'undefined' ||
        typeof targetZ !== 'undefined';
    targetX = targetX || 0;
    targetY = targetY || 0;
    targetZ = targetZ || 0;
    const prevHandle = device.gl.bindFramebuffer(36160, handle);
    // TODO - support gl.readBuffer (WebGL2 only)
    // const prevBuffer = gl.readBuffer(attachment);
    // assert(target);
    let texture = null;
    let textureTarget;
    if (destinationTexture instanceof WEBGLTexture) {
        texture = destinationTexture;
        width = Number.isFinite(width) ? width : texture.width;
        height = Number.isFinite(height) ? height : texture.height;
        texture?.bind(0);
        // @ts-ignore
        textureTarget = texture.target;
    }
    else {
        // @ts-ignore
        textureTarget = target;
    }
    if (!isSubCopy) {
        device.gl.copyTexImage2D(textureTarget, targetMipmaplevel, targetInternalFormat, sourceX, sourceY, width, height, 0 /* border must be 0 */);
    }
    else {
        switch (textureTarget) {
            case 3553:
            case 34067:
                device.gl.copyTexSubImage2D(textureTarget, targetMipmaplevel, targetX, targetY, sourceX, sourceY, width, height);
                break;
            case 35866:
            case 32879:
                device.gl.copyTexSubImage3D(textureTarget, targetMipmaplevel, targetX, targetY, targetZ, sourceX, sourceY, width, height);
                break;
            default:
        }
    }
    if (texture) {
        texture.unbind();
    }
    // @ts-expect-error
    device.gl.bindFramebuffer(36160, prevHandle || null);
    if (deleteFramebuffer) {
        framebuffer.destroy();
    }
    return texture;
}
function getFramebuffer(source) {
    if (!(source instanceof Framebuffer)) {
        return { framebuffer: toFramebuffer(source), deleteFramebuffer: true };
    }
    return { framebuffer: source, deleteFramebuffer: false };
}
/**
 * Wraps a given texture into a framebuffer object, that can be further used
 * to read data from the texture object.
 */
export function toFramebuffer(texture, props) {
    const { device, width, height, id } = texture;
    const framebuffer = device.createFramebuffer({
        ...props,
        id: `framebuffer-for-${id}`,
        width,
        height,
        colorAttachments: [texture]
    });
    return framebuffer;
}
// eslint-disable-next-line max-params
function getPixelArray(pixelArray, glType, glFormat, width, height, depth) {
    if (pixelArray) {
        return pixelArray;
    }
    // const formatInfo = decodeTextureFormat(format);
    // Allocate pixel array if not already available, using supplied type
    glType ||= 5121;
    const ArrayType = getTypedArrayFromGLType(glType, { clamped: false });
    const components = glFormatToComponents(glFormat);
    // TODO - check for composite type (components = 1).
    return new ArrayType(width * height * components);
}
//# sourceMappingURL=webgl-texture-utils.js.map