// luma.gl
// SPDX-License-Identifier: MIT
// Copyright (c) vis.gl contributors
// Goal is to make WebGL2 contexts look like WebGL1
// @note Partly inspired by with some older code from the `regl` library
/* eslint-disable camelcase */
import { GL } from '@luma.gl/constants';
// webgl1 extensions natively supported by webgl2
const WEBGL1_STATIC_EXTENSIONS = {
    WEBGL_depth_texture: {
        UNSIGNED_INT_24_8_WEBGL: 34042
    },
    OES_element_index_uint: {},
    OES_texture_float: {},
    OES_texture_half_float: {
        // @ts-expect-error different numbers?
        HALF_FLOAT_OES: 5131
    },
    EXT_color_buffer_float: {},
    OES_standard_derivatives: {
        FRAGMENT_SHADER_DERIVATIVE_HINT_OES: 35723
    },
    EXT_frag_depth: {},
    EXT_blend_minmax: {
        MIN_EXT: 32775,
        MAX_EXT: 32776
    },
    EXT_shader_texture_lod: {}
};
const getWEBGL_draw_buffers = (gl) => ({
    drawBuffersWEBGL(buffers) {
        return gl.drawBuffers(buffers);
    },
    COLOR_ATTACHMENT0_WEBGL: 36064,
    COLOR_ATTACHMENT1_WEBGL: 36065,
    COLOR_ATTACHMENT2_WEBGL: 36066,
    COLOR_ATTACHMENT3_WEBGL: 36067
}); // - too many fields
const getOES_vertex_array_object = (gl) => ({
    VERTEX_ARRAY_BINDING_OES: 34229,
    createVertexArrayOES() {
        return gl.createVertexArray();
    },
    deleteVertexArrayOES(vertexArray) {
        return gl.deleteVertexArray(vertexArray);
    },
    isVertexArrayOES(vertexArray) {
        return gl.isVertexArray(vertexArray);
    },
    bindVertexArrayOES(vertexArray) {
        return gl.bindVertexArray(vertexArray);
    }
});
const getANGLE_instanced_arrays = (gl) => ({
    VERTEX_ATTRIB_ARRAY_DIVISOR_ANGLE: 0x88fe,
    drawArraysInstancedANGLE(...args) {
        return gl.drawArraysInstanced(...args);
    },
    drawElementsInstancedANGLE(...args) {
        return gl.drawElementsInstanced(...args);
    },
    vertexAttribDivisorANGLE(...args) {
        return gl.vertexAttribDivisor(...args);
    }
});
/**
 * Make browser return WebGL2 contexts even if WebGL1 contexts are requested
 * @param enforce
 * @returns
 */
export function enforceWebGL2(enforce = true) {
    const prototype = HTMLCanvasElement.prototype;
    if (!enforce && prototype.originalGetContext) {
        // Reset the original getContext function
        prototype.getContext = prototype.originalGetContext;
        prototype.originalGetContext = undefined;
        return;
    }
    // Store the original getContext function
    prototype.originalGetContext = prototype.getContext;
    // Override the getContext function
    prototype.getContext = function (contextId, options) {
        // Attempt to force WebGL2 for all WebGL1 contexts
        if (contextId === 'webgl' || contextId === 'experimental-webgl') {
            const context = this.originalGetContext('webgl2', options);
            // Work around test mocking
            if (context instanceof HTMLElement) {
                polyfillWebGL1Extensions(context);
            }
            return context;
        }
        // For any other type, return the original context
        return this.originalGetContext(contextId, options);
    };
}
/** Install WebGL1-only extensions on WebGL2 contexts */
export function polyfillWebGL1Extensions(gl) {
    // Enable, to support float and half-float textures
    gl.getExtension('EXT_color_buffer_float');
    // WebGL1 extensions implemented using WebGL2 APIs
    const boundExtensions = {
        ...WEBGL1_STATIC_EXTENSIONS,
        WEBGL_disjoint_timer_query: gl.getExtension('EXT_disjoint_timer_query_webgl2'),
        WEBGL_draw_buffers: getWEBGL_draw_buffers(gl),
        OES_vertex_array_object: getOES_vertex_array_object(gl),
        ANGLE_instanced_arrays: getANGLE_instanced_arrays(gl)
    };
    // Override gl.getExtension
    // eslint-disable-next-line @typescript-eslint/unbound-method
    const originalGetExtension = gl.getExtension;
    gl.getExtension = function (extensionName) {
        const ext = originalGetExtension.call(gl, extensionName);
        if (ext) {
            return ext;
        }
        // Injected extensions
        if (extensionName in boundExtensions) {
            return boundExtensions[extensionName];
        }
        return null;
    };
    // Override gl.getSupportedExtensions
    // eslint-disable-next-line @typescript-eslint/unbound-method
    const originalGetSupportedExtensions = gl.getSupportedExtensions;
    gl.getSupportedExtensions = function () {
        const extensions = originalGetSupportedExtensions.apply(gl) || [];
        return extensions?.concat(Object.keys(boundExtensions));
    };
}
// Update unsized WebGL1 formats to sized WebGL2 formats
// todo move to texture format file
// export function getInternalFormat(gl: WebGL2RenderingContext, format: GL, type: GL): GL {
//   // webgl2 texture formats
//   // https://webgl2fundamentals.org/webgl/lessons/webgl-data-textures.html
//   switch (format) {
//     case GL.DEPTH_COMPONENT:
//       return GL.DEPTH_COMPONENT24;
//     case GL.DEPTH_STENCIL:
//       return GL.DEPTH24_STENCIL8;
//     case GL.RGBA:
//       return type === GL.HALF_FLOAT ? GL.RGBA16F : GL.RGBA32F;
//     case GL.RGB:
//       return type === GL.HALF_FLOAT ? GL.RGB16F : GL.RGB32F;
//     default:
//       return format;
//   }
// }
/*
// texture type to update on the fly
export function getTextureType(gl: WebGL2RenderingContext, type: GL): GL {
  if (type === HALF_FLOAT_OES) {
    return GL.HALF_FLOAT;
  }
  return type;
}

  // And texImage2D to convert the internalFormat to webgl2.
  const webgl2 = this;
  const origTexImage = gl.texImage2D;
  gl.texImage2D = function (target, miplevel, iformat, a, typeFor6, c, d, typeFor9, f) {
    if (arguments.length == 6) {
      var ifmt = webgl2.getInternalFormat(gl, iformat, typeFor6);
      origTexImage.apply(gl, [target, miplevel, ifmt, a, webgl.getTextureType(gl, typeFor6), c]);
    } else {
      // arguments.length == 9
      var ifmt = webgl2.getInternalFormat(gl, iformat, typeFor9);
      origTexImage.apply(gl, [
        target,
        miplevel,
        ifmt,
        a,
        typeFor6,
        c,
        d,
        webgl2.getTextureType(gl, typeFor9),
        f
      ]);
    }
  };
};
*/
//# sourceMappingURL=polyfill-webgl1-extensions.js.map