import { DisposableDelegate } from '@lumino/disposable';

/**
 * An object which manages a drag-drop operation.
 *
 * A drag object dispatches four different events to drop targets:
 *
 * - `'lm-dragenter'` - Dispatched when the mouse enters the target
 *   element. This event must be canceled in order to receive any
 *   of the other events.
 *
 * - `'lm-dragover'` - Dispatched when the mouse moves over the drop
 *   target. It must cancel the event and set the `dropAction` to one
 *   of the supported actions in order to receive drop events.
 *
 * - `'lm-dragleave'` - Dispatched when the mouse leaves the target
 *   element. This includes moving the mouse into child elements.
 *
 * - `'lm-drop'`- Dispatched when the mouse is released over the target
 *   element when the target indicates an appropriate drop action. If
 *   the event is canceled, the indicated drop action is returned to
 *   the initiator through the resolved promise.
 *
 * A drag operation can be terminated at any time by pressing `Escape`
 * or by disposing the drag object.
 *
 * A drag object has the ability to automatically scroll a scrollable
 * element when the mouse is hovered near one of its edges. To enable
 * this, add the `data-lm-dragscroll` attribute to any element which
 * the drag object should consider for scrolling.
 *
 * #### Notes
 * This class is designed to be used when dragging and dropping custom
 * data *within* a single application. It is *not* a replacement for
 * the native drag-drop API. Instead, it provides an API which allows
 * drag operations to be initiated programmatically and enables the
 * transfer of arbitrary non-string objects; features which are not
 * possible with the native drag-drop API.
 */
class Drag {
    /**
     * Construct a new drag object.
     *
     * @param options - The options for initializing the drag.
     */
    constructor(options) {
        /**
         * The scroll loop handler function.
         */
        this._onScrollFrame = () => {
            // Bail early if there is no scroll target.
            if (!this._scrollTarget) {
                return;
            }
            // Unpack the scroll target.
            let { element, edge, distance } = this._scrollTarget;
            // Calculate the scroll delta using nonlinear acceleration.
            let d = Private.SCROLL_EDGE_SIZE - distance;
            let f = Math.pow(d / Private.SCROLL_EDGE_SIZE, 2);
            let s = Math.max(1, Math.round(f * Private.SCROLL_EDGE_SIZE));
            // Scroll the element in the specified direction.
            switch (edge) {
                case 'top':
                    element.scrollTop -= s;
                    break;
                case 'left':
                    element.scrollLeft -= s;
                    break;
                case 'right':
                    element.scrollLeft += s;
                    break;
                case 'bottom':
                    element.scrollTop += s;
                    break;
            }
            // Request the next cycle of the scroll loop.
            requestAnimationFrame(this._onScrollFrame);
        };
        this._disposed = false;
        this._dropAction = 'none';
        this._override = null;
        this._currentTarget = null;
        this._currentElement = null;
        this._promise = null;
        this._scrollTarget = null;
        this._resolve = null;
        this.document = options.document || document;
        this.mimeData = options.mimeData;
        this.dragImage = options.dragImage || null;
        this.proposedAction = options.proposedAction || 'copy';
        this.supportedActions = options.supportedActions || 'all';
        this.source = options.source || null;
    }
    /**
     * Dispose of the resources held by the drag object.
     *
     * #### Notes
     * This will cancel the drag operation if it is active.
     */
    dispose() {
        // Do nothing if the drag object is already disposed.
        if (this._disposed) {
            return;
        }
        this._disposed = true;
        // If there is a current target, dispatch a drag leave event.
        if (this._currentTarget) {
            let event = new PointerEvent('pointerup', {
                bubbles: true,
                cancelable: true,
                clientX: -1,
                clientY: -1
            });
            Private.dispatchDragLeave(this, this._currentTarget, null, event);
        }
        // Finalize the drag object with `'none'`.
        this._finalize('none');
    }
    /**
     * Test whether the drag object is disposed.
     */
    get isDisposed() {
        return this._disposed;
    }
    /**
     * Start the drag operation at the specified client position.
     *
     * @param clientX - The client X position for the drag start.
     *
     * @param clientY - The client Y position for the drag start.
     *
     * @returns A promise which resolves to the result of the drag.
     *
     * #### Notes
     * If the drag has already been started, the promise created by the
     * first call to `start` is returned.
     *
     * If the drag operation has ended, or if the drag object has been
     * disposed, the returned promise will resolve to `'none'`.
     *
     * The drag object will be automatically disposed when drag operation
     * completes. This means `Drag` objects are for single-use only.
     *
     * This method assumes the left mouse button is already held down.
     */
    start(clientX, clientY) {
        // If the drag object is already disposed, resolve to `none`.
        if (this._disposed) {
            return Promise.resolve('none');
        }
        // If the drag has already been started, return the promise.
        if (this._promise) {
            return this._promise;
        }
        // Install the document listeners for the drag object.
        this._addListeners();
        // Attach the drag image at the specified client position.
        this._attachDragImage(clientX, clientY);
        // Create the promise which will be resolved on completion.
        this._promise = new Promise(resolve => {
            this._resolve = resolve;
        });
        // Trigger a fake move event to kick off the drag operation.
        let event = new PointerEvent('pointermove', {
            bubbles: true,
            cancelable: true,
            clientX,
            clientY
        });
        document.dispatchEvent(event);
        // Return the pending promise for the drag operation.
        return this._promise;
    }
    /**
     * Handle the DOM events for the drag operation.
     *
     * @param event - The DOM event sent to the drag object.
     *
     * #### Notes
     * This method implements the DOM `EventListener` interface and is
     * called in response to events on the document. It should not be
     * called directly by user code.
     */
    handleEvent(event) {
        switch (event.type) {
            case 'pointermove':
                this._evtPointerMove(event);
                break;
            case 'pointerup':
                this._evtPointerUp(event);
                break;
            case 'keydown':
                this._evtKeyDown(event);
                break;
            default:
                // Stop all other events during drag-drop.
                event.preventDefault();
                event.stopPropagation();
                break;
        }
    }
    /**
     * Move the drag image element to the specified location.
     *
     * This is a no-op if there is no drag image element.
     */
    moveDragImage(clientX, clientY) {
        if (!this.dragImage) {
            return;
        }
        let style = this.dragImage.style;
        style.top = `${clientY}px`;
        style.left = `${clientX}px`;
    }
    /**
     * Handle the `'pointermove'` event for the drag object.
     */
    _evtPointerMove(event) {
        // Stop all input events during drag-drop.
        event.preventDefault();
        event.stopPropagation();
        // Update the current target node and dispatch enter/leave events.
        this._updateCurrentTarget(event);
        // Update the drag scroll element.
        this._updateDragScroll(event);
        // Move the drag image to the specified client position. This is
        // performed *after* dispatching to prevent unnecessary reflows.
        this.moveDragImage(event.clientX, event.clientY);
    }
    /**
     * Handle the `'pointerup'` event for the drag object.
     */
    _evtPointerUp(event) {
        // Stop all input events during drag-drop.
        event.preventDefault();
        event.stopPropagation();
        // Do nothing if the left button is not released.
        if (event.button !== 0) {
            return;
        }
        // Update the current target node and dispatch enter/leave events.
        // This prevents a subtle issue where the DOM mutates under the
        // cursor after the last move event but before the drop event.
        this._updateCurrentTarget(event);
        // If there is no current target, finalize with `'none'`.
        if (!this._currentTarget) {
            this._finalize('none');
            return;
        }
        // If the last drop action was `'none'`, dispatch a leave event
        // to the current target and finalize the drag with `'none'`.
        if (this._dropAction === 'none') {
            Private.dispatchDragLeave(this, this._currentTarget, null, event);
            this._finalize('none');
            return;
        }
        // Dispatch the drop event at the current target and finalize
        // with the resulting drop action.
        let action = Private.dispatchDrop(this, this._currentTarget, event);
        this._finalize(action);
    }
    /**
     * Handle the `'keydown'` event for the drag object.
     */
    _evtKeyDown(event) {
        // Stop all input events during drag-drop.
        event.preventDefault();
        event.stopPropagation();
        // Cancel the drag if `Escape` is pressed.
        if (event.keyCode === 27) {
            this.dispose();
        }
    }
    /**
     * Add the document event listeners for the drag object.
     */
    _addListeners() {
        document.addEventListener('pointerdown', this, true);
        document.addEventListener('pointermove', this, true);
        document.addEventListener('pointerup', this, true);
        document.addEventListener('pointerenter', this, true);
        document.addEventListener('pointerleave', this, true);
        document.addEventListener('pointerover', this, true);
        document.addEventListener('pointerout', this, true);
        document.addEventListener('keydown', this, true);
        document.addEventListener('keyup', this, true);
        document.addEventListener('keypress', this, true);
        document.addEventListener('contextmenu', this, true);
    }
    /**
     * Remove the document event listeners for the drag object.
     */
    _removeListeners() {
        document.removeEventListener('pointerdown', this, true);
        document.removeEventListener('pointermove', this, true);
        document.removeEventListener('pointerup', this, true);
        document.removeEventListener('pointerenter', this, true);
        document.removeEventListener('pointerleave', this, true);
        document.removeEventListener('pointerover', this, true);
        document.removeEventListener('pointerout', this, true);
        document.removeEventListener('keydown', this, true);
        document.removeEventListener('keyup', this, true);
        document.removeEventListener('keypress', this, true);
        document.removeEventListener('contextmenu', this, true);
    }
    /**
     * Update the drag scroll element under the mouse.
     */
    _updateDragScroll(event) {
        // Find the scroll target under the mouse.
        let target = Private.findScrollTarget(event);
        // Bail if there is nothing to scroll.
        if (!this._scrollTarget && !target) {
            return;
        }
        // Start the scroll loop if needed.
        if (!this._scrollTarget) {
            setTimeout(this._onScrollFrame, 500);
        }
        // Update the scroll target.
        this._scrollTarget = target;
    }
    /**
     * Update the current target node using the given mouse event.
     */
    _updateCurrentTarget(event) {
        // Fetch common local state.
        let prevTarget = this._currentTarget;
        let currTarget = this._currentTarget;
        let prevElem = this._currentElement;
        // Find the current indicated element at the given position.
        let currElem = this.document.elementFromPoint(event.clientX, event.clientY);
        // Update the current element reference.
        this._currentElement = currElem;
        // If the indicated element changes from the previous iteration,
        // and is different from the current target, dispatch the exit
        // event to the target.
        if (currElem !== prevElem && currElem !== currTarget) {
            Private.dispatchDragExit(this, currTarget, currElem, event);
        }
        // If the indicated element changes from the previous iteration,
        // and is different from the current target, dispatch the enter
        // event and compute the new target element.
        if (currElem !== prevElem && currElem !== currTarget) {
            currTarget = Private.dispatchDragEnter(this, currElem, currTarget, event);
        }
        // If the current target element has changed, update the current
        // target reference and dispatch the leave event to the old target.
        if (currTarget !== prevTarget) {
            this._currentTarget = currTarget;
            Private.dispatchDragLeave(this, prevTarget, currTarget, event);
        }
        // Dispatch the drag over event and update the drop action.
        let action = Private.dispatchDragOver(this, currTarget, event);
        this._setDropAction(action);
    }
    /**
     * Attach the drag image element at the specified location.
     *
     * This is a no-op if there is no drag image element.
     */
    _attachDragImage(clientX, clientY) {
        if (!this.dragImage) {
            return;
        }
        this.dragImage.classList.add('lm-mod-drag-image');
        let style = this.dragImage.style;
        style.pointerEvents = 'none';
        style.position = 'fixed';
        style.top = `${clientY}px`;
        style.left = `${clientX}px`;
        const body = this.document instanceof Document
            ? this.document.body
            : this.document.firstElementChild;
        body.appendChild(this.dragImage);
    }
    /**
     * Detach the drag image element from the DOM.
     *
     * This is a no-op if there is no drag image element.
     */
    _detachDragImage() {
        if (!this.dragImage) {
            return;
        }
        let parent = this.dragImage.parentNode;
        if (!parent) {
            return;
        }
        parent.removeChild(this.dragImage);
    }
    /**
     * Set the internal drop action state and update the drag cursor.
     */
    _setDropAction(action) {
        action = Private.validateAction(action, this.supportedActions);
        if (this._override && this._dropAction === action) {
            return;
        }
        switch (action) {
            case 'none':
                this._dropAction = action;
                this._override = Drag.overrideCursor('no-drop', this.document);
                break;
            case 'copy':
                this._dropAction = action;
                this._override = Drag.overrideCursor('copy', this.document);
                break;
            case 'link':
                this._dropAction = action;
                this._override = Drag.overrideCursor('alias', this.document);
                break;
            case 'move':
                this._dropAction = action;
                this._override = Drag.overrideCursor('move', this.document);
                break;
        }
    }
    /**
     * Finalize the drag operation and resolve the drag promise.
     */
    _finalize(action) {
        // Store the resolve function as a temp variable.
        let resolve = this._resolve;
        // Remove the document event listeners.
        this._removeListeners();
        // Detach the drag image.
        this._detachDragImage();
        // Dispose of the cursor override.
        if (this._override) {
            this._override.dispose();
            this._override = null;
        }
        // Clear the mime data.
        this.mimeData.clear();
        // Clear the rest of the internal drag state.
        this._disposed = true;
        this._dropAction = 'none';
        this._currentTarget = null;
        this._currentElement = null;
        this._scrollTarget = null;
        this._promise = null;
        this._resolve = null;
        // Finally, resolve the promise to the given drop action.
        if (resolve) {
            resolve(action);
        }
    }
}
/**
 * The namespace for the `Drag` class statics.
 */
(function (Drag) {
    /**
     * A custom event used for drag-drop operations.
     *
     * #### Notes
     * In order to receive `'lm-dragover'`, `'lm-dragleave'`, or `'lm-drop'`
     * events, a drop target must cancel the `'lm-dragenter'` event by
     * calling the event's `preventDefault()` method.
     */
    class Event extends DragEvent {
        constructor(event, options) {
            super(options.type, {
                bubbles: true,
                cancelable: true,
                altKey: event.altKey,
                button: event.button,
                clientX: event.clientX,
                clientY: event.clientY,
                ctrlKey: event.ctrlKey,
                detail: 0,
                metaKey: event.metaKey,
                relatedTarget: options.related,
                screenX: event.screenX,
                screenY: event.screenY,
                shiftKey: event.shiftKey,
                view: window
            });
            const { drag } = options;
            this.dropAction = 'none';
            this.mimeData = drag.mimeData;
            this.proposedAction = drag.proposedAction;
            this.supportedActions = drag.supportedActions;
            this.source = drag.source;
        }
    }
    Drag.Event = Event;
    /**
     * Override the cursor icon for the entire document.
     *
     * @param cursor - The string representing the cursor style.
     *
     * @returns A disposable which will clear the override when disposed.
     *
     * #### Notes
     * The most recent call to `overrideCursor` takes precedence.
     * Disposing an old override has no effect on the current override.
     *
     * This utility function is used by the `Drag` class to override the
     * mouse cursor during a drag-drop operation, but it can also be used
     * by other classes to fix the cursor icon during normal mouse drags.
     *
     * #### Example
     * ```typescript
     * import { Drag } from '@lumino/dragdrop';
     *
     * // Force the cursor to be 'wait' for the entire document.
     * let override = Drag.overrideCursor('wait');
     *
     * // Clear the override by disposing the return value.
     * override.dispose();
     * ```
     */
    function overrideCursor(cursor, doc = document) {
        let id = ++overrideCursorID;
        const body = doc instanceof Document
            ? doc.body
            : doc.firstElementChild;
        body.style.cursor = cursor;
        body.classList.add('lm-mod-override-cursor');
        return new DisposableDelegate(() => {
            if (id === overrideCursorID) {
                body.style.cursor = '';
                body.classList.remove('lm-mod-override-cursor');
            }
        });
    }
    Drag.overrideCursor = overrideCursor;
    /**
     * The internal id for the active cursor override.
     */
    let overrideCursorID = 0;
})(Drag || (Drag = {}));
/**
 * The namespace for the module implementation details.
 */
var Private;
(function (Private) {
    /**
     * The size of a drag scroll edge, in pixels.
     */
    Private.SCROLL_EDGE_SIZE = 20;
    /**
     * Validate the given action is one of the supported actions.
     *
     * Returns the given action or `'none'` if the action is unsupported.
     */
    function validateAction(action, supported) {
        return actionTable[action] & supportedTable[supported] ? action : 'none';
    }
    Private.validateAction = validateAction;
    /**
     * Find the drag scroll target under the mouse, if any.
     */
    function findScrollTarget(event) {
        // Look up the client mouse position.
        let x = event.clientX;
        let y = event.clientY;
        // Get the element under the mouse.
        let element = document.elementFromPoint(x, y);
        // Search for a scrollable target based on the mouse position.
        // The null assert in third clause of for-loop is required due to:
        // https://github.com/Microsoft/TypeScript/issues/14143
        for (; element; element = element.parentElement) {
            // Ignore elements which are not marked as scrollable.
            if (!element.hasAttribute('data-lm-dragscroll')) {
                continue;
            }
            // Set up the coordinate offsets for the element.
            let offsetX = 0;
            let offsetY = 0;
            if (element === document.body) {
                offsetX = window.pageXOffset;
                offsetY = window.pageYOffset;
            }
            // Get the element bounds in viewport coordinates.
            let r = element.getBoundingClientRect();
            let top = r.top + offsetY;
            let left = r.left + offsetX;
            let right = left + r.width;
            let bottom = top + r.height;
            // Skip the element if it's not under the mouse.
            if (x < left || x >= right || y < top || y >= bottom) {
                continue;
            }
            // Compute the distance to each edge.
            let dl = x - left + 1;
            let dt = y - top + 1;
            let dr = right - x;
            let db = bottom - y;
            // Find the smallest of the edge distances.
            let distance = Math.min(dl, dt, dr, db);
            // Skip the element if the mouse is not within a scroll edge.
            if (distance > Private.SCROLL_EDGE_SIZE) {
                continue;
            }
            // Set up the edge result variable.
            let edge;
            // Find the edge for the computed distance.
            switch (distance) {
                case db:
                    edge = 'bottom';
                    break;
                case dt:
                    edge = 'top';
                    break;
                case dr:
                    edge = 'right';
                    break;
                case dl:
                    edge = 'left';
                    break;
                default:
                    throw 'unreachable';
            }
            // Compute how much the element can scroll in width and height.
            let dsw = element.scrollWidth - element.clientWidth;
            let dsh = element.scrollHeight - element.clientHeight;
            // Determine if the element should be scrolled for the edge.
            let shouldScroll;
            switch (edge) {
                case 'top':
                    shouldScroll = dsh > 0 && element.scrollTop > 0;
                    break;
                case 'left':
                    shouldScroll = dsw > 0 && element.scrollLeft > 0;
                    break;
                case 'right':
                    shouldScroll = dsw > 0 && element.scrollLeft < dsw;
                    break;
                case 'bottom':
                    shouldScroll = dsh > 0 && element.scrollTop < dsh;
                    break;
                default:
                    throw 'unreachable';
            }
            // Skip the element if it should not be scrolled.
            if (!shouldScroll) {
                continue;
            }
            // Return the drag scroll target.
            return { element, edge, distance };
        }
        // No drag scroll target was found.
        return null;
    }
    Private.findScrollTarget = findScrollTarget;
    /**
     * Dispatch a drag enter event to the indicated element.
     *
     * @param drag - The drag object associated with the action.
     *
     * @param currElem - The currently indicated element, or `null`. This
     *   is the "immediate user selection" from the whatwg spec.
     *
     * @param currTarget - The current drag target element, or `null`. This
     *   is the "current target element" from the whatwg spec.
     *
     * @param event - The mouse event related to the action.
     *
     * @returns The element to use as the current drag target. This is the
     *   "current target element" from the whatwg spec, and may be `null`.
     *
     * #### Notes
     * This largely implements the drag enter portion of the whatwg spec:
     * https://html.spec.whatwg.org/multipage/interaction.html#drag-and-drop-processing-model
     */
    function dispatchDragEnter(drag, currElem, currTarget, event) {
        // If the current element is null, return null as the new target.
        if (!currElem) {
            return null;
        }
        // Dispatch a drag enter event to the current element.
        let dragEvent = new Drag.Event(event, {
            drag,
            related: currTarget,
            type: 'lm-dragenter'
        });
        let canceled = !currElem.dispatchEvent(dragEvent);
        // If the event was canceled, use the current element as the new target.
        if (canceled) {
            return currElem;
        }
        // If the current element is the document body, keep the original target.
        const body = drag.document instanceof Document
            ? drag.document.body
            : drag.document.firstElementChild;
        if (currElem === body) {
            return currTarget;
        }
        // Dispatch a drag enter event on the document body.
        dragEvent = new Drag.Event(event, {
            drag,
            related: currTarget,
            type: 'lm-dragenter'
        });
        body.dispatchEvent(dragEvent);
        // Ignore the event cancellation, and use the body as the new target.
        return body;
    }
    Private.dispatchDragEnter = dispatchDragEnter;
    /**
     * Dispatch a drag exit event to the indicated element.
     *
     * @param drag - The drag object associated with the action.
     *
     * @param prevTarget - The previous target element, or `null`. This
     *   is the previous "current target element" from the whatwg spec.
     *
     * @param currTarget - The current drag target element, or `null`. This
     *   is the "current target element" from the whatwg spec.
     *
     * @param event - The mouse event related to the action.
     *
     * #### Notes
     * This largely implements the drag exit portion of the whatwg spec:
     * https://html.spec.whatwg.org/multipage/interaction.html#drag-and-drop-processing-model
     */
    function dispatchDragExit(drag, prevTarget, currTarget, event) {
        // If the previous target is null, do nothing.
        if (!prevTarget) {
            return;
        }
        // Dispatch the drag exit event to the previous target.
        let dragEvent = new Drag.Event(event, {
            drag,
            related: currTarget,
            type: 'lm-dragexit'
        });
        prevTarget.dispatchEvent(dragEvent);
    }
    Private.dispatchDragExit = dispatchDragExit;
    /**
     * Dispatch a drag leave event to the indicated element.
     *
     * @param drag - The drag object associated with the action.
     *
     * @param prevTarget - The previous target element, or `null`. This
     *   is the previous "current target element" from the whatwg spec.
     *
     * @param currTarget - The current drag target element, or `null`. This
     *   is the "current target element" from the whatwg spec.
     *
     * @param event - The mouse event related to the action.
     *
     * #### Notes
     * This largely implements the drag leave portion of the whatwg spec:
     * https://html.spec.whatwg.org/multipage/interaction.html#drag-and-drop-processing-model
     */
    function dispatchDragLeave(drag, prevTarget, currTarget, event) {
        // If the previous target is null, do nothing.
        if (!prevTarget) {
            return;
        }
        // Dispatch the drag leave event to the previous target.
        let dragEvent = new Drag.Event(event, {
            drag,
            related: currTarget,
            type: 'lm-dragleave'
        });
        prevTarget.dispatchEvent(dragEvent);
    }
    Private.dispatchDragLeave = dispatchDragLeave;
    /**
     * Dispatch a drag over event to the indicated element.
     *
     * @param drag - The drag object associated with the action.
     *
     * @param currTarget - The current drag target element, or `null`. This
     *   is the "current target element" from the whatwg spec.
     *
     * @param event - The mouse event related to the action.
     *
     * @returns The `DropAction` result of the drag over event.
     *
     * #### Notes
     * This largely implements the drag over portion of the whatwg spec:
     * https://html.spec.whatwg.org/multipage/interaction.html#drag-and-drop-processing-model
     */
    function dispatchDragOver(drag, currTarget, event) {
        // If there is no current target, the drop action is none.
        if (!currTarget) {
            return 'none';
        }
        // Dispatch the drag over event to the current target.
        let dragEvent = new Drag.Event(event, {
            drag,
            related: null,
            type: 'lm-dragover'
        });
        let canceled = !currTarget.dispatchEvent(dragEvent);
        // If the event was canceled, return the drop action result.
        if (canceled) {
            return dragEvent.dropAction;
        }
        // Otherwise, the effective drop action is none.
        return 'none';
    }
    Private.dispatchDragOver = dispatchDragOver;
    /**
     * Dispatch a drop event to the indicated element.
     *
     * @param drag - The drag object associated with the action.
     *
     * @param currTarget - The current drag target element, or `null`. This
     *   is the "current target element" from the whatwg spec.
     *
     * @param event - The mouse event related to the action.
     *
     * @returns The `DropAction` result of the drop event.
     *
     * #### Notes
     * This largely implements the drag over portion of the whatwg spec:
     * https://html.spec.whatwg.org/multipage/interaction.html#drag-and-drop-processing-model
     */
    function dispatchDrop(drag, currTarget, event) {
        // If there is no current target, the drop action is none.
        if (!currTarget) {
            return 'none';
        }
        // Dispatch the drop event to the current target.
        let dragEvent = new Drag.Event(event, {
            drag,
            related: null,
            type: 'lm-drop'
        });
        let canceled = !currTarget.dispatchEvent(dragEvent);
        // If the event was canceled, return the drop action result.
        if (canceled) {
            return dragEvent.dropAction;
        }
        // Otherwise, the effective drop action is none.
        return 'none';
    }
    Private.dispatchDrop = dispatchDrop;
    /**
     * A lookup table from drop action to bit value.
     */
    const actionTable = {
        none: 0x0,
        copy: 0x1,
        link: 0x2,
        move: 0x4
    };
    /**
     * A lookup table from supported action to drop action bit mask.
     */
    const supportedTable = {
        none: actionTable['none'],
        copy: actionTable['copy'],
        link: actionTable['link'],
        move: actionTable['move'],
        'copy-link': actionTable['copy'] | actionTable['link'],
        'copy-move': actionTable['copy'] | actionTable['move'],
        'link-move': actionTable['link'] | actionTable['move'],
        all: actionTable['copy'] | actionTable['link'] | actionTable['move']
    };
})(Private || (Private = {}));

export { Drag };
//# sourceMappingURL=index.es6.js.map
