import { Message } from '@lumino/messaging';
import { AccordionLayout } from './accordionlayout';
import { SplitLayout } from './splitlayout';
import { SplitPanel } from './splitpanel';
import { Title } from './title';
import { Widget } from './widget';
/**
 * A panel which arranges its widgets into resizable sections separated by a title widget.
 *
 * #### Notes
 * This class provides a convenience wrapper around [[AccordionLayout]].
 */
export declare class AccordionPanel extends SplitPanel {
    /**
     * Construct a new accordion panel.
     *
     * @param options - The options for initializing the accordion panel.
     */
    constructor(options?: AccordionPanel.IOptions);
    /**
     * The renderer used by the accordion panel.
     */
    get renderer(): AccordionPanel.IRenderer;
    /**
     * The section title space.
     *
     * This is the height if the panel is vertical and the width if it is
     * horizontal.
     */
    get titleSpace(): number;
    set titleSpace(value: number);
    /**
     * A read-only array of the section titles in the panel.
     */
    get titles(): ReadonlyArray<HTMLElement>;
    /**
     * Add a widget to the end of the panel.
     *
     * @param widget - The widget to add to the panel.
     *
     * #### Notes
     * If the widget is already contained in the panel, it will be moved.
     */
    addWidget(widget: Widget): void;
    /**
     * Collapse the widget at position `index`.
     *
     * #### Notes
     * If no widget is found for `index`, this will bail.
     *
     * @param index Widget index
     */
    collapse(index: number): void;
    /**
     * Expand the widget at position `index`.
     *
     * #### Notes
     * If no widget is found for `index`, this will bail.
     *
     * @param index Widget index
     */
    expand(index: number): void;
    /**
     * Insert a widget at the specified index.
     *
     * @param index - The index at which to insert the widget.
     *
     * @param widget - The widget to insert into to the panel.
     *
     * #### Notes
     * If the widget is already contained in the panel, it will be moved.
     */
    insertWidget(index: number, widget: Widget): void;
    /**
     * Handle the DOM events for the accordion panel.
     *
     * @param event - The DOM event sent to the panel.
     *
     * #### Notes
     * This method implements the DOM `EventListener` interface and is
     * called in response to events on the panel's DOM node. It should
     * not be called directly by user code.
     */
    handleEvent(event: Event): void;
    /**
     * A message handler invoked on a `'before-attach'` message.
     */
    protected onBeforeAttach(msg: Message): void;
    /**
     * A message handler invoked on an `'after-detach'` message.
     */
    protected onAfterDetach(msg: Message): void;
    /**
     * Handle the `changed` signal of a title object.
     */
    private _onTitleChanged;
    /**
     * Compute the size of widgets in this panel on the title click event.
     * On closing, the size of the widget is cached and we will try to expand
     * the last opened widget.
     * On opening, we will use the cached size if it is available to restore the
     * widget.
     * In both cases, if we can not compute the size of widgets, we will let
     * `SplitLayout` decide.
     *
     * @param index - The index of widget to be opened of closed
     *
     * @returns Relative size of widgets in this panel, if this size can
     * not be computed, return `undefined`
     */
    private _computeWidgetSize;
    /**
     * Handle the `'click'` event for the accordion panel
     */
    private _evtClick;
    /**
     * Handle the `'keydown'` event for the accordion panel.
     */
    private _eventKeyDown;
    private _toggleExpansion;
    private _widgetSizesCache;
}
/**
 * The namespace for the `AccordionPanel` class statics.
 */
export declare namespace AccordionPanel {
    /**
     * A type alias for a accordion panel orientation.
     */
    type Orientation = SplitLayout.Orientation;
    /**
     * A type alias for a accordion panel alignment.
     */
    type Alignment = SplitLayout.Alignment;
    /**
     * A type alias for a accordion panel renderer.
     */
    type IRenderer = AccordionLayout.IRenderer;
    /**
     * An options object for initializing a accordion panel.
     */
    interface IOptions extends Partial<AccordionLayout.IOptions> {
        /**
         * The accordion layout to use for the accordion panel.
         *
         * If this is provided, the other options are ignored.
         *
         * The default is a new `AccordionLayout`.
         */
        layout?: AccordionLayout;
    }
    /**
     * The default implementation of `IRenderer`.
     */
    class Renderer extends SplitPanel.Renderer implements IRenderer {
        constructor();
        /**
         * A selector which matches any title node in the accordion.
         */
        readonly titleClassName = "lm-AccordionPanel-title";
        /**
         * Render the collapse indicator for a section title.
         *
         * @param data - The data to use for rendering the section title.
         *
         * @returns A element representing the collapse indicator.
         */
        createCollapseIcon(data: Title<Widget>): HTMLElement;
        /**
         * Render the element for a section title.
         *
         * @param data - The data to use for rendering the section title.
         *
         * @returns A element representing the section title.
         */
        createSectionTitle(data: Title<Widget>): HTMLElement;
        /**
         * Create a unique render key for the title.
         *
         * @param data - The data to use for the title.
         *
         * @returns The unique render key for the title.
         *
         * #### Notes
         * This method caches the key against the section title the first time
         * the key is generated.
         */
        createTitleKey(data: Title<Widget>): string;
        private static _nInstance;
        private readonly _uuid;
        private _titleID;
        private _titleKeys;
    }
    /**
     * The default `Renderer` instance.
     */
    const defaultRenderer: Renderer;
}
