import { Message } from '@lumino/messaging';
import { ElementARIAAttrs, ElementDataset, h, VirtualElement } from '@lumino/virtualdom';
import { Menu } from './menu';
import { Title } from './title';
import { Widget } from './widget';
/**
 * A widget which displays menus as a canonical menu bar.
 */
export declare class MenuBar extends Widget {
    /**
     * Construct a new menu bar.
     *
     * @param options - The options for initializing the menu bar.
     */
    constructor(options?: MenuBar.IOptions);
    /**
     * Dispose of the resources held by the widget.
     */
    dispose(): void;
    /**
     * The renderer used by the menu bar.
     */
    readonly renderer: MenuBar.IRenderer;
    /**
     * The child menu of the menu bar.
     *
     * #### Notes
     * This will be `null` if the menu bar does not have an open menu.
     */
    get childMenu(): Menu | null;
    /**
     * Get the menu bar content node.
     *
     * #### Notes
     * This is the node which holds the menu title nodes.
     *
     * Modifying this node directly can lead to undefined behavior.
     */
    get contentNode(): HTMLUListElement;
    /**
     * Get the currently active menu.
     */
    get activeMenu(): Menu | null;
    /**
     * Set the currently active menu.
     *
     * #### Notes
     * If the menu does not exist, the menu will be set to `null`.
     */
    set activeMenu(value: Menu | null);
    /**
     * Get the index of the currently active menu.
     *
     * #### Notes
     * This will be `-1` if no menu is active.
     */
    get activeIndex(): number;
    /**
     * Set the index of the currently active menu.
     *
     * #### Notes
     * If the menu cannot be activated, the index will be set to `-1`.
     */
    set activeIndex(value: number);
    /**
     * A read-only array of the menus in the menu bar.
     */
    get menus(): ReadonlyArray<Menu>;
    /**
     * Open the active menu and activate its first menu item.
     *
     * #### Notes
     * If there is no active menu, this is a no-op.
     */
    openActiveMenu(): void;
    /**
     * Add a menu to the end of the menu bar.
     *
     * @param menu - The menu to add to the menu bar.
     *
     * #### Notes
     * If the menu is already added to the menu bar, it will be moved.
     */
    addMenu(menu: Menu): void;
    /**
     * Insert a menu into the menu bar at the specified index.
     *
     * @param index - The index at which to insert the menu.
     *
     * @param menu - The menu to insert into the menu bar.
     *
     * #### Notes
     * The index will be clamped to the bounds of the menus.
     *
     * If the menu is already added to the menu bar, it will be moved.
     */
    insertMenu(index: number, menu: Menu): void;
    /**
     * Remove a menu from the menu bar.
     *
     * @param menu - The menu to remove from the menu bar.
     *
     * #### Notes
     * This is a no-op if the menu is not in the menu bar.
     */
    removeMenu(menu: Menu): void;
    /**
     * Remove the menu at a given index from the menu bar.
     *
     * @param index - The index of the menu to remove.
     *
     * #### Notes
     * This is a no-op if the index is out of range.
     */
    removeMenuAt(index: number): void;
    /**
     * Remove all menus from the menu bar.
     */
    clearMenus(): void;
    /**
     * Handle the DOM events for the menu bar.
     *
     * @param event - The DOM event sent to the menu bar.
     *
     * #### Notes
     * This method implements the DOM `EventListener` interface and is
     * called in response to events on the menu bar's DOM nodes. It
     * should not be called directly by user code.
     */
    handleEvent(event: Event): void;
    /**
     * A message handler invoked on a `'before-attach'` message.
     */
    protected onBeforeAttach(msg: Message): void;
    /**
     * A message handler invoked on an `'after-detach'` message.
     */
    protected onAfterDetach(msg: Message): void;
    /**
     * A message handler invoked on an `'activate-request'` message.
     */
    protected onActivateRequest(msg: Message): void;
    /**
     * A message handler invoked on an `'update-request'` message.
     */
    protected onUpdateRequest(msg: Message): void;
    /**
     * Handle the `'keydown'` event for the menu bar.
     *
     * #### Notes
     * All keys are trapped except the tab key that is ignored.
     */
    private _evtKeyDown;
    /**
     * Handle the `'mousedown'` event for the menu bar.
     */
    private _evtMouseDown;
    /**
     * Handle the `'mousemove'` event for the menu bar.
     */
    private _evtMouseMove;
    /**
     * Handle the `'mouseleave'` event for the menu bar.
     */
    private _evtMouseLeave;
    /**
     * Open the child menu at the active index immediately.
     *
     * If a different child menu is already open, it will be closed,
     * even if there is no active menu.
     */
    private _openChildMenu;
    /**
     * Close the child menu immediately.
     *
     * This is a no-op if a child menu is not open.
     */
    private _closeChildMenu;
    /**
     * Handle the `aboutToClose` signal of a menu.
     */
    private _onMenuAboutToClose;
    /**
     * Handle the `menuRequested` signal of a child menu.
     */
    private _onMenuMenuRequested;
    /**
     * Handle the `changed` signal of a title object.
     */
    private _onTitleChanged;
    private _activeIndex;
    private _forceItemsPosition;
    private _menus;
    private _childMenu;
}
/**
 * The namespace for the `MenuBar` class statics.
 */
export declare namespace MenuBar {
    /**
     * An options object for creating a menu bar.
     */
    interface IOptions {
        /**
         * A custom renderer for creating menu bar content.
         *
         * The default is a shared renderer instance.
         */
        renderer?: IRenderer;
        /**
         * Whether to force the position of the menu. The MenuBar forces the
         * coordinates of its menus by default. With this option you can disable it.
         *
         * Setting to `false` will enable the logic which repositions the
         * coordinates of the menu if it will not fit entirely on screen.
         *
         * The default is `true`.
         */
        forceItemsPosition?: Menu.IOpenOptions;
    }
    /**
     * An object which holds the data to render a menu bar item.
     */
    interface IRenderData {
        /**
         * The title to be rendered.
         */
        readonly title: Title<Widget>;
        /**
         * Whether the item is the active item.
         */
        readonly active: boolean;
        readonly onfocus?: (event: FocusEvent) => void;
    }
    /**
     * A renderer for use with a menu bar.
     */
    interface IRenderer {
        /**
         * Render the virtual element for a menu bar item.
         *
         * @param data - The data to use for rendering the item.
         *
         * @returns A virtual element representing the item.
         */
        renderItem(data: IRenderData): VirtualElement;
    }
    /**
     * The default implementation of `IRenderer`.
     *
     * #### Notes
     * Subclasses are free to reimplement rendering methods as needed.
     */
    class Renderer implements IRenderer {
        /**
         * Render the virtual element for a menu bar item.
         *
         * @param data - The data to use for rendering the item.
         *
         * @returns A virtual element representing the item.
         */
        renderItem(data: IRenderData): VirtualElement;
        /**
         * Render the icon element for a menu bar item.
         *
         * @param data - The data to use for rendering the icon.
         *
         * @returns A virtual element representing the item icon.
         */
        renderIcon(data: IRenderData): VirtualElement;
        /**
         * Render the label element for a menu item.
         *
         * @param data - The data to use for rendering the label.
         *
         * @returns A virtual element representing the item label.
         */
        renderLabel(data: IRenderData): VirtualElement;
        /**
         * Create the class name for the menu bar item.
         *
         * @param data - The data to use for the class name.
         *
         * @returns The full class name for the menu item.
         */
        createItemClass(data: IRenderData): string;
        /**
         * Create the dataset for a menu bar item.
         *
         * @param data - The data to use for the item.
         *
         * @returns The dataset for the menu bar item.
         */
        createItemDataset(data: IRenderData): ElementDataset;
        /**
         * Create the aria attributes for menu bar item.
         *
         * @param data - The data to use for the aria attributes.
         *
         * @returns The aria attributes object for the item.
         */
        createItemARIA(data: IRenderData): ElementARIAAttrs;
        /**
         * Create the class name for the menu bar item icon.
         *
         * @param data - The data to use for the class name.
         *
         * @returns The full class name for the item icon.
         */
        createIconClass(data: IRenderData): string;
        /**
         * Create the render content for the label node.
         *
         * @param data - The data to use for the label content.
         *
         * @returns The content to add to the label node.
         */
        formatLabel(data: IRenderData): h.Child;
    }
    /**
     * The default `Renderer` instance.
     */
    const defaultRenderer: Renderer;
}
