"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
Object.defineProperty(exports, "__esModule", { value: true });
var dist_commonjs_1 = require("@iiif/vocabulary/dist-commonjs");
var normalize_url_1 = require("normalize-url");
var CanvasWorld = /** @class */ (function () {
    /**
     * @param {Array} canvases - Array of Manifesto:Canvas objects to create a
     * world from.
     */
    function CanvasWorld(canvases, layers, viewingDirection) {
        if (viewingDirection === void 0) { viewingDirection = dist_commonjs_1.ViewingDirection.LEFT_TO_RIGHT; }
        this.canvases = canvases; //.map(c => new Canvas(c)); UV has already parsed the canvases by this point
        this.layers = layers;
        this.viewingDirection = viewingDirection;
    }
    Object.defineProperty(CanvasWorld.prototype, "canvasIds", {
        get: function () {
            return this.canvases.map(function (canvas) { return canvas.id; });
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(CanvasWorld.prototype, "canvasDimensions", {
        get: function () {
            if (this._canvasDimensions) {
                return this._canvasDimensions;
            }
            var _a = this.canvasDirection, dirX = _a[0], dirY = _a[1];
            var scale = dirY === 0
                ? Math.min.apply(Math, this.canvases.map(function (c) { return c.getHeight(); })) : Math.min.apply(Math, this.canvases.map(function (c) { return c.getWidth(); }));
            var incX = 0;
            var incY = 0;
            var canvasDims = this.canvases.reduce(function (acc, canvas) {
                var canvasHeight = 0;
                var canvasWidth = 0;
                if (!isNaN(canvas.aspectRatio)) {
                    if (dirY === 0) {
                        // constant height
                        canvasHeight = scale;
                        canvasWidth = Math.floor(scale * canvas.aspectRatio);
                    }
                    else {
                        // constant width
                        canvasWidth = scale;
                        canvasHeight = Math.floor(scale * (1 / canvas.aspectRatio));
                    }
                }
                acc.push({
                    canvas: canvas,
                    height: canvasHeight,
                    width: canvasWidth,
                    x: incX,
                    y: incY
                });
                incX += dirX * canvasWidth;
                incY += dirY * canvasHeight;
                return acc;
            }, []);
            var worldHeight = dirY === 0 ? scale : Math.abs(incY);
            var worldWidth = dirX === 0 ? scale : Math.abs(incX);
            this._canvasDimensions = canvasDims.reduce(function (acc, dims) {
                acc.push(__assign(__assign({}, dims), { x: dirX === -1 ? dims.x + worldWidth - dims.width : dims.x, y: dirY === -1 ? dims.y + worldHeight - dims.height : dims.y }));
                return acc;
            }, []);
            return this._canvasDimensions;
        },
        enumerable: true,
        configurable: true
    });
    /**
     * contentResourceToWorldCoordinates - calculates the contentResource coordinates
     * respective to the world.
     */
    CanvasWorld.prototype.contentResourceToWorldCoordinates = function (contentResource) {
        var canvasIndex = this.canvases.findIndex(function (c) {
            return c.imageResources.find(function (r) { return r.id === contentResource.id; });
        });
        var canvas = this.canvases[canvasIndex];
        if (!canvas)
            return [];
        var _a = this.canvasToWorldCoordinates(canvas.id), x = _a[0], y = _a[1], w = _a[2], h = _a[3];
        var fragmentOffset = canvas.onFragment(contentResource.id);
        if (fragmentOffset) {
            return [
                x + fragmentOffset[0],
                y + fragmentOffset[1],
                fragmentOffset[2],
                fragmentOffset[3]
            ];
        }
        return [x, y, w, h];
    };
    /** */
    CanvasWorld.prototype.canvasToWorldCoordinates = function (canvasId) {
        var canvasDimensions = this.canvasDimensions.find(function (c) { return c.canvas.id === canvasId; });
        return [
            canvasDimensions.x,
            canvasDimensions.y,
            canvasDimensions.width,
            canvasDimensions.height
        ];
    };
    Object.defineProperty(CanvasWorld.prototype, "canvasDirection", {
        /** */
        get: function () {
            switch (this.viewingDirection) {
                case dist_commonjs_1.ViewingDirection.LEFT_TO_RIGHT:
                    return [1, 0];
                case dist_commonjs_1.ViewingDirection.RIGHT_TO_LEFT:
                    return [-1, 0];
                case dist_commonjs_1.ViewingDirection.TOP_TO_BOTTOM:
                    return [0, 1];
                case dist_commonjs_1.ViewingDirection.BOTTOM_TO_TOP:
                    return [0, -1];
                default:
                    return [1, 0];
            }
        },
        enumerable: true,
        configurable: true
    });
    /** Get the IIIF content resource for an image */
    CanvasWorld.prototype.contentResource = function (infoResponseId) {
        var canvas = this.canvases.find(function (c) {
            return c.imageServiceIds.some(function (id) {
                return normalize_url_1.default(id, { stripAuthentication: false }) ===
                    normalize_url_1.default(infoResponseId, { stripAuthentication: false });
            });
        });
        if (!canvas)
            return undefined;
        return canvas.imageResources.find(function (r) {
            return normalize_url_1.default(r.getServices()[0].id, { stripAuthentication: false }) ===
                normalize_url_1.default(infoResponseId, { stripAuthentication: false });
        });
    };
    /** @private */
    CanvasWorld.prototype.getLayerMetadata = function (contentResource) {
        if (!this.layers)
            return undefined;
        var canvas = this.canvases.find(function (c) {
            return c.imageResources.find(function (r) { return r.id === contentResource.id; });
        });
        if (!canvas)
            return undefined;
        var resourceIndex = canvas.imageResources.findIndex(function (r) { return r.id === contentResource.id; });
        var layer = this.layers[canvas.id];
        var imageResourceLayer = layer && layer[contentResource.id];
        return __assign({ index: resourceIndex, opacity: 1, total: canvas.imageResources.length, visibility: true }, imageResourceLayer);
    };
    /** */
    CanvasWorld.prototype.layerOpacityOfImageResource = function (contentResource) {
        var layer = this.getLayerMetadata(contentResource);
        if (!layer)
            return 1;
        if (!layer.visibility)
            return 0;
        return layer.opacity;
    };
    /** */
    CanvasWorld.prototype.layerIndexOfImageResource = function (contentResource) {
        var layer = this.getLayerMetadata(contentResource);
        if (!layer)
            return undefined;
        return layer.total - layer.index - 1;
    };
    /**
     * offsetByCanvas - calculates the offset for a given canvas target. Currently
     * assumes a horizontal only layout.
     */
    CanvasWorld.prototype.offsetByCanvas = function (canvasTarget) {
        var coordinates = this.canvasToWorldCoordinates(canvasTarget);
        return {
            x: coordinates[0],
            y: coordinates[1]
        };
    };
    /**
     * worldBounds - calculates the "World" bounds. World in this case is canvases
     * lined up horizontally starting from left to right.
     */
    CanvasWorld.prototype.worldBounds = function () {
        var worldWidth = Math.max.apply(Math, this.canvasDimensions.map(function (c) { return c.x + c.width; }));
        var worldHeight = Math.max.apply(Math, this.canvasDimensions.map(function (c) { return c.y + c.height; }));
        return [0, 0, worldWidth, worldHeight];
    };
    /** */
    CanvasWorld.prototype.canvasAtPoint = function (point) {
        var canvasDimensions = this.canvasDimensions.find(function (c) {
            return c.x <= point.x &&
                point.x <= c.x + c.width &&
                c.y <= point.y &&
                point.y <= c.y + c.height;
        });
        return canvasDimensions && canvasDimensions.canvas;
    };
    return CanvasWorld;
}());
exports.CanvasWorld = CanvasWorld;
//# sourceMappingURL=CanvasWorld.js.map