'use strict';

var validateConstants = require('./validate/validate_constants');
var validate = require('./validate/validate');
var latestStyleSpec = require('../reference/latest.min');
var validateGlyphsURL = require('./validate/validate_glyphs_url');

/**
 * Validate a Mapbox GL style against the style specification. This entrypoint,
 * `mapbox-gl-style-spec/lib/validate_style.min`, is designed to produce as
 * small a browserify bundle as possible by omitting unnecessary functionality
 * and legacy style specifications.
 *
 * @param {Object} style The style to be validated.
 * @param {Object} [styleSpec] The style specification to validate against.
 *     If omitted, the latest style spec is used.
 * @returns {Array<ValidationError>}
 * @example
 *   var validate = require('mapbox-gl-style-spec/lib/validate_style.min');
 *   var errors = validate(style);
 */
function validateStyleMin(style, styleSpec) {
    styleSpec = styleSpec || latestStyleSpec;

    var errors = [];

    errors = errors.concat(validate({
        key: '',
        value: style,
        valueSpec: styleSpec.$root,
        styleSpec: styleSpec,
        style: style,
        objectElementValidators: {
            glyphs: validateGlyphsURL
        }
    }));

    if (styleSpec.$version > 7 && style.constants) {
        errors = errors.concat(validateConstants({
            key: 'constants',
            value: style.constants,
            style: style,
            styleSpec: styleSpec
        }));
    }

    return sortErrors(errors);
}

validateStyleMin.source = wrapCleanErrors(require('./validate/validate_source'));
validateStyleMin.layer = wrapCleanErrors(require('./validate/validate_layer'));
validateStyleMin.filter = wrapCleanErrors(require('./validate/validate_filter'));
validateStyleMin.paintProperty = wrapCleanErrors(require('./validate/validate_paint_property'));
validateStyleMin.layoutProperty = wrapCleanErrors(require('./validate/validate_layout_property'));

function sortErrors(errors) {
    return [].concat(errors).sort(function (a, b) {
        return a.line - b.line;
    });
}

function wrapCleanErrors(inner) {
    return function() {
        return sortErrors(inner.apply(this, arguments));
    };
}

module.exports = validateStyleMin;
