/// <reference path="../../typings/index.d.ts" />

import * as rx from "rx";

import {Camera, Transform} from "../Geo";
import {Node} from "../Graph";
import {RenderCamera, RenderMode, ISize} from "../Render";
import {IFrame} from "../State";

interface IRenderCameraOperation {
    (rc: RenderCamera): RenderCamera;
}

export class RenderService {
    private _element: HTMLElement;
    private _currentFrame$: rx.Observable<IFrame>;

    private _renderCameraOperation$: rx.Subject<IRenderCameraOperation>;
    private _renderCameraHolder$: rx.Observable<RenderCamera>;
    private _renderCameraFrame$: rx.Observable<RenderCamera>;
    private _renderCamera$: rx.Observable<RenderCamera>;

    private _resize$: rx.Subject<void>;
    private _size$: rx.BehaviorSubject<ISize>;

    private _renderMode$: rx.BehaviorSubject<RenderMode>;

    constructor(element: HTMLElement, currentFrame$: rx.Observable<IFrame>, renderMode: RenderMode) {
        this._element = element;
        this._currentFrame$ = currentFrame$;

        renderMode = renderMode != null ? renderMode : RenderMode.Letterbox;

        this._resize$ = new rx.Subject<void>();
        this._renderCameraOperation$ = new rx.Subject<IRenderCameraOperation>();

        this._size$ =
            new rx.BehaviorSubject<ISize>(
                {
                    height: this._element.offsetHeight,
                    width: this._element.offsetWidth,
                });

        this._resize$
            .map<ISize>(
                (): ISize => {
                    return { height: this._element.offsetHeight, width: this._element.offsetWidth };
                })
            .subscribe(this._size$);

        this._renderMode$ = new rx.BehaviorSubject<RenderMode>(renderMode);

        this._renderCameraHolder$ = this._renderCameraOperation$
            .startWith(
                (rc: RenderCamera): RenderCamera => {
                    return rc;
                })
            .scan<RenderCamera>(
                (rc: RenderCamera, operation: IRenderCameraOperation): RenderCamera => {
                    return operation(rc);
                },
                new RenderCamera(this._element.offsetWidth / this._element.offsetHeight, renderMode))
            .shareReplay(1);

        this._renderCameraFrame$ = this._currentFrame$
            .withLatestFrom(
                this._renderCameraHolder$,
                (frame: IFrame, renderCamera: RenderCamera): [IFrame, RenderCamera] => {
                    return [frame, renderCamera];
                })
            .do(
                (args: [IFrame, RenderCamera]): void => {
                    let frame: IFrame = args[0];
                    let rc: RenderCamera = args[1];

                    let camera: Camera = frame.state.camera;

                    if (rc.alpha !== frame.state.alpha ||
                        rc.zoom !== frame.state.zoom ||
                        rc.camera.diff(camera) > 0.00001) {

                        let currentTransform: Transform = frame.state.currentTransform;
                        let previousTransform: Transform =
                            frame.state.previousTransform != null ?
                                frame.state.previousTransform :
                                frame.state.currentTransform;

                        let previousNode: Node =
                            frame.state.previousNode != null ?
                                frame.state.previousNode :
                                frame.state.currentNode;

                        rc.currentAspect = currentTransform.basicAspect;
                        rc.currentPano = frame.state.currentNode.fullPano;
                        rc.previousAspect = previousTransform.basicAspect;
                        rc.previousPano = previousNode.fullPano;

                        rc.alpha = frame.state.alpha;
                        rc.zoom = frame.state.zoom;

                        rc.camera.copy(camera);
                        rc.updatePerspective(camera);

                        rc.updateProjection();
                    }

                    rc.frameId = frame.id;
                })
            .map<RenderCamera>(
                (args: [IFrame, RenderCamera]): RenderCamera => {
                    return args[1];
                })
            .shareReplay(1);

        this._renderCamera$ = this._renderCameraFrame$
            .filter(
                (rc: RenderCamera): boolean => {
                    return rc.changed;
                })
            .shareReplay(1);

        this._size$
            .skip(1)
            .map<IRenderCameraOperation>(
                (size: ISize) => {
                    return (rc: RenderCamera): RenderCamera => {
                        rc.perspective.aspect = size.width / size.height;
                        rc.updateProjection();

                        return rc;
                    };
                })
            .subscribe(this._renderCameraOperation$);

        this._renderMode$
            .skip(1)
            .map<IRenderCameraOperation>(
                (rm: RenderMode) => {
                    return (rc: RenderCamera): RenderCamera => {
                        rc.renderMode = rm;
                        rc.updateProjection();

                        return rc;
                    };
                })
            .subscribe(this._renderCameraOperation$);

        this._renderCameraHolder$.subscribe();
        this._size$.subscribe();
        this._renderMode$.subscribe();
    }

    public get element(): HTMLElement {
        return this._element;
    }

    public get resize$(): rx.Subject<void> {
        return this._resize$;
    }

    public get size$(): rx.Observable<ISize> {
        return this._size$;
    }

    public get renderMode$(): rx.Subject<RenderMode> {
        return this._renderMode$;
    }

    public get renderCameraFrame$(): rx.Observable<RenderCamera> {
        return this._renderCameraFrame$;
    }

    public get renderCamera$(): rx.Observable<RenderCamera> {
        return this._renderCamera$;
    }
}

export default RenderService;
