"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = computeEigenDecomposition;

var _core = require("@math.gl/core");

var scratchMatrix = new _core.Matrix3();
var scratchUnitary = new _core.Matrix3();
var scratchDiagonal = new _core.Matrix3();
var jMatrix = new _core.Matrix3();
var jMatrixTranspose = new _core.Matrix3();

function computeEigenDecomposition(matrix) {
  var result = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
  var EIGEN_TOLERANCE = _core._MathUtils.EPSILON20;
  var EIGEN_MAX_SWEEPS = 10;
  var count = 0;
  var sweep = 0;
  var unitaryMatrix = scratchUnitary;
  var diagonalMatrix = scratchDiagonal;
  unitaryMatrix.identity();
  diagonalMatrix.copy(matrix);
  var epsilon = EIGEN_TOLERANCE * computeFrobeniusNorm(diagonalMatrix);

  while (sweep < EIGEN_MAX_SWEEPS && offDiagonalFrobeniusNorm(diagonalMatrix) > epsilon) {
    shurDecomposition(diagonalMatrix, jMatrix);
    jMatrixTranspose.copy(jMatrix).transpose();
    diagonalMatrix.multiplyRight(jMatrix);
    diagonalMatrix.multiplyLeft(jMatrixTranspose);
    unitaryMatrix.multiplyRight(jMatrix);

    if (++count > 2) {
      ++sweep;
      count = 0;
    }
  }

  result.unitary = unitaryMatrix.toTarget(result.unitary);
  result.diagonal = diagonalMatrix.toTarget(result.diagonal);
  return result;
}

function computeFrobeniusNorm(matrix) {
  var norm = 0.0;

  for (var i = 0; i < 9; ++i) {
    var temp = matrix[i];
    norm += temp * temp;
  }

  return Math.sqrt(norm);
}

var rowVal = [1, 0, 0];
var colVal = [2, 2, 1];

function offDiagonalFrobeniusNorm(matrix) {
  var norm = 0.0;

  for (var i = 0; i < 3; ++i) {
    var temp = matrix[scratchMatrix.getElementIndex(colVal[i], rowVal[i])];
    norm += 2.0 * temp * temp;
  }

  return Math.sqrt(norm);
}

function shurDecomposition(matrix, result) {
  var tolerance = _core._MathUtils.EPSILON15;
  var maxDiagonal = 0.0;
  var rotAxis = 1;

  for (var i = 0; i < 3; ++i) {
    var temp = Math.abs(matrix[scratchMatrix.getElementIndex(colVal[i], rowVal[i])]);

    if (temp > maxDiagonal) {
      rotAxis = i;
      maxDiagonal = temp;
    }
  }

  var p = rowVal[rotAxis];
  var q = colVal[rotAxis];
  var c = 1.0;
  var s = 0.0;

  if (Math.abs(matrix[scratchMatrix.getElementIndex(q, p)]) > tolerance) {
    var qq = matrix[scratchMatrix.getElementIndex(q, q)];
    var pp = matrix[scratchMatrix.getElementIndex(p, p)];
    var qp = matrix[scratchMatrix.getElementIndex(q, p)];
    var tau = (qq - pp) / 2.0 / qp;
    var t;

    if (tau < 0.0) {
      t = -1.0 / (-tau + Math.sqrt(1.0 + tau * tau));
    } else {
      t = 1.0 / (tau + Math.sqrt(1.0 + tau * tau));
    }

    c = 1.0 / Math.sqrt(1.0 + t * t);
    s = t * c;
  }

  _core.Matrix3.IDENTITY.to(result);

  result[scratchMatrix.getElementIndex(p, p)] = result[scratchMatrix.getElementIndex(q, q)] = c;
  result[scratchMatrix.getElementIndex(q, p)] = s;
  result[scratchMatrix.getElementIndex(p, q)] = -s;
  return result;
}
//# sourceMappingURL=compute-eigen-decomposition.js.map