import { Vector3 } from '@math.gl/core';
import BoundingSphere from '../bounding-volumes/bounding-sphere';
const fromPointsXMin = new Vector3();
const fromPointsYMin = new Vector3();
const fromPointsZMin = new Vector3();
const fromPointsXMax = new Vector3();
const fromPointsYMax = new Vector3();
const fromPointsZMax = new Vector3();
const fromPointsCurrentPos = new Vector3();
const fromPointsScratch = new Vector3();
const fromPointsRitterCenter = new Vector3();
const fromPointsMinBoxPt = new Vector3();
const fromPointsMaxBoxPt = new Vector3();
const fromPointsNaiveCenterScratch = new Vector3();
const volumeConstant = 4.0 / 3.0 * Math.PI;
export default function makeBoundingSphereFromPoints(positions, result = new BoundingSphere()) {
  if (!positions || positions.length === 0) {
    return result.fromCenterRadius([0, 0, 0], 0);
  }

  const currentPos = fromPointsCurrentPos.copy(positions[0]);
  const xMin = fromPointsXMin.copy(currentPos);
  const yMin = fromPointsYMin.copy(currentPos);
  const zMin = fromPointsZMin.copy(currentPos);
  const xMax = fromPointsXMax.copy(currentPos);
  const yMax = fromPointsYMax.copy(currentPos);
  const zMax = fromPointsZMax.copy(currentPos);

  for (const position of positions) {
    currentPos.copy(position);
    const x = currentPos.x;
    const y = currentPos.y;
    const z = currentPos.z;

    if (x < xMin.x) {
      xMin.copy(currentPos);
    }

    if (x > xMax.x) {
      xMax.copy(currentPos);
    }

    if (y < yMin.y) {
      yMin.copy(currentPos);
    }

    if (y > yMax.y) {
      yMax.copy(currentPos);
    }

    if (z < zMin.z) {
      zMin.copy(currentPos);
    }

    if (z > zMax.z) {
      zMax.copy(currentPos);
    }
  }

  const xSpan = fromPointsScratch.copy(xMax).subtract(xMin).magnitudeSquared();
  const ySpan = fromPointsScratch.copy(yMax).subtract(yMin).magnitudeSquared();
  const zSpan = fromPointsScratch.copy(zMax).subtract(zMin).magnitudeSquared();
  let diameter1 = xMin;
  let diameter2 = xMax;
  let maxSpan = xSpan;

  if (ySpan > maxSpan) {
    maxSpan = ySpan;
    diameter1 = yMin;
    diameter2 = yMax;
  }

  if (zSpan > maxSpan) {
    maxSpan = zSpan;
    diameter1 = zMin;
    diameter2 = zMax;
  }

  const ritterCenter = fromPointsRitterCenter;
  ritterCenter.x = (diameter1.x + diameter2.x) * 0.5;
  ritterCenter.y = (diameter1.y + diameter2.y) * 0.5;
  ritterCenter.z = (diameter1.z + diameter2.z) * 0.5;
  let radiusSquared = fromPointsScratch.copy(diameter2).subtract(ritterCenter).magnitudeSquared();
  let ritterRadius = Math.sqrt(radiusSquared);
  const minBoxPt = fromPointsMinBoxPt;
  minBoxPt.x = xMin.x;
  minBoxPt.y = yMin.y;
  minBoxPt.z = zMin.z;
  const maxBoxPt = fromPointsMaxBoxPt;
  maxBoxPt.x = xMax.x;
  maxBoxPt.y = yMax.y;
  maxBoxPt.z = zMax.z;
  const naiveCenter = fromPointsNaiveCenterScratch.copy(minBoxPt).add(maxBoxPt).multiplyByScalar(0.5);
  let naiveRadius = 0;

  for (const position of positions) {
    currentPos.copy(position);
    const r = fromPointsScratch.copy(currentPos).subtract(naiveCenter).magnitude();

    if (r > naiveRadius) {
      naiveRadius = r;
    }

    const oldCenterToPointSquared = fromPointsScratch.copy(currentPos).subtract(ritterCenter).magnitudeSquared();

    if (oldCenterToPointSquared > radiusSquared) {
      const oldCenterToPoint = Math.sqrt(oldCenterToPointSquared);
      ritterRadius = (ritterRadius + oldCenterToPoint) * 0.5;
      radiusSquared = ritterRadius * ritterRadius;
      const oldToNew = oldCenterToPoint - ritterRadius;
      ritterCenter.x = (ritterRadius * ritterCenter.x + oldToNew * currentPos.x) / oldCenterToPoint;
      ritterCenter.y = (ritterRadius * ritterCenter.y + oldToNew * currentPos.y) / oldCenterToPoint;
      ritterCenter.z = (ritterRadius * ritterCenter.z + oldToNew * currentPos.z) / oldCenterToPoint;
    }
  }

  if (ritterRadius < naiveRadius) {
    ritterCenter.to(result.center);
    result.radius = ritterRadius;
  } else {
    naiveCenter.to(result.center);
    result.radius = naiveRadius;
  }

  return result;
}
//# sourceMappingURL=bounding-sphere-from-points.js.map