import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";
import { Vector3 } from '@math.gl/core';
import * as mat4 from 'gl-matrix/mat4';
import { INTERSECTION } from '../../constants';
const scratchVector = new Vector3();
const scratchVector2 = new Vector3();
export default class BoundingSphere {
  constructor(center = [0, 0, 0], radius = 0.0) {
    _defineProperty(this, "center", void 0);

    _defineProperty(this, "radius", void 0);

    this.radius = -0;
    this.center = new Vector3();
    this.fromCenterRadius(center, radius);
  }

  fromCenterRadius(center, radius) {
    this.center.from(center);
    this.radius = radius;
    return this;
  }

  fromCornerPoints(corner, oppositeCorner) {
    oppositeCorner = scratchVector.from(oppositeCorner);
    this.center = new Vector3().from(corner).add(oppositeCorner).scale(0.5);
    this.radius = this.center.distance(oppositeCorner);
    return this;
  }

  equals(right) {
    return this === right || Boolean(right) && this.center.equals(right.center) && this.radius === right.radius;
  }

  clone() {
    return new BoundingSphere(this.center, this.radius);
  }

  union(boundingSphere) {
    const leftCenter = this.center;
    const leftRadius = this.radius;
    const rightCenter = boundingSphere.center;
    const rightRadius = boundingSphere.radius;
    const toRightCenter = scratchVector.copy(rightCenter).subtract(leftCenter);
    const centerSeparation = toRightCenter.magnitude();

    if (leftRadius >= centerSeparation + rightRadius) {
      return this.clone();
    }

    if (rightRadius >= centerSeparation + leftRadius) {
      return boundingSphere.clone();
    }

    const halfDistanceBetweenTangentPoints = (leftRadius + centerSeparation + rightRadius) * 0.5;
    scratchVector2.copy(toRightCenter).scale((-leftRadius + halfDistanceBetweenTangentPoints) / centerSeparation).add(leftCenter);
    this.center.copy(scratchVector2);
    this.radius = halfDistanceBetweenTangentPoints;
    return this;
  }

  expand(point) {
    const scratchPoint = scratchVector.from(point);
    const radius = scratchPoint.subtract(this.center).magnitude();

    if (radius > this.radius) {
      this.radius = radius;
    }

    return this;
  }

  transform(transform) {
    this.center.transform(transform);
    const scale = mat4.getScaling(scratchVector, transform);
    this.radius = Math.max(scale[0], Math.max(scale[1], scale[2])) * this.radius;
    return this;
  }

  distanceSquaredTo(point) {
    const d = this.distanceTo(point);
    return d * d;
  }

  distanceTo(point) {
    const scratchPoint = scratchVector.from(point);
    const delta = scratchPoint.subtract(this.center);
    return Math.max(0, delta.len() - this.radius);
  }

  intersectPlane(plane) {
    const center = this.center;
    const radius = this.radius;
    const normal = plane.normal;
    const distanceToPlane = normal.dot(center) + plane.distance;

    if (distanceToPlane < -radius) {
      return INTERSECTION.OUTSIDE;
    }

    if (distanceToPlane < radius) {
      return INTERSECTION.INTERSECTING;
    }

    return INTERSECTION.INSIDE;
  }

}
//# sourceMappingURL=bounding-sphere.js.map