import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";
import { Vector3, Matrix3, Quaternion } from '@math.gl/core';
import BoundingSphere from './bounding-sphere';
import { INTERSECTION } from '../../constants';
const scratchVector3 = new Vector3();
const scratchOffset = new Vector3();
const scratchVectorU = new Vector3();
const scratchVectorV = new Vector3();
const scratchVectorW = new Vector3();
const scratchCorner = new Vector3();
const scratchToCenter = new Vector3();
const MATRIX3 = {
  COLUMN0ROW0: 0,
  COLUMN0ROW1: 1,
  COLUMN0ROW2: 2,
  COLUMN1ROW0: 3,
  COLUMN1ROW1: 4,
  COLUMN1ROW2: 5,
  COLUMN2ROW0: 6,
  COLUMN2ROW1: 7,
  COLUMN2ROW2: 8
};
export default class OrientedBoundingBox {
  constructor(center = [0, 0, 0], halfAxes = [0, 0, 0, 0, 0, 0, 0, 0, 0]) {
    _defineProperty(this, "center", void 0);

    _defineProperty(this, "halfAxes", void 0);

    this.center = new Vector3().from(center);
    this.halfAxes = new Matrix3(halfAxes);
  }

  get halfSize() {
    const xAxis = this.halfAxes.getColumn(0);
    const yAxis = this.halfAxes.getColumn(1);
    const zAxis = this.halfAxes.getColumn(2);
    return [new Vector3(xAxis).len(), new Vector3(yAxis).len(), new Vector3(zAxis).len()];
  }

  get quaternion() {
    const xAxis = this.halfAxes.getColumn(0);
    const yAxis = this.halfAxes.getColumn(1);
    const zAxis = this.halfAxes.getColumn(2);
    const normXAxis = new Vector3(xAxis).normalize();
    const normYAxis = new Vector3(yAxis).normalize();
    const normZAxis = new Vector3(zAxis).normalize();
    return new Quaternion().fromMatrix3(new Matrix3([...normXAxis, ...normYAxis, ...normZAxis]));
  }

  fromCenterHalfSizeQuaternion(center, halfSize, quaternion) {
    const quaternionObject = new Quaternion(quaternion);
    const directionsMatrix = new Matrix3().fromQuaternion(quaternionObject);
    directionsMatrix[0] = directionsMatrix[0] * halfSize[0];
    directionsMatrix[1] = directionsMatrix[1] * halfSize[0];
    directionsMatrix[2] = directionsMatrix[2] * halfSize[0];
    directionsMatrix[3] = directionsMatrix[3] * halfSize[1];
    directionsMatrix[4] = directionsMatrix[4] * halfSize[1];
    directionsMatrix[5] = directionsMatrix[5] * halfSize[1];
    directionsMatrix[6] = directionsMatrix[6] * halfSize[2];
    directionsMatrix[7] = directionsMatrix[7] * halfSize[2];
    directionsMatrix[8] = directionsMatrix[8] * halfSize[2];
    this.center = new Vector3().from(center);
    this.halfAxes = directionsMatrix;
    return this;
  }

  clone() {
    return new OrientedBoundingBox(this.center, this.halfAxes);
  }

  equals(right) {
    return this === right || Boolean(right) && this.center.equals(right.center) && this.halfAxes.equals(right.halfAxes);
  }

  getBoundingSphere(result = new BoundingSphere()) {
    const halfAxes = this.halfAxes;
    const u = halfAxes.getColumn(0, scratchVectorU);
    const v = halfAxes.getColumn(1, scratchVectorV);
    const w = halfAxes.getColumn(2, scratchVectorW);
    const cornerVector = scratchVector3.copy(u).add(v).add(w);
    result.center.copy(this.center);
    result.radius = cornerVector.magnitude();
    return result;
  }

  intersectPlane(plane) {
    const center = this.center;
    const normal = plane.normal;
    const halfAxes = this.halfAxes;
    const normalX = normal.x;
    const normalY = normal.y;
    const normalZ = normal.z;
    const radEffective = Math.abs(normalX * halfAxes[MATRIX3.COLUMN0ROW0] + normalY * halfAxes[MATRIX3.COLUMN0ROW1] + normalZ * halfAxes[MATRIX3.COLUMN0ROW2]) + Math.abs(normalX * halfAxes[MATRIX3.COLUMN1ROW0] + normalY * halfAxes[MATRIX3.COLUMN1ROW1] + normalZ * halfAxes[MATRIX3.COLUMN1ROW2]) + Math.abs(normalX * halfAxes[MATRIX3.COLUMN2ROW0] + normalY * halfAxes[MATRIX3.COLUMN2ROW1] + normalZ * halfAxes[MATRIX3.COLUMN2ROW2]);
    const distanceToPlane = normal.dot(center) + plane.distance;

    if (distanceToPlane <= -radEffective) {
      return INTERSECTION.OUTSIDE;
    } else if (distanceToPlane >= radEffective) {
      return INTERSECTION.INSIDE;
    }

    return INTERSECTION.INTERSECTING;
  }

  distanceTo(point) {
    return Math.sqrt(this.distanceSquaredTo(point));
  }

  distanceSquaredTo(point) {
    const offset = scratchOffset.from(point).subtract(this.center);
    const halfAxes = this.halfAxes;
    const u = halfAxes.getColumn(0, scratchVectorU);
    const v = halfAxes.getColumn(1, scratchVectorV);
    const w = halfAxes.getColumn(2, scratchVectorW);
    const uHalf = u.magnitude();
    const vHalf = v.magnitude();
    const wHalf = w.magnitude();
    u.normalize();
    v.normalize();
    w.normalize();
    let distanceSquared = 0.0;
    let d;
    d = Math.abs(offset.dot(u)) - uHalf;

    if (d > 0) {
      distanceSquared += d * d;
    }

    d = Math.abs(offset.dot(v)) - vHalf;

    if (d > 0) {
      distanceSquared += d * d;
    }

    d = Math.abs(offset.dot(w)) - wHalf;

    if (d > 0) {
      distanceSquared += d * d;
    }

    return distanceSquared;
  }

  computePlaneDistances(position, direction, result = [-0, -0]) {
    let minDist = Number.POSITIVE_INFINITY;
    let maxDist = Number.NEGATIVE_INFINITY;
    const center = this.center;
    const halfAxes = this.halfAxes;
    const u = halfAxes.getColumn(0, scratchVectorU);
    const v = halfAxes.getColumn(1, scratchVectorV);
    const w = halfAxes.getColumn(2, scratchVectorW);
    const corner = scratchCorner.copy(u).add(v).add(w).add(center);
    const toCenter = scratchToCenter.copy(corner).subtract(position);
    let mag = direction.dot(toCenter);
    minDist = Math.min(mag, minDist);
    maxDist = Math.max(mag, maxDist);
    corner.copy(center).add(u).add(v).subtract(w);
    toCenter.copy(corner).subtract(position);
    mag = direction.dot(toCenter);
    minDist = Math.min(mag, minDist);
    maxDist = Math.max(mag, maxDist);
    corner.copy(center).add(u).subtract(v).add(w);
    toCenter.copy(corner).subtract(position);
    mag = direction.dot(toCenter);
    minDist = Math.min(mag, minDist);
    maxDist = Math.max(mag, maxDist);
    corner.copy(center).add(u).subtract(v).subtract(w);
    toCenter.copy(corner).subtract(position);
    mag = direction.dot(toCenter);
    minDist = Math.min(mag, minDist);
    maxDist = Math.max(mag, maxDist);
    center.copy(corner).subtract(u).add(v).add(w);
    toCenter.copy(corner).subtract(position);
    mag = direction.dot(toCenter);
    minDist = Math.min(mag, minDist);
    maxDist = Math.max(mag, maxDist);
    center.copy(corner).subtract(u).add(v).subtract(w);
    toCenter.copy(corner).subtract(position);
    mag = direction.dot(toCenter);
    minDist = Math.min(mag, minDist);
    maxDist = Math.max(mag, maxDist);
    center.copy(corner).subtract(u).subtract(v).add(w);
    toCenter.copy(corner).subtract(position);
    mag = direction.dot(toCenter);
    minDist = Math.min(mag, minDist);
    maxDist = Math.max(mag, maxDist);
    center.copy(corner).subtract(u).subtract(v).subtract(w);
    toCenter.copy(corner).subtract(position);
    mag = direction.dot(toCenter);
    minDist = Math.min(mag, minDist);
    maxDist = Math.max(mag, maxDist);
    result[0] = minDist;
    result[1] = maxDist;
    return result;
  }

  transform(transformation) {
    this.center.transformAsPoint(transformation);
    const xAxis = this.halfAxes.getColumn(0, scratchVectorU);
    xAxis.transformAsPoint(transformation);
    const yAxis = this.halfAxes.getColumn(1, scratchVectorV);
    yAxis.transformAsPoint(transformation);
    const zAxis = this.halfAxes.getColumn(2, scratchVectorW);
    zAxis.transformAsPoint(transformation);
    this.halfAxes = new Matrix3([...xAxis, ...yAxis, ...zAxis]);
    return this;
  }

  getTransform() {
    throw new Error('not implemented');
  }

}
//# sourceMappingURL=oriented-bounding-box.js.map