import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";
import { assert, Vector2 } from '@math.gl/core';
import PerspectiveOffCenterFrustum from './perspective-off-center-frustum';

const defined = val => val !== null && typeof val !== 'undefined';

export default class PerspectiveFrustum {
  constructor(options = {}) {
    _defineProperty(this, "_offCenterFrustum", new PerspectiveOffCenterFrustum());

    _defineProperty(this, "fov", void 0);

    _defineProperty(this, "_fov", void 0);

    _defineProperty(this, "_fovy", void 0);

    _defineProperty(this, "_sseDenominator", void 0);

    _defineProperty(this, "aspectRatio", void 0);

    _defineProperty(this, "_aspectRatio", void 0);

    _defineProperty(this, "near", void 0);

    _defineProperty(this, "_near", void 0);

    _defineProperty(this, "far", void 0);

    _defineProperty(this, "_far", void 0);

    _defineProperty(this, "xOffset", void 0);

    _defineProperty(this, "_xOffset", void 0);

    _defineProperty(this, "yOffset", void 0);

    _defineProperty(this, "_yOffset", void 0);

    const {
      fov,
      aspectRatio,
      near = 1.0,
      far = 500000000.0,
      xOffset = 0.0,
      yOffset = 0.0
    } = options;
    this.fov = fov;
    this.aspectRatio = aspectRatio;
    this.near = near;
    this.far = far;
    this.xOffset = xOffset;
    this.yOffset = yOffset;
  }

  clone() {
    return new PerspectiveFrustum({
      aspectRatio: this.aspectRatio,
      fov: this.fov,
      near: this.near,
      far: this.far
    });
  }

  equals(other) {
    if (!defined(other) || !(other instanceof PerspectiveFrustum)) {
      return false;
    }

    this._update();

    other._update();

    return this.fov === other.fov && this.aspectRatio === other.aspectRatio && this.near === other.near && this.far === other.far && this._offCenterFrustum.equals(other._offCenterFrustum);
  }

  get projectionMatrix() {
    this._update();

    return this._offCenterFrustum.projectionMatrix;
  }

  get infiniteProjectionMatrix() {
    this._update();

    return this._offCenterFrustum.infiniteProjectionMatrix;
  }

  get fovy() {
    this._update();

    return this._fovy;
  }

  get sseDenominator() {
    this._update();

    return this._sseDenominator;
  }

  computeCullingVolume(position, direction, up) {
    this._update();

    return this._offCenterFrustum.computeCullingVolume(position, direction, up);
  }

  getPixelDimensions(drawingBufferWidth, drawingBufferHeight, distance, result) {
    this._update();

    return this._offCenterFrustum.getPixelDimensions(drawingBufferWidth, drawingBufferHeight, distance, result || new Vector2());
  }

  _update() {
    assert(Number.isFinite(this.fov) && Number.isFinite(this.aspectRatio) && Number.isFinite(this.near) && Number.isFinite(this.far));
    const f = this._offCenterFrustum;

    if (this.fov !== this._fov || this.aspectRatio !== this._aspectRatio || this.near !== this._near || this.far !== this._far || this.xOffset !== this._xOffset || this.yOffset !== this._yOffset) {
      assert(this.fov >= 0 && this.fov < Math.PI);
      assert(this.aspectRatio > 0);
      assert(this.near >= 0 && this.near < this.far);
      this._aspectRatio = this.aspectRatio;
      this._fov = this.fov;
      this._fovy = this.aspectRatio <= 1 ? this.fov : Math.atan(Math.tan(this.fov * 0.5) / this.aspectRatio) * 2.0;
      this._near = this.near;
      this._far = this.far;
      this._sseDenominator = 2.0 * Math.tan(0.5 * this._fovy);
      this._xOffset = this.xOffset;
      this._yOffset = this.yOffset;
      f.top = this.near * Math.tan(0.5 * this._fovy);
      f.bottom = -f.top;
      f.right = this.aspectRatio * f.top;
      f.left = -f.right;
      f.near = this.near;
      f.far = this.far;
      f.right += this.xOffset;
      f.left += this.xOffset;
      f.top += this.yOffset;
      f.bottom += this.yOffset;
    }
  }

}
//# sourceMappingURL=perspective-frustum.js.map