import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";
import { Vector3, Matrix4, assert } from '@math.gl/core';
import CullingVolume from './culling-volume';
import Plane from './plane';
const scratchPlaneUpVector = new Vector3();
const scratchPlaneRightVector = new Vector3();
const scratchPlaneNearCenter = new Vector3();
const scratchPlaneFarCenter = new Vector3();
const scratchPlaneNormal = new Vector3();
export default class PerspectiveOffCenterFrustum {
  constructor(options = {}) {
    _defineProperty(this, "left", void 0);

    _defineProperty(this, "_left", void 0);

    _defineProperty(this, "right", void 0);

    _defineProperty(this, "_right", void 0);

    _defineProperty(this, "top", void 0);

    _defineProperty(this, "_top", void 0);

    _defineProperty(this, "bottom", void 0);

    _defineProperty(this, "_bottom", void 0);

    _defineProperty(this, "near", void 0);

    _defineProperty(this, "_near", void 0);

    _defineProperty(this, "far", void 0);

    _defineProperty(this, "_far", void 0);

    _defineProperty(this, "_cullingVolume", new CullingVolume([new Plane(), new Plane(), new Plane(), new Plane(), new Plane(), new Plane()]));

    _defineProperty(this, "_perspectiveMatrix", new Matrix4());

    _defineProperty(this, "_infinitePerspective", new Matrix4());

    const {
      near = 1.0,
      far = 500000000.0
    } = options;
    this.left = options.left;
    this._left = undefined;
    this.right = options.right;
    this._right = undefined;
    this.top = options.top;
    this._top = undefined;
    this.bottom = options.bottom;
    this._bottom = undefined;
    this.near = near;
    this._near = near;
    this.far = far;
    this._far = far;
  }

  clone() {
    return new PerspectiveOffCenterFrustum({
      right: this.right,
      left: this.left,
      top: this.top,
      bottom: this.bottom,
      near: this.near,
      far: this.far
    });
  }

  equals(other) {
    return other && other instanceof PerspectiveOffCenterFrustum && this.right === other.right && this.left === other.left && this.top === other.top && this.bottom === other.bottom && this.near === other.near && this.far === other.far;
  }

  get projectionMatrix() {
    this._update();

    return this._perspectiveMatrix;
  }

  get infiniteProjectionMatrix() {
    this._update();

    return this._infinitePerspective;
  }

  computeCullingVolume(position, direction, up) {
    assert(position, 'position is required.');
    assert(direction, 'direction is required.');
    assert(up, 'up is required.');
    const planes = this._cullingVolume.planes;
    up = scratchPlaneUpVector.copy(up).normalize();
    const right = scratchPlaneRightVector.copy(direction).cross(up).normalize();
    const nearCenter = scratchPlaneNearCenter.copy(direction).multiplyByScalar(this.near).add(position);
    const farCenter = scratchPlaneFarCenter.copy(direction).multiplyByScalar(this.far).add(position);
    let normal = scratchPlaneNormal;
    normal.copy(right).multiplyByScalar(this.left).add(nearCenter).subtract(position).cross(up);
    planes[0].fromPointNormal(position, normal);
    normal.copy(right).multiplyByScalar(this.right).add(nearCenter).subtract(position).cross(up).negate();
    planes[1].fromPointNormal(position, normal);
    normal.copy(up).multiplyByScalar(this.bottom).add(nearCenter).subtract(position).cross(right).negate();
    planes[2].fromPointNormal(position, normal);
    normal.copy(up).multiplyByScalar(this.top).add(nearCenter).subtract(position).cross(right);
    planes[3].fromPointNormal(position, normal);
    normal = new Vector3().copy(direction);
    planes[4].fromPointNormal(nearCenter, normal);
    normal.negate();
    planes[5].fromPointNormal(farCenter, normal);
    return this._cullingVolume;
  }

  getPixelDimensions(drawingBufferWidth, drawingBufferHeight, distance, result) {
    this._update();

    assert(Number.isFinite(drawingBufferWidth) && Number.isFinite(drawingBufferHeight));
    assert(drawingBufferWidth > 0);
    assert(drawingBufferHeight > 0);
    assert(distance > 0);
    assert(result);
    const inverseNear = 1.0 / this.near;
    let tanTheta = this.top * inverseNear;
    const pixelHeight = 2.0 * distance * tanTheta / drawingBufferHeight;
    tanTheta = this.right * inverseNear;
    const pixelWidth = 2.0 * distance * tanTheta / drawingBufferWidth;
    result.x = pixelWidth;
    result.y = pixelHeight;
    return result;
  }

  _update() {
    assert(Number.isFinite(this.right) && Number.isFinite(this.left) && Number.isFinite(this.top) && Number.isFinite(this.bottom) && Number.isFinite(this.near) && Number.isFinite(this.far));
    const {
      top,
      bottom,
      right,
      left,
      near,
      far
    } = this;

    if (top !== this._top || bottom !== this._bottom || left !== this._left || right !== this._right || near !== this._near || far !== this._far) {
      assert(this.near > 0 && this.near < this.far, 'near must be greater than zero and less than far.');
      this._left = left;
      this._right = right;
      this._top = top;
      this._bottom = bottom;
      this._near = near;
      this._far = far;
      this._perspectiveMatrix = new Matrix4().frustum({
        left,
        right,
        bottom,
        top,
        near,
        far
      });
      this._infinitePerspective = new Matrix4().frustum({
        left,
        right,
        bottom,
        top,
        near,
        far: Infinity
      });
    }
  }

}
//# sourceMappingURL=perspective-off-center-frustum.js.map