import { Matrix4, NumericArray, Vector2 } from '@math.gl/core';
import CullingVolume from './culling-volume';
declare type PerspectiveFrustumOptions = {
    /** The angle of the field of view (FOV), in radians. */
    fov?: number;
    /** The aspect ratio of the frustum's width to it's height. */
    aspectRatio?: number;
    /** The distance of the near plane. */
    near?: number;
    /** The distance of the far plane. */
    far?: number;
    /** The offset in the x direction. */
    xOffset?: number;
    /** The offset in the y direction. */
    yOffset?: number;
};
/**
 * The viewing frustum is defined by 6 planes.
 * Each plane is represented by a {@link Vector4} object, where the x, y, and z components
 * define the unit vector normal to the plane, and the w component is the distance of the
 * plane from the origin/camera position.
 *
 * @alias PerspectiveFrustum
 *
 * @example
 * var frustum = new PerspectiveFrustum({
 *     fov : Math.PI_OVER_THREE,
 *     aspectRatio : canvas.clientWidth / canvas.clientHeight
 *     near : 1.0,
 *     far : 1000.0
 * });
 *
 * @see PerspectiveOffCenterFrustum
 */
export default class PerspectiveFrustum {
    private _offCenterFrustum;
    /**
     * The angle of the field of view (FOV), in radians.  This angle will be used
     * as the horizontal FOV if the width is greater than the height, otherwise
     * it will be the vertical FOV.
     */
    fov?: number;
    private _fov;
    private _fovy;
    private _sseDenominator;
    /**
     * The aspect ratio of the frustum's width to it's height.
     */
    aspectRatio?: number;
    private _aspectRatio;
    /**
     * The distance of the near plane.
     * @default 1.0
     */
    near: number;
    private _near;
    /**
     * The distance of the far plane.
     * @default 500000000.0
     */
    far: number;
    private _far;
    /**
     * Offsets the frustum in the x direction.
     * @default 0.0
     */
    xOffset: number;
    private _xOffset;
    /**
     * Offsets the frustum in the y direction.
     * @default 0.0
     */
    yOffset: number;
    private _yOffset;
    constructor(options?: PerspectiveFrustumOptions);
    /**
     * Returns a duplicate of a PerspectiveFrustum instance.
     */
    clone(): PerspectiveFrustum;
    /**
     * Compares the provided PerspectiveFrustum componentwise and returns
     * <code>true</code> if they are equal, <code>false</code> otherwise.
     */
    equals(other: PerspectiveFrustum): boolean;
    /**
     * Gets the perspective projection matrix computed from the view this.
     */
    get projectionMatrix(): Matrix4;
    /**
     * The perspective projection matrix computed from the view frustum with an infinite far plane.
     */
    get infiniteProjectionMatrix(): Matrix4;
    /**
     * Gets the angle of the vertical field of view, in radians.
     */
    get fovy(): number;
    /**
     * @private
     */
    get sseDenominator(): number;
    /**
     * Creates a culling volume for this this.ion.
     * @returns {CullingVolume} A culling volume at the given position and orientation.
     *
     * @example
     * // Check if a bounding volume intersects the this.
     * var cullingVolume = this.computeCullingVolume(cameraPosition, cameraDirection, cameraUp);
     * var intersect = cullingVolume.computeVisibility(boundingVolume);
     */
    computeCullingVolume(
    /** A Vector3 defines the eye position. */
    position: Readonly<NumericArray>, 
    /** A Vector3 defines the view direction. */
    direction: Readonly<NumericArray>, 
    /** A Vector3 defines the up direction. */
    up: Readonly<NumericArray>): CullingVolume;
    /**
     * Returns the pixel's width and height in meters.
     * @returns {Vector2} The modified result parameter or a new instance of {@link Vector2} with the pixel's width and height in the x and y properties, respectively.
     *
     * @exception {DeveloperError} drawingBufferWidth must be greater than zero.
     * @exception {DeveloperError} drawingBufferHeight must be greater than zero.
     *
     * @example
     * // Example 1
     * // Get the width and height of a pixel.
     * var pixelSize = camera.this.getPixelDimensions(scene.drawingBufferWidth, scene.drawingBufferHeight, 1.0, new Vector2());
     *
     * @example
     * // Example 2
     * // Get the width and height of a pixel if the near plane was set to 'distance'.
     * // For example, get the size of a pixel of an image on a billboard.
     * var position = camera.position;
     * var direction = camera.direction;
     * var toCenter = Vector3.subtract(primitive.boundingVolume.center, position, new Vector3());      // vector from camera to a primitive
     * var toCenterProj = Vector3.multiplyByScalar(direction, Vector3.dot(direction, toCenter), new Vector3()); // project vector onto camera direction vector
     * var distance = Vector3.magnitude(toCenterProj);
     * var pixelSize = camera.this.getPixelDimensions(scene.drawingBufferWidth, scene.drawingBufferHeight, distance, new Vector2());
     */
    getPixelDimensions(
    /** The width of the drawing buffer. */
    drawingBufferWidth: number, 
    /** The height of the drawing buffer. */
    drawingBufferHeight: number, 
    /** The distance to the near plane in meters. */
    distance: number, 
    /** The object onto which to store the result. */
    result?: Vector2): Vector2;
    private _update;
}
export {};
//# sourceMappingURL=perspective-frustum.d.ts.map