import { Vector2, Matrix4, NumericArray } from '@math.gl/core';
import CullingVolume from './culling-volume';
declare type PerspectiveOffCenterFrustumOptions = {
    left?: number;
    right?: number;
    top?: number;
    bottom?: number;
    near?: number;
    far?: number;
};
export default class PerspectiveOffCenterFrustum {
    /**
     * Defines the left clipping plane.
     * @type {Number}
     * @default undefined
     */
    left?: number;
    private _left?;
    /**
     * Defines the right clipping plane.
     * @type {Number}
     * @default undefined
     */
    right?: number;
    private _right?;
    /**
     * Defines the top clipping plane.
     * @type {Number}
     * @default undefined
     */
    top?: number;
    private _top?;
    /**
     * Defines the bottom clipping plane.
     * @type {Number}
     * @default undefined
     */
    bottom?: number;
    private _bottom?;
    /**
     * The distance of the near plane.
     * @type {Number}
     * @default 1.0
     */
    near: number;
    private _near;
    /**
     * The distance of the far plane.
     * @type {Number}
     * @default 500000000.0
     */
    far: number;
    private _far;
    private _cullingVolume;
    private _perspectiveMatrix;
    private _infinitePerspective;
    /**
     * The viewing frustum is defined by 6 planes.
     * Each plane is represented by a {@link Vector4} object, where the x, y, and z components
     * define the unit vector normal to the plane, and the w component is the distance of the
     * plane from the origin/camera position.
     *
     * @alias PerspectiveOffCenterFrustum
     *
     * @example
     * const frustum = new PerspectiveOffCenterFrustum({
     *     left : -1.0,
     *     right : 1.0,
     *     top : 1.0,
     *     bottom : -1.0,
     *     near : 1.0,
     *     far : 100.0
     * });
     *
     * @see PerspectiveFrustum
     */
    constructor(options?: PerspectiveOffCenterFrustumOptions);
    /**
     * Returns a duplicate of a PerspectiveOffCenterFrustum instance.
     * @returns {PerspectiveOffCenterFrustum} A new PerspectiveFrustum instance.
     * */
    clone(): PerspectiveOffCenterFrustum;
    /**
     * Compares the provided PerspectiveOffCenterFrustum componentwise and returns
     * <code>true</code> if they are equal, <code>false</code> otherwise.
     *
     * @returns {Boolean} <code>true</code> if they are equal, <code>false</code> otherwise.
     */
    equals(other: PerspectiveOffCenterFrustum): boolean;
    /**
     * Gets the perspective projection matrix computed from the view frustum.
     * @memberof PerspectiveOffCenterFrustum.prototype
     * @type {Matrix4}
     *
     * @see PerspectiveOffCenterFrustum#infiniteProjectionMatrix
     */
    get projectionMatrix(): Matrix4;
    /**
     * Gets the perspective projection matrix computed from the view frustum with an infinite far plane.
     * @memberof PerspectiveOffCenterFrustum.prototype
     * @type {Matrix4}
     *
     * @see PerspectiveOffCenterFrustum#projectionMatrix
     */
    get infiniteProjectionMatrix(): Matrix4;
    /**
     * Creates a culling volume for this frustum.
     * @returns {CullingVolume} A culling volume at the given position and orientation.
     *
     * @example
     * // Check if a bounding volume intersects the frustum.
     * const cullingVolume = frustum.computeCullingVolume(cameraPosition, cameraDirection, cameraUp);
     * const intersect = cullingVolume.computeVisibility(boundingVolume);
     */
    computeCullingVolume(
    /** A Vector3 defines the eye position. */
    position: Readonly<NumericArray>, 
    /** A Vector3 defines the view direction. */
    direction: Readonly<NumericArray>, 
    /** A Vector3 defines the up direction. */
    up: Readonly<NumericArray>): CullingVolume;
    /**
     * Returns the pixel's width and height in meters.
     *
     * @returns {Vector2} The modified result parameter or a new instance of {@link Vector2} with the pixel's width and height in the x and y properties, respectively.
     *
     * @exception {DeveloperError} drawingBufferWidth must be greater than zero.
     * @exception {DeveloperError} drawingBufferHeight must be greater than zero.
     *
     * @example
     * // Example 1
     * // Get the width and height of a pixel.
     * const pixelSize = camera.frustum.getPixelDimensions(scene.drawingBufferWidth, scene.drawingBufferHeight, 1.0, new Vector2());
     *
     * @example
     * // Example 2
     * // Get the width and height of a pixel if the near plane was set to 'distance'.
     * // For example, get the size of a pixel of an image on a billboard.
     * const position = camera.position;
     * const direction = camera.direction;
     * const toCenter = Vector3.subtract(primitive.boundingVolume.center, position, new Vector3());      // vector from camera to a primitive
     * const toCenterProj = Vector3.multiplyByScalar(direction, Vector3.dot(direction, toCenter), new Vector3()); // project vector onto camera direction vector
     * const distance = Vector3.magnitude(toCenterProj);
     * const pixelSize = camera.frustum.getPixelDimensions(scene.drawingBufferWidth, scene.drawingBufferHeight, distance, new Vector2());
     */
    getPixelDimensions(
    /** The width of the drawing buffer. */
    drawingBufferWidth: number, 
    /** The height of the drawing buffer. */
    drawingBufferHeight: number, 
    /** The distance to the near plane in meters. */
    distance: number, 
    /** The object onto which to store the result. */
    result: Vector2): Vector2;
    private _update;
}
export {};
//# sourceMappingURL=perspective-off-center-frustum.d.ts.map