"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fromCartographic = fromCartographic;
exports.fromCartographicToRadians = fromCartographicToRadians;
exports.fromCartographicToDegrees = fromCartographicToDegrees;
exports.toCartographic = toCartographic;
exports.toCartographicFromRadians = toCartographicFromRadians;
exports.toCartographicFromDegrees = toCartographicFromDegrees;
exports.isWGS84 = isWGS84;

var _core = require("@math.gl/core");

var _constants = require("./constants");

function identity(x) {
  return x;
}

var scratchVector = new _core.Vector3();

function fromCartographic(cartographic) {
  var result = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : [];
  var map = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : identity;

  if ('longitude' in cartographic) {
    result[0] = map(cartographic.longitude);
    result[1] = map(cartographic.latitude);
    result[2] = cartographic.height;
  } else if ('x' in cartographic) {
    result[0] = map(cartographic.x);
    result[1] = map(cartographic.y);
    result[2] = cartographic.z;
  } else {
    result[0] = map(cartographic[0]);
    result[1] = map(cartographic[1]);
    result[2] = cartographic[2];
  }

  return result;
}

function fromCartographicToRadians(cartographic) {
  var vector = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : [];
  return fromCartographic(cartographic, vector, _core.config._cartographicRadians ? identity : _core.toRadians);
}

function fromCartographicToDegrees(cartographic) {
  var vector = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : [];
  return fromCartographic(cartographic, vector, _core.config._cartographicRadians ? _core.toDegrees : identity);
}

function toCartographic(vector, cartographic) {
  var map = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : identity;

  if ('longitude' in cartographic) {
    cartographic.longitude = map(vector[0]);
    cartographic.latitude = map(vector[1]);
    cartographic.height = vector[2];
  } else if ('x' in cartographic) {
    cartographic.x = map(vector[0]);
    cartographic.y = map(vector[1]);
    cartographic.z = vector[2];
  } else {
    cartographic[0] = map(vector[0]);
    cartographic[1] = map(vector[1]);
    cartographic[2] = vector[2];
  }

  return cartographic;
}

function toCartographicFromRadians(vector, cartographic) {
  return toCartographic(vector, cartographic, _core.config._cartographicRadians ? identity : _core.toDegrees);
}

function toCartographicFromDegrees(vector, cartographic) {
  return toCartographic(vector, cartographic, _core.config._cartographicRadians ? _core.toRadians : identity);
}

function isWGS84(vector) {
  if (!vector) {
    return false;
  }

  scratchVector.from(vector);
  var oneOverRadiiSquared = _constants.WGS84_CONSTANTS.oneOverRadiiSquared,
      centerToleranceSquared = _constants.WGS84_CONSTANTS.centerToleranceSquared;
  var x2 = vector[0] * vector[0] * oneOverRadiiSquared[0];
  var y2 = vector[1] * vector[1] * oneOverRadiiSquared[1];
  var z2 = vector[2] * vector[2] * oneOverRadiiSquared[2];
  return Math.abs(x2 + y2 + z2 - 1) < centerToleranceSquared;
}
//# sourceMappingURL=type-utils.js.map