import _defineProperty from "@babel/runtime/helpers/esm/defineProperty";
import { Vector3, Matrix4, assert, equals, _MathUtils } from '@math.gl/core';
import * as vec3 from 'gl-matrix/vec3';
import { WGS84_RADIUS_X, WGS84_RADIUS_Y, WGS84_RADIUS_Z } from '../constants';
import { fromCartographicToRadians, toCartographicFromRadians } from '../type-utils';
import scaleToGeodeticSurface from './helpers/scale-to-geodetic-surface';
import localFrameToFixedFrame from './helpers/ellipsoid-transform';
const scratchVector = new Vector3();
const scratchNormal = new Vector3();
const scratchK = new Vector3();
const scratchPosition = new Vector3();
const scratchHeight = new Vector3();
const scratchCartesian = new Vector3();
let wgs84;
export default class Ellipsoid {
  constructor(x = 0.0, y = 0.0, z = 0.0) {
    _defineProperty(this, "radii", void 0);

    _defineProperty(this, "radiiSquared", void 0);

    _defineProperty(this, "radiiToTheFourth", void 0);

    _defineProperty(this, "oneOverRadii", void 0);

    _defineProperty(this, "oneOverRadiiSquared", void 0);

    _defineProperty(this, "minimumRadius", void 0);

    _defineProperty(this, "maximumRadius", void 0);

    _defineProperty(this, "centerToleranceSquared", _MathUtils.EPSILON1);

    _defineProperty(this, "squaredXOverSquaredZ", void 0);

    assert(x >= 0.0);
    assert(y >= 0.0);
    assert(z >= 0.0);
    this.radii = new Vector3(x, y, z);
    this.radiiSquared = new Vector3(x * x, y * y, z * z);
    this.radiiToTheFourth = new Vector3(x * x * x * x, y * y * y * y, z * z * z * z);
    this.oneOverRadii = new Vector3(x === 0.0 ? 0.0 : 1.0 / x, y === 0.0 ? 0.0 : 1.0 / y, z === 0.0 ? 0.0 : 1.0 / z);
    this.oneOverRadiiSquared = new Vector3(x === 0.0 ? 0.0 : 1.0 / (x * x), y === 0.0 ? 0.0 : 1.0 / (y * y), z === 0.0 ? 0.0 : 1.0 / (z * z));
    this.minimumRadius = Math.min(x, y, z);
    this.maximumRadius = Math.max(x, y, z);

    if (this.radiiSquared.z !== 0) {
      this.squaredXOverSquaredZ = this.radiiSquared.x / this.radiiSquared.z;
    }

    Object.freeze(this);
  }

  equals(right) {
    return this === right || Boolean(right && this.radii.equals(right.radii));
  }

  toString() {
    return this.radii.toString();
  }

  cartographicToCartesian(cartographic, result = [0, 0, 0]) {
    const normal = scratchNormal;
    const k = scratchK;
    const [,, height] = cartographic;
    this.geodeticSurfaceNormalCartographic(cartographic, normal);
    k.copy(this.radiiSquared).scale(normal);
    const gamma = Math.sqrt(normal.dot(k));
    k.scale(1 / gamma);
    normal.scale(height);
    k.add(normal);
    return k.to(result);
  }

  cartesianToCartographic(cartesian, result = [0, 0, 0]) {
    scratchCartesian.from(cartesian);
    const point = this.scaleToGeodeticSurface(scratchCartesian, scratchPosition);

    if (!point) {
      return undefined;
    }

    const normal = this.geodeticSurfaceNormal(point, scratchNormal);
    const h = scratchHeight;
    h.copy(scratchCartesian).subtract(point);
    const longitude = Math.atan2(normal.y, normal.x);
    const latitude = Math.asin(normal.z);
    const height = Math.sign(vec3.dot(h, scratchCartesian)) * vec3.length(h);
    return toCartographicFromRadians([longitude, latitude, height], result);
  }

  eastNorthUpToFixedFrame(origin, result = new Matrix4()) {
    return localFrameToFixedFrame(this, 'east', 'north', 'up', origin, result);
  }

  localFrameToFixedFrame(firstAxis, secondAxis, thirdAxis, origin, result = new Matrix4()) {
    return localFrameToFixedFrame(this, firstAxis, secondAxis, thirdAxis, origin, result);
  }

  geocentricSurfaceNormal(cartesian, result = [0, 0, 0]) {
    return scratchVector.from(cartesian).normalize().to(result);
  }

  geodeticSurfaceNormalCartographic(cartographic, result = [0, 0, 0]) {
    const cartographicVectorRadians = fromCartographicToRadians(cartographic);
    const longitude = cartographicVectorRadians[0];
    const latitude = cartographicVectorRadians[1];
    const cosLatitude = Math.cos(latitude);
    scratchVector.set(cosLatitude * Math.cos(longitude), cosLatitude * Math.sin(longitude), Math.sin(latitude)).normalize();
    return scratchVector.to(result);
  }

  geodeticSurfaceNormal(cartesian, result = [0, 0, 0]) {
    return scratchVector.from(cartesian).scale(this.oneOverRadiiSquared).normalize().to(result);
  }

  scaleToGeodeticSurface(cartesian, result) {
    return scaleToGeodeticSurface(cartesian, this, result);
  }

  scaleToGeocentricSurface(cartesian, result = [0, 0, 0]) {
    scratchPosition.from(cartesian);
    const positionX = scratchPosition.x;
    const positionY = scratchPosition.y;
    const positionZ = scratchPosition.z;
    const oneOverRadiiSquared = this.oneOverRadiiSquared;
    const beta = 1.0 / Math.sqrt(positionX * positionX * oneOverRadiiSquared.x + positionY * positionY * oneOverRadiiSquared.y + positionZ * positionZ * oneOverRadiiSquared.z);
    return scratchPosition.multiplyScalar(beta).to(result);
  }

  transformPositionToScaledSpace(position, result = [0, 0, 0]) {
    return scratchPosition.from(position).scale(this.oneOverRadii).to(result);
  }

  transformPositionFromScaledSpace(position, result = [0, 0, 0]) {
    return scratchPosition.from(position).scale(this.radii).to(result);
  }

  getSurfaceNormalIntersectionWithZAxis(position, buffer = 0, result = [0, 0, 0]) {
    assert(equals(this.radii.x, this.radii.y, _MathUtils.EPSILON15));
    assert(this.radii.z > 0);
    scratchPosition.from(position);
    const z = scratchPosition.z * (1 - this.squaredXOverSquaredZ);

    if (Math.abs(z) >= this.radii.z - buffer) {
      return undefined;
    }

    return scratchPosition.set(0.0, 0.0, z).to(result);
  }

}

_defineProperty(Ellipsoid, "WGS84", new Ellipsoid(WGS84_RADIUS_X, WGS84_RADIUS_Y, WGS84_RADIUS_Z));
//# sourceMappingURL=ellipsoid.js.map