"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.cutPolylineByGrid = cutPolylineByGrid;
exports.cutPolygonByGrid = cutPolygonByGrid;

var _lineclip = require("./lineclip");

var _utils = require("./utils");

function cutPolylineByGrid(positions, options) {
  var _ref = options || {},
      _ref$size = _ref.size,
      size = _ref$size === void 0 ? 2 : _ref$size,
      _ref$broken = _ref.broken,
      broken = _ref$broken === void 0 ? false : _ref$broken,
      _ref$gridResolution = _ref.gridResolution,
      gridResolution = _ref$gridResolution === void 0 ? 10 : _ref$gridResolution,
      _ref$gridOffset = _ref.gridOffset,
      gridOffset = _ref$gridOffset === void 0 ? [0, 0] : _ref$gridOffset,
      _ref$startIndex = _ref.startIndex,
      startIndex = _ref$startIndex === void 0 ? 0 : _ref$startIndex,
      _ref$endIndex = _ref.endIndex,
      endIndex = _ref$endIndex === void 0 ? positions.length : _ref$endIndex;

  var numPoints = (endIndex - startIndex) / size;
  var part = [];
  var result = [part];
  var a = (0, _utils.getPointAtIndex)(positions, 0, size, startIndex);
  var b;
  var codeB;
  var cell = getGridCell(a, gridResolution, gridOffset, []);
  var scratchPoint = [];
  (0, _utils.push)(part, a);

  for (var i = 1; i < numPoints; i++) {
    b = (0, _utils.getPointAtIndex)(positions, i, size, startIndex, b);
    codeB = (0, _lineclip.bitCode)(b, cell);

    while (codeB) {
      (0, _lineclip.intersect)(a, b, codeB, cell, scratchPoint);
      var codeAlt = (0, _lineclip.bitCode)(scratchPoint, cell);

      if (codeAlt) {
        (0, _lineclip.intersect)(a, scratchPoint, codeAlt, cell, scratchPoint);
        codeB = codeAlt;
      }

      (0, _utils.push)(part, scratchPoint);
      (0, _utils.copy)(a, scratchPoint);
      moveToNeighborCell(cell, gridResolution, codeB);

      if (broken && part.length > size) {
        part = [];
        result.push(part);
        (0, _utils.push)(part, a);
      }

      codeB = (0, _lineclip.bitCode)(b, cell);
    }

    (0, _utils.push)(part, b);
    (0, _utils.copy)(a, b);
  }

  return broken ? result : result[0];
}

var TYPE_INSIDE = 0;
var TYPE_BORDER = 1;

function concatInPlace(arr1, arr2) {
  for (var i = 0; i < arr2.length; i++) {
    arr1.push(arr2[i]);
  }

  return arr1;
}

function cutPolygonByGrid(positions) {
  var holeIndices = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : null;
  var options = arguments.length > 2 ? arguments[2] : undefined;

  if (!positions.length) {
    return [];
  }

  var _ref2 = options || {},
      _ref2$size = _ref2.size,
      size = _ref2$size === void 0 ? 2 : _ref2$size,
      _ref2$gridResolution = _ref2.gridResolution,
      gridResolution = _ref2$gridResolution === void 0 ? 10 : _ref2$gridResolution,
      _ref2$gridOffset = _ref2.gridOffset,
      gridOffset = _ref2$gridOffset === void 0 ? [0, 0] : _ref2$gridOffset,
      _ref2$edgeTypes = _ref2.edgeTypes,
      edgeTypes = _ref2$edgeTypes === void 0 ? false : _ref2$edgeTypes;

  var result = [];
  var queue = [{
    pos: positions,
    types: edgeTypes ? new Array(positions.length / size).fill(TYPE_BORDER) : null,
    holes: holeIndices || []
  }];
  var bbox = [[], []];
  var cell = [];

  while (queue.length) {
    var _queue$shift = queue.shift(),
        pos = _queue$shift.pos,
        types = _queue$shift.types,
        holes = _queue$shift.holes;

    getBoundingBox(pos, size, holes[0] || pos.length, bbox);
    cell = getGridCell(bbox[0], gridResolution, gridOffset, cell);
    var code = (0, _lineclip.bitCode)(bbox[1], cell);

    if (code) {
      var parts = bisectPolygon(pos, types, size, 0, holes[0] || pos.length, cell, code);
      var polygonLow = {
        pos: parts[0].pos,
        types: parts[0].types,
        holes: []
      };
      var polygonHigh = {
        pos: parts[1].pos,
        types: parts[1].types,
        holes: []
      };
      queue.push(polygonLow, polygonHigh);

      for (var i = 0; i < holes.length; i++) {
        parts = bisectPolygon(pos, types, size, holes[i], holes[i + 1] || pos.length, cell, code);

        if (parts[0]) {
          polygonLow.holes.push(polygonLow.pos.length);
          polygonLow.pos = concatInPlace(polygonLow.pos, parts[0].pos);

          if (edgeTypes) {
            polygonLow.types = concatInPlace(polygonLow.types, parts[0].types);
          }
        }

        if (parts[1]) {
          polygonHigh.holes.push(polygonHigh.pos.length);
          polygonHigh.pos = concatInPlace(polygonHigh.pos, parts[1].pos);

          if (edgeTypes) {
            polygonHigh.types = concatInPlace(polygonHigh.types, parts[1].types);
          }
        }
      }
    } else {
      var polygon = {
        positions: pos
      };

      if (edgeTypes) {
        polygon.edgeTypes = types;
      }

      if (holes.length) {
        polygon.holeIndices = holes;
      }

      result.push(polygon);
    }
  }

  return result;
}

function bisectPolygon(positions, edgeTypes, size, startIndex, endIndex, bbox, edge) {
  var numPoints = (endIndex - startIndex) / size;
  var resultLow = [];
  var resultHigh = [];
  var typesLow = [];
  var typesHigh = [];
  var scratchPoint = [];
  var p;
  var side;
  var type;
  var prev = (0, _utils.getPointAtIndex)(positions, numPoints - 1, size, startIndex);
  var prevSide = Math.sign(edge & 8 ? prev[1] - bbox[3] : prev[0] - bbox[2]);
  var prevType = edgeTypes && edgeTypes[numPoints - 1];
  var lowPointCount = 0;
  var highPointCount = 0;

  for (var i = 0; i < numPoints; i++) {
    p = (0, _utils.getPointAtIndex)(positions, i, size, startIndex, p);
    side = Math.sign(edge & 8 ? p[1] - bbox[3] : p[0] - bbox[2]);
    type = edgeTypes && edgeTypes[startIndex / size + i];

    if (side && prevSide && prevSide !== side) {
      (0, _lineclip.intersect)(prev, p, edge, bbox, scratchPoint);
      (0, _utils.push)(resultLow, scratchPoint) && typesLow.push(prevType);
      (0, _utils.push)(resultHigh, scratchPoint) && typesHigh.push(prevType);
    }

    if (side <= 0) {
      (0, _utils.push)(resultLow, p) && typesLow.push(type);
      lowPointCount -= side;
    } else if (typesLow.length) {
      typesLow[typesLow.length - 1] = TYPE_INSIDE;
    }

    if (side >= 0) {
      (0, _utils.push)(resultHigh, p) && typesHigh.push(type);
      highPointCount += side;
    } else if (typesHigh.length) {
      typesHigh[typesHigh.length - 1] = TYPE_INSIDE;
    }

    (0, _utils.copy)(prev, p);
    prevSide = side;
    prevType = type;
  }

  return [lowPointCount ? {
    pos: resultLow,
    types: edgeTypes && typesLow
  } : null, highPointCount ? {
    pos: resultHigh,
    types: edgeTypes && typesHigh
  } : null];
}

function getGridCell(p, gridResolution, gridOffset, out) {
  var left = Math.floor((p[0] - gridOffset[0]) / gridResolution) * gridResolution + gridOffset[0];
  var bottom = Math.floor((p[1] - gridOffset[1]) / gridResolution) * gridResolution + gridOffset[1];
  out[0] = left;
  out[1] = bottom;
  out[2] = left + gridResolution;
  out[3] = bottom + gridResolution;
  return out;
}

function moveToNeighborCell(cell, gridResolution, edge) {
  if (edge & 8) {
    cell[1] += gridResolution;
    cell[3] += gridResolution;
  } else if (edge & 4) {
    cell[1] -= gridResolution;
    cell[3] -= gridResolution;
  } else if (edge & 2) {
    cell[0] += gridResolution;
    cell[2] += gridResolution;
  } else if (edge & 1) {
    cell[0] -= gridResolution;
    cell[2] -= gridResolution;
  }
}

function getBoundingBox(positions, size, endIndex, out) {
  var minX = Infinity;
  var maxX = -Infinity;
  var minY = Infinity;
  var maxY = -Infinity;

  for (var i = 0; i < endIndex; i += size) {
    var x = positions[i];
    var y = positions[i + 1];
    minX = x < minX ? x : minX;
    maxX = x > maxX ? x : maxX;
    minY = y < minY ? y : minY;
    maxY = y > maxY ? y : maxY;
  }

  out[0][0] = minX;
  out[0][1] = minY;
  out[1][0] = maxX;
  out[1][1] = maxY;
  return out;
}
//# sourceMappingURL=cut-by-grid.js.map