import { cutPolylineByGrid, cutPolygonByGrid } from './cut-by-grid';
import { getPointAtIndex, push } from './utils';
const DEFAULT_MAX_LATITUDE = 85.051129;
export function cutPolylineByMercatorBounds(positions, options) {
  const {
    size = 2,
    startIndex = 0,
    endIndex = positions.length,
    normalize = true
  } = options || {};
  const newPositions = positions.slice(startIndex, endIndex);
  wrapLongitudesForShortestPath(newPositions, size, 0, endIndex - startIndex);
  const parts = cutPolylineByGrid(newPositions, {
    size,
    broken: true,
    gridResolution: 360,
    gridOffset: [-180, -180]
  });

  if (normalize) {
    for (const part of parts) {
      shiftLongitudesIntoRange(part, size);
    }
  }

  return parts;
}
export function cutPolygonByMercatorBounds(positions, holeIndices = null, options) {
  const {
    size = 2,
    normalize = true,
    edgeTypes = false
  } = options || {};
  holeIndices = holeIndices || [];
  const newPositions = [];
  const newHoleIndices = [];
  let srcStartIndex = 0;
  let targetIndex = 0;

  for (let ringIndex = 0; ringIndex <= holeIndices.length; ringIndex++) {
    const srcEndIndex = holeIndices[ringIndex] || positions.length;
    const targetStartIndex = targetIndex;
    const splitIndex = findSplitIndex(positions, size, srcStartIndex, srcEndIndex);

    for (let i = splitIndex; i < srcEndIndex; i++) {
      newPositions[targetIndex++] = positions[i];
    }

    for (let i = srcStartIndex; i < splitIndex; i++) {
      newPositions[targetIndex++] = positions[i];
    }

    wrapLongitudesForShortestPath(newPositions, size, targetStartIndex, targetIndex);
    insertPoleVertices(newPositions, size, targetStartIndex, targetIndex, options === null || options === void 0 ? void 0 : options.maxLatitude);
    srcStartIndex = srcEndIndex;
    newHoleIndices[ringIndex] = targetIndex;
  }

  newHoleIndices.pop();
  const parts = cutPolygonByGrid(newPositions, newHoleIndices, {
    size,
    gridResolution: 360,
    gridOffset: [-180, -180],
    edgeTypes
  });

  if (normalize) {
    for (const part of parts) {
      shiftLongitudesIntoRange(part.positions, size);
    }
  }

  return parts;
}

function findSplitIndex(positions, size, startIndex, endIndex) {
  let maxLat = -1;
  let pointIndex = -1;

  for (let i = startIndex + 1; i < endIndex; i += size) {
    const lat = Math.abs(positions[i]);

    if (lat > maxLat) {
      maxLat = lat;
      pointIndex = i - 1;
    }
  }

  return pointIndex;
}

function insertPoleVertices(positions, size, startIndex, endIndex, maxLatitude = DEFAULT_MAX_LATITUDE) {
  const firstLng = positions[startIndex];
  const lastLng = positions[endIndex - size];

  if (Math.abs(firstLng - lastLng) > 180) {
    const p = getPointAtIndex(positions, 0, size, startIndex);
    p[0] += Math.round((lastLng - firstLng) / 360) * 360;
    push(positions, p);
    p[1] = Math.sign(p[1]) * maxLatitude;
    push(positions, p);
    p[0] = firstLng;
    push(positions, p);
  }
}

function wrapLongitudesForShortestPath(positions, size, startIndex, endIndex) {
  let prevLng = positions[0];
  let lng;

  for (let i = startIndex; i < endIndex; i += size) {
    lng = positions[i];
    const delta = lng - prevLng;

    if (delta > 180 || delta < -180) {
      lng -= Math.round(delta / 360) * 360;
    }

    positions[i] = prevLng = lng;
  }
}

function shiftLongitudesIntoRange(positions, size) {
  let refLng;
  const pointCount = positions.length / size;

  for (let i = 0; i < pointCount; i++) {
    refLng = positions[i * size];

    if ((refLng + 180) % 360 !== 0) {
      break;
    }
  }

  const delta = -Math.round(refLng / 360) * 360;

  if (delta === 0) {
    return;
  }

  for (let i = 0; i < pointCount; i++) {
    positions[i * size] += delta;
  }
}
//# sourceMappingURL=cut-by-mercator-bounds.js.map