import type { NumericArray } from '@math.gl/core';
export declare const WINDING: {
    readonly CLOCKWISE: 1;
    readonly COUNTER_CLOCKWISE: -1;
};
/** Polygon representation where each point is represented as a separate array of positions. */
declare type PointsArray = NumericArray[];
/** Segment visitor callback type for polygons defined with flat arrays, */
declare type SegmentVisitorFlat = (p1x: number, p1y: number, p2x: number, p2y: number, i1: number, i2: number) => void;
/** Segment visitor callback type for polygons defined with array of points. */
export declare type SegmentVisitorPoints = (p1: NumericArray, p2: NumericArray, i1: number, i2: number) => void;
/** Parameters of a polygon. */
declare type PolygonParams = {
    start?: number;
    end?: number;
    size?: number;
    isClosed?: boolean;
};
/**
 * Checks winding direction of the polygon and reverses the polygon in case of opposite winding direction.
 * Note: points are modified in-place.
 * @param points An array that represents points of the polygon.
 * @param direction Requested winding direction. 1 is for clockwise, -1 for counterclockwise winding direction.
 * @param options Parameters of the polygon.
 * @return Returns true if the winding direction was changed.
 */
export declare function modifyPolygonWindingDirection(points: NumericArray, direction: number, options?: PolygonParams): boolean;
/**
 * Returns winding direction of the polygon.
 * @param points An array that represents points of the polygon.
 * @param options Parameters of the polygon.
 * @returns Winding direction of the polygon.
 */
export declare function getPolygonWindingDirection(points: NumericArray, options?: PolygonParams): number;
/**
 * Returns signed area of the polygon.
 * @param points An array that represents points of the polygon.
 * @param options Parameters of the polygon.
 * @returns Signed area of the polygon.
 * https://en.wikipedia.org/wiki/Shoelace_formula
 */
export declare function getPolygonSignedArea(points: NumericArray, options?: PolygonParams): number;
/**
 * Calls the visitor callback for each segment in the polygon.
 * @param points An array that represents points of the polygon
 * @param visitor A callback to call for each segment.
 * @param options Parameters of the polygon.
 */
export declare function forEachSegmentInPolygon(points: NumericArray, visitor: SegmentVisitorFlat, options?: PolygonParams): void;
/**
 * Checks winding direction of the polygon and reverses the polygon in case of opposite winding direction.
 * Note: points are modified in-place.
 * @param points Array of points that represent the polygon.
 * @param direction Requested winding direction. 1 is for clockwise, -1 for counterclockwise winding direction.
 * @param options Parameters of the polygon.
 * @return Returns true if the winding direction was changed.
 */
export declare function modifyPolygonWindingDirectionPoints(points: PointsArray, direction: number, options?: PolygonParams): boolean;
/**
 * Returns winding direction of the polygon.
 * @param points Array of points that represent the polygon.
 * @param options Parameters of the polygon.
 * @returns Winding direction of the polygon.
 */
export declare function getPolygonWindingDirectionPoints(points: PointsArray, options?: PolygonParams): number;
/**
 * Returns signed area of the polygon.
 * @param points Array of points that represent the polygon.
 * @param options Parameters of the polygon.
 * @returns Signed area of the polygon.
 */
export declare function getPolygonSignedAreaPoints(points: PointsArray, options?: PolygonParams): number;
/**
 * Calls visitor callback for each segment in the polygon.
 * @param points Array of points that represent the polygon.
 * @param visitor A callback to call for each segment.
 * @param options Parameters of the polygon.
 */
export declare function forEachSegmentInPolygonPoints(points: PointsArray, visitor: SegmentVisitorPoints, options?: PolygonParams): void;
export {};
//# sourceMappingURL=polygon-utils.d.ts.map