"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _apply = require("babel-runtime/core-js/reflect/apply");

var _apply2 = _interopRequireDefault(_apply);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Since some events can fire at a high rate, the event handler should be limited to execute computationally
 * expensive operations, such as DOM modifications, inside a single rendered frame.
 * When listening to e.g. scroll and resize events, the browser tends to fire off more events per
 * second than are actually useful. For instance, if your event listener sets some element positions, then it
 * is possible for those positions to be updated multiple times in a single rendered frame. In this case, all of
 * the layout calculations triggered by setting the elements' positions will be wasted except for the one time that
 * it runs immediately prior to the browser rendering the updated layout to the screen.
 * To avoid wasting cycles, we can use requestAnimationFrame to only run the event listener once just before the page
 * is rendered to the screen.
 * *
 * @param callback the function to throttle
 * @param context  optional context of this, default to global
 * @return {function(...[*])}
 */
var fullThrottle = function fullThrottle(callback, context) {

  if (!context) {
    context = undefined || window;
  }

  var throttling = false;

  return function () {
    for (var _len = arguments.length, args = Array(_len), _key = 0; _key < _len; _key++) {
      args[_key] = arguments[_key];
    }

    if (!throttling) {
      throttling = true;
      window.requestAnimationFrame(function () {
        throttling = false;
        return (0, _apply2.default)(callback, context, args);
      });
    }
  };
};

exports.default = fullThrottle;
module.exports = exports["default"];