"use strict";
// Copyright (c) Microsoft Corporation. All rights reserved. Licensed under the MIT license.
// See LICENSE in the project root for license information.
Object.defineProperty(exports, "__esModule", { value: true });
const tsdoc = require("@microsoft/tsdoc");
const ApiItem_1 = require("./ApiItem");
const AedocDefinitions_1 = require("../aedoc/AedocDefinitions");
/**
 * An abstract base class for API declarations that can have an associated TSDoc comment.
 *
 * @remarks
 *
 * This is part of the {@link ApiModel} hierarchy of classes, which are serializable representations of
 * API declarations.
 *
 * @public
 */
class ApiDocumentedItem extends ApiItem_1.ApiItem {
    constructor(options) {
        super(options);
        this._tsdocComment = options.docComment;
    }
    /** @override */
    static onDeserializeInto(options, context, jsonObject) {
        super.onDeserializeInto(options, context, jsonObject);
        const documentedJson = jsonObject;
        if (documentedJson.docComment) {
            const tsdocParser = new tsdoc.TSDocParser(AedocDefinitions_1.AedocDefinitions.tsdocConfiguration);
            // NOTE: For now, we ignore TSDoc errors found in a serialized .api.json file.
            // Normally these errors would have already been reported by API Extractor during analysis.
            // However, they could also arise if the JSON file was edited manually, or if the file was saved
            // using a different release of the software that used an incompatible syntax.
            const parserContext = tsdocParser.parseString(documentedJson.docComment);
            options.docComment = parserContext.docComment;
        }
    }
    get tsdocComment() {
        return this._tsdocComment;
    }
    /** @override */
    serializeInto(jsonObject) {
        super.serializeInto(jsonObject);
        if (this.tsdocComment !== undefined) {
            jsonObject.docComment = this.tsdocComment.emitAsTsdoc();
        }
        else {
            jsonObject.docComment = '';
        }
    }
}
exports.ApiDocumentedItem = ApiDocumentedItem;
//# sourceMappingURL=ApiDocumentedItem.js.map