"use strict";
// Copyright (c) Microsoft Corporation. All rights reserved. Licensed under the MIT license.
// See LICENSE in the project root for license information.
Object.defineProperty(exports, "__esModule", { value: true });
const node_core_library_1 = require("@rushstack/node-core-library");
/** @public */
class ExcerptToken {
    constructor(kind, text, canonicalReference) {
        this._kind = kind;
        // Standardize the newlines across operating systems. Even though this may deviate from the actual
        // input source file that was parsed, it's useful because the newline gets serialized inside
        // a string literal in .api.json, which cannot be automatically normalized by Git.
        this._text = node_core_library_1.Text.convertToLf(text);
        this._canonicalReference = canonicalReference;
    }
    get kind() {
        return this._kind;
    }
    get text() {
        return this._text;
    }
    get canonicalReference() {
        return this._canonicalReference;
    }
}
exports.ExcerptToken = ExcerptToken;
/**
 * This class is used by {@link ApiDeclaredItem} to represent a source code excerpt containing
 * a TypeScript declaration.
 *
 * @remarks
 *
 * The main excerpt is parsed into an array of tokens, and the main excerpt's token range will span all of these
 * tokens.  The declaration may also have have "captured" excerpts, which are other subranges of tokens.
 * For example, if the main excerpt is a function declaration, it will also have a captured excerpt corresponding
 * to the return type of the function.
 *
 * An excerpt may be empty (i.e. a token range containing zero tokens).  For example, if a function's return value
 * is not explicitly declared, then the returnTypeExcerpt will be empty.  By contrast, a class constructor cannot
 * have a return value, so ApiConstructor has no returnTypeExcerpt property at all.
 *
 * @public
 */
class Excerpt {
    constructor(tokens, tokenRange) {
        this.tokens = tokens;
        this.tokenRange = tokenRange;
        if (this.tokenRange.startIndex < 0 || this.tokenRange.endIndex > this.tokens.length
            || this.tokenRange.startIndex > this.tokenRange.endIndex) {
            throw new Error('Invalid token range');
        }
    }
    get text() {
        if (this._text === undefined) {
            this._text = this.tokens.slice(this.tokenRange.startIndex, this.tokenRange.endIndex)
                .map(x => x.text).join('');
        }
        return this._text;
    }
    get isEmpty() {
        return this.tokenRange.startIndex === this.tokenRange.endIndex;
    }
}
exports.Excerpt = Excerpt;
//# sourceMappingURL=Excerpt.js.map