"use strict";
// Copyright (c) Microsoft Corporation. All rights reserved. Licensed under the MIT license.
// See LICENSE in the project root for license information.
Object.defineProperty(exports, "__esModule", { value: true });
const node_core_library_1 = require("@rushstack/node-core-library");
/**
 * Indicates the import kind for an `AstImport`.
 */
var AstImportKind;
(function (AstImportKind) {
    /**
     * An import statement such as `import X from "y";`.
     */
    AstImportKind[AstImportKind["DefaultImport"] = 0] = "DefaultImport";
    /**
     * An import statement such as `import { X } from "y";`.
     */
    AstImportKind[AstImportKind["NamedImport"] = 1] = "NamedImport";
    /**
     * An import statement such as `import * as x from "y";`.
     */
    AstImportKind[AstImportKind["StarImport"] = 2] = "StarImport";
    /**
     * An import statement such as `import x = require("y");`.
     */
    AstImportKind[AstImportKind["EqualsImport"] = 3] = "EqualsImport";
})(AstImportKind = exports.AstImportKind || (exports.AstImportKind = {}));
/**
 * For a symbol that was imported from an external package, this tracks the import
 * statement that was used to reach it.
 */
class AstImport {
    constructor(options) {
        this.importKind = options.importKind;
        this.modulePath = options.modulePath;
        this.exportName = options.exportName;
        this.key = AstImport.getKey(options);
    }
    /**
     * Allows `AstEntity.localName` to be used as a convenient generalization of `AstSymbol.localName` and
     * `AstImport.exportName`.
     */
    get localName() {
        return this.exportName;
    }
    /**
     * Calculates the lookup key used with `AstImport.key`
     */
    static getKey(options) {
        switch (options.importKind) {
            case AstImportKind.DefaultImport:
                return `${options.modulePath}:${options.exportName}`;
            case AstImportKind.NamedImport:
                return `${options.modulePath}:${options.exportName}`;
            case AstImportKind.StarImport:
                return `${options.modulePath}:*`;
            case AstImportKind.EqualsImport:
                return `${options.modulePath}:=`;
            default:
                throw new node_core_library_1.InternalError('Unknown AstImportKind');
        }
    }
}
exports.AstImport = AstImport;
//# sourceMappingURL=AstImport.js.map