"use strict";
// Copyright (c) Microsoft Corporation. All rights reserved. Licensed under the MIT license.
// See LICENSE in the project root for license information.
Object.defineProperty(exports, "__esModule", { value: true });
const colors = require("colors");
const os = require("os");
const path = require("path");
const node_core_library_1 = require("@rushstack/node-core-library");
const ts_command_line_1 = require("@rushstack/ts-command-line");
const Extractor_1 = require("../api/Extractor");
const ExtractorConfig_1 = require("../api/ExtractorConfig");
class RunAction extends ts_command_line_1.CommandLineAction {
    constructor(parser) {
        super({
            actionName: 'run',
            summary: 'Invoke API Extractor on a project',
            documentation: 'Invoke API Extractor on a project'
        });
    }
    onDefineParameters() {
        this._configFileParameter = this.defineStringParameter({
            parameterLongName: '--config',
            parameterShortName: '-c',
            argumentName: 'FILE',
            description: `Use the specified ${ExtractorConfig_1.ExtractorConfig.FILENAME} file path, rather than guessing its location`
        });
        this._localParameter = this.defineFlagParameter({
            parameterLongName: '--local',
            parameterShortName: '-l',
            description: 'Indicates that API Extractor is running as part of a local build,'
                + ' e.g. on a developer\'s machine. This disables certain validation that would'
                + ' normally be performed for a ship/production build. For example, the *.api.md'
                + ' report file is automatically copied in a local build.'
        });
        this._verboseParameter = this.defineFlagParameter({
            parameterLongName: '--verbose',
            parameterShortName: '-v',
            description: 'Show additional informational messages in the output.'
        });
        this._diagnosticsParameter = this.defineFlagParameter({
            parameterLongName: '--diagnostics',
            description: 'Show diagnostic messages used for troubleshooting problems with API Extractor.'
                + '  This flag also enables the "--verbose" flag.'
        });
        this._typescriptCompilerFolder = this.defineStringParameter({
            parameterLongName: '--typescript-compiler-folder',
            argumentName: 'PATH',
            description: 'API Extractor uses its own TypeScript compiler engine to analyze your project.  If your project'
                + ' is built with a significantly different TypeScript version, sometimes API Extractor may report compilation'
                + ' errors due to differences in the system typings (e.g. lib.dom.d.ts).  You can use the'
                + ' "--typescriptCompilerFolder" option to specify the folder path where you installed the TypeScript package,'
                + ' and API Extractor\'s compiler will use those system typings instead.'
        });
    }
    onExecute() {
        const lookup = new node_core_library_1.PackageJsonLookup();
        let configFilename;
        let typescriptCompilerFolder = this._typescriptCompilerFolder.value;
        if (typescriptCompilerFolder) {
            typescriptCompilerFolder = path.normalize(typescriptCompilerFolder);
            if (node_core_library_1.FileSystem.exists(typescriptCompilerFolder)) {
                typescriptCompilerFolder = lookup.tryGetPackageFolderFor(typescriptCompilerFolder);
                const typescriptCompilerPackageJson = typescriptCompilerFolder
                    ? lookup.tryLoadPackageJsonFor(typescriptCompilerFolder)
                    : undefined;
                if (!typescriptCompilerPackageJson) {
                    throw new Error(`The path specified in the ${this._typescriptCompilerFolder.longName} parameter is not a package.`);
                }
                else if (typescriptCompilerPackageJson.name !== 'typescript') {
                    throw new Error(`The path specified in the ${this._typescriptCompilerFolder.longName} parameter is not a TypeScript`
                        + ' compiler package.');
                }
            }
            else {
                throw new Error(`The path specified in the ${this._typescriptCompilerFolder.longName} parameter does not exist.`);
            }
        }
        if (this._configFileParameter.value) {
            configFilename = path.normalize(this._configFileParameter.value);
            if (!node_core_library_1.FileSystem.exists(configFilename)) {
                throw new Error('Config file not found: ' + this._configFileParameter.value);
            }
        }
        else {
            // Otherwise, figure out which project we're in and look for the config file
            // at the project root
            const packageFolder = lookup.tryGetPackageFolderFor('.');
            // If there is no package, then try the current directory
            const baseFolder = packageFolder ? packageFolder : process.cwd();
            // First try the standard "config" subfolder:
            configFilename = path.join(baseFolder, 'config', ExtractorConfig_1.ExtractorConfig.FILENAME);
            if (node_core_library_1.FileSystem.exists(configFilename)) {
                if (node_core_library_1.FileSystem.exists(path.join(baseFolder, ExtractorConfig_1.ExtractorConfig.FILENAME))) {
                    throw new Error(`Found conflicting ${ExtractorConfig_1.ExtractorConfig.FILENAME} files in "." and "./config" folders`);
                }
            }
            else {
                // Otherwise try the top-level folder
                configFilename = path.join(baseFolder, ExtractorConfig_1.ExtractorConfig.FILENAME);
                if (!node_core_library_1.FileSystem.exists(configFilename)) {
                    throw new Error(`Unable to find an ${ExtractorConfig_1.ExtractorConfig.FILENAME} file`);
                }
            }
            console.log(`Using configuration from ${configFilename}` + os.EOL);
        }
        const configObjectFullPath = path.resolve(configFilename);
        const configObject = ExtractorConfig_1.ExtractorConfig.loadFile(configObjectFullPath);
        const extractorConfig = ExtractorConfig_1.ExtractorConfig.prepare({
            configObject: configObject,
            configObjectFullPath: configObjectFullPath,
            packageJsonFullPath: lookup.tryGetPackageJsonFilePathFor(configObjectFullPath)
        });
        const extractorResult = Extractor_1.Extractor.invoke(extractorConfig, {
            localBuild: this._localParameter.value,
            showVerboseMessages: this._verboseParameter.value,
            showDiagnostics: this._diagnosticsParameter.value,
            typescriptCompilerFolder: typescriptCompilerFolder
        });
        if (extractorResult.succeeded) {
            console.log(os.EOL + 'API Extractor completed successfully');
        }
        else {
            process.exitCode = 1;
            if (extractorResult.errorCount > 0) {
                console.log(os.EOL + colors.red('API Extractor completed with errors'));
            }
            else {
                console.log(os.EOL + colors.yellow('API Extractor completed with warnings'));
            }
        }
        return Promise.resolve();
    }
}
exports.RunAction = RunAction;
//# sourceMappingURL=RunAction.js.map