import * as ts from 'typescript';
import { PackageJsonLookup } from '@rushstack/node-core-library';
import { CollectorEntity } from './CollectorEntity';
import { AstSymbolTable, AstEntity } from '../analyzer/AstSymbolTable';
import { AstSymbol } from '../analyzer/AstSymbol';
import { AstDeclaration } from '../analyzer/AstDeclaration';
import { WorkingPackage } from './WorkingPackage';
import { DeclarationMetadata } from './DeclarationMetadata';
import { ApiItemMetadata } from './ApiItemMetadata';
import { SymbolMetadata } from './SymbolMetadata';
import { MessageRouter } from './MessageRouter';
import { AstReferenceResolver } from '../analyzer/AstReferenceResolver';
import { ExtractorConfig } from '../api/ExtractorConfig';
/**
 * Options for Collector constructor.
 */
export interface ICollectorOptions {
    /**
     * Configuration for the TypeScript compiler.  The most important options to set are:
     *
     * - target: ts.ScriptTarget.ES5
     * - module: ts.ModuleKind.CommonJS
     * - moduleResolution: ts.ModuleResolutionKind.NodeJs
     * - rootDir: inputFolder
     */
    program: ts.Program;
    messageRouter: MessageRouter;
    extractorConfig: ExtractorConfig;
}
/**
 * The `Collector` manages the overall data set that is used by `ApiModelGenerator`,
 * `DtsRollupGenerator`, and `ApiReportGenerator`.  Starting from the working package's entry point,
 * the `Collector` collects all exported symbols, determines how to import any symbols they reference,
 * assigns unique names, and sorts everything into a normalized alphabetical ordering.
 */
export declare class Collector {
    readonly program: ts.Program;
    readonly typeChecker: ts.TypeChecker;
    readonly astSymbolTable: AstSymbolTable;
    readonly astReferenceResolver: AstReferenceResolver;
    readonly packageJsonLookup: PackageJsonLookup;
    readonly messageRouter: MessageRouter;
    readonly workingPackage: WorkingPackage;
    readonly extractorConfig: ExtractorConfig;
    private readonly _program;
    private readonly _tsdocParser;
    private _astEntryPoint;
    private readonly _entities;
    private readonly _entitiesByAstEntity;
    private readonly _starExportedExternalModulePaths;
    private readonly _dtsTypeReferenceDirectives;
    private readonly _dtsLibReferenceDirectives;
    private readonly _cachedOverloadIndexesByDeclaration;
    constructor(options: ICollectorOptions);
    /**
     * Returns a list of names (e.g. "example-library") that should appear in a reference like this:
     *
     * ```
     * /// <reference types="example-library" />
     * ```
     */
    readonly dtsTypeReferenceDirectives: ReadonlySet<string>;
    /**
     * A list of names (e.g. "runtime-library") that should appear in a reference like this:
     *
     * ```
     * /// <reference lib="runtime-library" />
     * ```
     */
    readonly dtsLibReferenceDirectives: ReadonlySet<string>;
    readonly entities: ReadonlyArray<CollectorEntity>;
    /**
     * A list of module specifiers (e.g. `"@rushstack/node-core-library/lib/FileSystem"`) that should be emitted
     * as star exports (e.g. `export * from "@rushstack/node-core-library/lib/FileSystem"`).
     */
    readonly starExportedExternalModulePaths: ReadonlyArray<string>;
    /**
     * Perform the analysis.
     */
    analyze(): void;
    /**
     * For a given ts.Identifier that is part of an AstSymbol that we analyzed, return the CollectorEntity that
     * it refers to.  Returns undefined if it doesn't refer to anything interesting.
     * @remarks
     * Throws an Error if the ts.Identifier is not part of node tree that was analyzed.
     */
    tryGetEntityForIdentifierNode(identifier: ts.Identifier): CollectorEntity | undefined;
    /**
     * Returns the associated `CollectorEntity` for the given `astEntity`, if one was created during analysis.
     */
    tryGetCollectorEntity(astEntity: AstEntity): CollectorEntity | undefined;
    fetchSymbolMetadata(astSymbol: AstSymbol): SymbolMetadata;
    fetchDeclarationMetadata(astDeclaration: AstDeclaration): DeclarationMetadata;
    fetchApiItemMetadata(astDeclaration: AstDeclaration): ApiItemMetadata;
    tryFetchMetadataForAstEntity(astEntity: AstEntity): SymbolMetadata | undefined;
    isAncillaryDeclaration(astDeclaration: AstDeclaration): boolean;
    getNonAncillaryDeclarations(astSymbol: AstSymbol): ReadonlyArray<AstDeclaration>;
    /**
     * Removes the leading underscore, for example: "_Example" --> "example*Example*_"
     *
     * @remarks
     * This causes internal definitions to sort alphabetically case-insensitive, then case-sensitive, and
     * initially ignoring the underscore prefix, while still deterministically comparing it.
     * The star is used as a delimiter because it is not a legal  identifier character.
     */
    static getSortKeyIgnoringUnderscore(identifier: string): string;
    /**
     * For function-like signatures, this returns the TSDoc "overload index" which can be used to identify
     * a specific overload.
     */
    getOverloadIndex(astDeclaration: AstDeclaration): number;
    private _createCollectorEntity;
    private _createEntityForIndirectReferences;
    /**
     * Ensures a unique name for each item in the package typings file.
     */
    private _makeUniqueNames;
    /**
     * Adds global names to the usedNames set, to prevent API Extractor from emitting names that conflict with
     * a global name.
     */
    private _collectGlobalNames;
    private _fetchSymbolMetadata;
    private _calculateDeclarationMetadataForDeclarations;
    private _addAncillaryDeclaration;
    private _calculateApiItemMetadata;
    private _parseTsdocForAstDeclaration;
    private _collectReferenceDirectives;
}
//# sourceMappingURL=Collector.d.ts.map