"use strict";
// Copyright (c) Microsoft Corporation. All rights reserved. Licensed under the MIT license.
// See LICENSE in the project root for license information.
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const ts = require("typescript");
const AstSymbol_1 = require("../analyzer/AstSymbol");
const api_extractor_model_1 = require("@microsoft/api-extractor-model");
class ValidationEnhancer {
    static analyze(collector) {
        const alreadyWarnedSymbols = new Set();
        for (const entity of collector.entities) {
            if (entity.astEntity instanceof AstSymbol_1.AstSymbol) {
                if (entity.exported) {
                    entity.astEntity.forEachDeclarationRecursive((astDeclaration) => {
                        ValidationEnhancer._checkReferences(collector, astDeclaration, alreadyWarnedSymbols);
                    });
                    const symbolMetadata = collector.fetchSymbolMetadata(entity.astEntity);
                    ValidationEnhancer._checkForInternalUnderscore(collector, entity, entity.astEntity, symbolMetadata);
                    ValidationEnhancer._checkForInconsistentReleaseTags(collector, entity.astEntity, symbolMetadata);
                }
            }
        }
    }
    static _checkForInternalUnderscore(collector, collectorEntity, astSymbol, symbolMetadata) {
        let needsUnderscore = false;
        if (symbolMetadata.maxEffectiveReleaseTag === api_extractor_model_1.ReleaseTag.Internal) {
            if (!astSymbol.parentAstSymbol) {
                // If it's marked as @internal and has no parent, then it needs and underscore.
                // We use maxEffectiveReleaseTag because a merged declaration would NOT need an underscore in a case like this:
                //
                //   /** @public */
                //   export enum X { }
                //
                //   /** @internal */
                //   export namespace X { }
                //
                // (The above normally reports an error "ae-different-release-tags", but that may be suppressed.)
                needsUnderscore = true;
            }
            else {
                // If it's marked as @internal and the parent isn't obviously already @internal, then it needs an underscore.
                //
                // For example, we WOULD need an underscore for a merged declaration like this:
                //
                //   /** @internal */
                //   export namespace X {
                //     export interface _Y { }
                //   }
                //
                //   /** @public */
                //   export class X {
                //     /** @internal */
                //     public static _Y(): void { }   // <==== different from parent
                //   }
                const parentSymbolMetadata = collector.fetchSymbolMetadata(astSymbol);
                if (parentSymbolMetadata.maxEffectiveReleaseTag > api_extractor_model_1.ReleaseTag.Internal) {
                    needsUnderscore = true;
                }
            }
        }
        if (needsUnderscore) {
            for (const exportName of collectorEntity.exportNames) {
                if (exportName[0] !== '_') {
                    collector.messageRouter.addAnalyzerIssue("ae-internal-missing-underscore" /* InternalMissingUnderscore */, `The name "${exportName}" should be prefixed with an underscore`
                        + ` because the declaration is marked as @internal`, astSymbol, { exportName });
                }
            }
        }
    }
    static _checkForInconsistentReleaseTags(collector, astSymbol, symbolMetadata) {
        if (astSymbol.isExternal) {
            // For now, don't report errors for external code.  If the developer cares about it, they should run
            // API Extractor separately on the external project
            return;
        }
        // Normally we will expect all release tags to be the same.  Arbitrarily we choose the maxEffectiveReleaseTag
        // as the thing they should all match.
        const expectedEffectiveReleaseTag = symbolMetadata.maxEffectiveReleaseTag;
        // This is set to true if we find a declaration whose release tag is different from expectedEffectiveReleaseTag
        let mixedReleaseTags = false;
        // This is set to false if we find a declaration that is not a function/method overload
        let onlyFunctionOverloads = true;
        // This is set to true if we find a declaration that is @internal
        let anyInternalReleaseTags = false;
        for (const astDeclaration of astSymbol.astDeclarations) {
            const apiItemMetadata = collector.fetchApiItemMetadata(astDeclaration);
            const effectiveReleaseTag = apiItemMetadata.effectiveReleaseTag;
            switch (astDeclaration.declaration.kind) {
                case ts.SyntaxKind.FunctionDeclaration:
                case ts.SyntaxKind.MethodDeclaration:
                    break;
                default:
                    onlyFunctionOverloads = false;
            }
            if (effectiveReleaseTag !== expectedEffectiveReleaseTag) {
                mixedReleaseTags = true;
            }
            if (effectiveReleaseTag === api_extractor_model_1.ReleaseTag.Internal) {
                anyInternalReleaseTags = true;
            }
        }
        if (mixedReleaseTags) {
            if (!onlyFunctionOverloads) {
                collector.messageRouter.addAnalyzerIssue("ae-different-release-tags" /* DifferentReleaseTags */, 'This symbol has another declaration with a different release tag', astSymbol);
            }
            if (anyInternalReleaseTags) {
                collector.messageRouter.addAnalyzerIssue("ae-internal-mixed-release-tag" /* InternalMixedReleaseTag */, `Mixed release tags are not allowed for "${astSymbol.localName}" because one of its declarations` +
                    ` is marked as @internal`, astSymbol);
            }
        }
    }
    static _checkReferences(collector, astDeclaration, alreadyWarnedSymbols) {
        const apiItemMetadata = collector.fetchApiItemMetadata(astDeclaration);
        const declarationReleaseTag = apiItemMetadata.effectiveReleaseTag;
        for (const referencedEntity of astDeclaration.referencedAstEntities) {
            if (referencedEntity instanceof AstSymbol_1.AstSymbol) {
                // If this is e.g. a member of a namespace, then we need to be checking the top-level scope to see
                // whether it's exported.
                //
                // TODO: Technically we should also check each of the nested scopes along the way.
                const rootSymbol = referencedEntity.rootAstSymbol;
                if (!rootSymbol.isExternal) {
                    const collectorEntity = collector.tryGetCollectorEntity(rootSymbol);
                    if (collectorEntity && collectorEntity.exported) {
                        const referencedMetadata = collector.fetchSymbolMetadata(referencedEntity);
                        const referencedReleaseTag = referencedMetadata.maxEffectiveReleaseTag;
                        if (api_extractor_model_1.ReleaseTag.compare(declarationReleaseTag, referencedReleaseTag) > 0) {
                            collector.messageRouter.addAnalyzerIssue("ae-incompatible-release-tags" /* IncompatibleReleaseTags */, `The symbol "${astDeclaration.astSymbol.localName}"`
                                + ` is marked as ${api_extractor_model_1.ReleaseTag.getTagName(declarationReleaseTag)},`
                                + ` but its signature references "${referencedEntity.localName}"`
                                + ` which is marked as ${api_extractor_model_1.ReleaseTag.getTagName(referencedReleaseTag)}`, astDeclaration);
                        }
                    }
                    else {
                        const entryPointFilename = path.basename(collector.workingPackage.entryPointSourceFile.fileName);
                        if (!alreadyWarnedSymbols.has(referencedEntity)) {
                            alreadyWarnedSymbols.add(referencedEntity);
                            // The main usage scenario for ECMAScript symbols is to attach private data to a JavaScript object,
                            // so as a special case, we do NOT report them as forgotten exports.
                            if (!ValidationEnhancer._isEcmaScriptSymbol(referencedEntity)) {
                                collector.messageRouter.addAnalyzerIssue("ae-forgotten-export" /* ForgottenExport */, `The symbol "${rootSymbol.localName}" needs to be exported`
                                    + ` by the entry point ${entryPointFilename}`, astDeclaration);
                            }
                        }
                    }
                }
            }
        }
    }
    // Detect an AstSymbol that refers to an ECMAScript symbol declaration such as:
    //
    // const mySymbol: unique symbol = Symbol('mySymbol');
    static _isEcmaScriptSymbol(astSymbol) {
        if (astSymbol.astDeclarations.length !== 1) {
            return false;
        }
        // We are matching a form like this:
        //
        // - VariableDeclaration:
        //   - Identifier:  pre=[mySymbol]
        //   - ColonToken:  pre=[:] sep=[ ]
        //   - TypeOperator:
        //     - UniqueKeyword:  pre=[unique] sep=[ ]
        //     - SymbolKeyword:  pre=[symbol]
        const astDeclaration = astSymbol.astDeclarations[0];
        if (ts.isVariableDeclaration(astDeclaration.declaration)) {
            const variableTypeNode = astDeclaration.declaration.type;
            if (variableTypeNode) {
                for (const token of variableTypeNode.getChildren()) {
                    if (token.kind === ts.SyntaxKind.SymbolKeyword) {
                        return true;
                    }
                }
            }
        }
        return false;
    }
}
exports.ValidationEnhancer = ValidationEnhancer;
//# sourceMappingURL=ValidationEnhancer.js.map