import { TSDocTagDefinition } from './TSDocTagDefinition';
import { TSDocValidationConfiguration } from './TSDocValidationConfiguration';
import { DocNodeManager } from './DocNodeManager';
/**
 * Configuration for the TSDocParser.
 */
export declare class TSDocConfiguration {
    private readonly _tagDefinitions;
    private readonly _tagDefinitionsByName;
    private readonly _supportedTagDefinitions;
    private readonly _validation;
    private readonly _docNodeManager;
    constructor();
    /**
     * The TSDoc block tag names that will be interpreted as modifier tags.
     */
    readonly tagDefinitions: ReadonlyArray<TSDocTagDefinition>;
    /**
     * Returns the subset of {@link TSDocConfiguration.tagDefinitions}
     * that are supported in this configuration.
     *
     * @remarks
     * This property is only used when
     * {@link TSDocValidationConfiguration.reportUnsupportedTags} is enabled.
     */
    readonly supportedTagDefinitions: ReadonlyArray<TSDocTagDefinition>;
    /**
     * Enable/disable validation checks performed by the parser.
     */
    readonly validation: TSDocValidationConfiguration;
    /**
     * Register custom DocNode subclasses.
     */
    readonly docNodeManager: DocNodeManager;
    /**
     * Return the tag that was defined with the specified name, or undefined
     * if not found.
     */
    tryGetTagDefinition(tagName: string): TSDocTagDefinition | undefined;
    /**
     * Return the tag that was defined with the specified name, or undefined
     * if not found.
     */
    tryGetTagDefinitionWithUpperCase(alreadyUpperCaseTagName: string): TSDocTagDefinition | undefined;
    /**
     * Define a new TSDoc tag to be recognized by the TSDocParser, and mark it as unsupported.
     * Use {@link TSDocConfiguration.setSupportForTag} to mark it as supported.
     *
     * @remarks
     * If a tag is "defined" this means that the parser recognizes it and understands its syntax.
     * Whereas if a tag is "supported", this means it is defined AND the application implements the tag.
     */
    addTagDefinition(tagDefinition: TSDocTagDefinition): void;
    /**
     * Calls {@link TSDocConfiguration.addTagDefinition} for a list of definitions,
     * and optionally marks them as supported.
     * @param tagDefinitions - the definitions to be added
     * @param supported - if specified, calls the {@link TSDocConfiguration.setSupportForTag}
     *    method to mark the definitions as supported or unsupported
     */
    addTagDefinitions(tagDefinitions: ReadonlyArray<TSDocTagDefinition>, supported?: boolean | undefined): void;
    /**
     * Returns true if the tag is supported in this configuration.
     */
    isTagSupported(tagDefinition: TSDocTagDefinition): boolean;
    /**
     * Specifies whether the tag definition is supported in this configuration.
     * The parser may issue warnings for unsupported tags.
     *
     * @remarks
     * If a tag is "defined" this means that the parser recognizes it and understands its syntax.
     * Whereas if a tag is "supported", this means it is defined AND the application implements the tag.
     *
     * This function automatically sets {@link TSDocValidationConfiguration.reportUnsupportedTags}
     * to true.
     */
    setSupportForTag(tagDefinition: TSDocTagDefinition, supported: boolean): void;
    /**
     * Calls {@link TSDocConfiguration.setSupportForTag} for multiple tag definitions.
     */
    setSupportForTags(tagDefinitions: ReadonlyArray<TSDocTagDefinition>, supported: boolean): void;
    private _requireTagToBeDefined;
}
