import { $ctx, $view } from '@milkdown/utils';
import { imageSchema } from '@milkdown/preset-commonmark';
import DOMPurify from 'dompurify';
import { h, defineComponent, ref, createApp, watchEffect } from 'vue';
import clsx from 'clsx';
import { customAlphabet } from 'nanoid';

var __defProp = Object.defineProperty;
var __getOwnPropSymbols = Object.getOwnPropertySymbols;
var __hasOwnProp = Object.prototype.hasOwnProperty;
var __propIsEnum = Object.prototype.propertyIsEnumerable;
var __defNormalProp = (obj, key, value) => key in obj ? __defProp(obj, key, { enumerable: true, configurable: true, writable: true, value }) : obj[key] = value;
var __spreadValues = (a, b) => {
  for (var prop in b || (b = {}))
    if (__hasOwnProp.call(b, prop))
      __defNormalProp(a, prop, b[prop]);
  if (__getOwnPropSymbols)
    for (var prop of __getOwnPropSymbols(b)) {
      if (__propIsEnum.call(b, prop))
        __defNormalProp(a, prop, b[prop]);
    }
  return a;
};
function withMeta(plugin, meta) {
  Object.assign(plugin, {
    meta: __spreadValues({
      package: "@milkdown/components"
    }, meta)
  });
  return plugin;
}

const defaultInlineImageConfig = {
  imageIcon: "\u{1F30C}",
  uploadButton: "Upload",
  confirmButton: "\u23CE",
  uploadPlaceholderText: "/Paste",
  onUpload: (file) => Promise.resolve(URL.createObjectURL(file))
};
const inlineImageConfig = $ctx(
  defaultInlineImageConfig,
  "inlineImageConfigCtx"
);
withMeta(inlineImageConfig, {
  displayName: "Config<image-inline>",
  group: "ImageInline"
});

function Icon({ icon, class: className, onClick }) {
  return /* @__PURE__ */ h(
    "span",
    {
      class: clsx("milkdown-icon", className),
      onPointerdown: onClick,
      ref: (el) => {
        if (el && icon) {
          el.innerHTML = DOMPurify.sanitize(icon.trim());
        }
      }
    }
  );
}
Icon.props = {
  icon: {
    type: String,
    required: false
  },
  class: {
    type: String,
    required: false
  },
  onClick: {
    type: Function,
    required: false
  }
};

const nanoid = customAlphabet("abcdefg", 8);
const ImageInput = defineComponent({
  props: {
    src: {
      type: Object,
      required: true
    },
    selected: {
      type: Object,
      required: true
    },
    readonly: {
      type: Object,
      required: true
    },
    setLink: {
      type: Function,
      required: true
    },
    imageIcon: {
      type: String,
      required: false
    },
    uploadButton: {
      type: String,
      required: false
    },
    confirmButton: {
      type: String,
      required: false
    },
    uploadPlaceholderText: {
      type: String,
      required: false
    },
    onUpload: {
      type: Function,
      required: true
    }
  },
  setup({
    readonly,
    src,
    setLink,
    onUpload,
    imageIcon,
    uploadButton,
    confirmButton,
    uploadPlaceholderText,
    className
  }) {
    var _a, _b;
    const focusLinkInput = ref(false);
    const linkInputRef = ref();
    const currentLink = ref((_a = src.value) != null ? _a : "");
    const uuid = ref(nanoid());
    const hidePlaceholder = ref(((_b = src.value) == null ? void 0 : _b.length) !== 0);
    const onEditLink = (e) => {
      const target = e.target;
      const value = target.value;
      hidePlaceholder.value = value.length !== 0;
      currentLink.value = value;
    };
    const onKeydown = (e) => {
      var _a2, _b2;
      if (e.key === "Enter") {
        setLink((_b2 = (_a2 = linkInputRef.value) == null ? void 0 : _a2.value) != null ? _b2 : "");
      }
    };
    const onConfirmLinkInput = () => {
      var _a2, _b2;
      setLink((_b2 = (_a2 = linkInputRef.value) == null ? void 0 : _a2.value) != null ? _b2 : "");
    };
    const onUploadFile = (e) => {
      var _a2;
      const file = (_a2 = e.target.files) == null ? void 0 : _a2[0];
      if (!file) return;
      onUpload(file).then((url) => {
        if (!url) return;
        setLink(url);
        hidePlaceholder.value = true;
      }).catch((err) => {
        console.error("An error occurred while uploading image");
        console.error(err);
      });
    };
    return () => {
      return /* @__PURE__ */ h("div", { class: clsx("image-edit", className) }, /* @__PURE__ */ h(Icon, { icon: imageIcon, class: "image-icon" }), /* @__PURE__ */ h("div", { class: clsx("link-importer", focusLinkInput.value && "focus") }, /* @__PURE__ */ h(
        "input",
        {
          ref: linkInputRef,
          draggable: "true",
          onDragstart: (e) => {
            e.preventDefault();
            e.stopPropagation();
          },
          disabled: readonly.value,
          class: "link-input-area",
          value: currentLink.value,
          onInput: onEditLink,
          onKeydown,
          onFocus: () => focusLinkInput.value = true,
          onBlur: () => focusLinkInput.value = false
        }
      ), !hidePlaceholder.value && /* @__PURE__ */ h("div", { class: "placeholder" }, /* @__PURE__ */ h(
        "input",
        {
          disabled: readonly.value,
          class: "hidden",
          id: uuid.value,
          type: "file",
          accept: "image/*",
          onChange: onUploadFile
        }
      ), /* @__PURE__ */ h("label", { class: "uploader", for: uuid.value }, /* @__PURE__ */ h(Icon, { icon: uploadButton })), /* @__PURE__ */ h("span", { class: "text", onClick: () => {
        var _a2;
        return (_a2 = linkInputRef.value) == null ? void 0 : _a2.focus();
      } }, uploadPlaceholderText))), currentLink.value && /* @__PURE__ */ h("div", { class: "confirm", onClick: () => onConfirmLinkInput() }, /* @__PURE__ */ h(Icon, { icon: confirmButton })));
    };
  }
});

const MilkdownImageInline = defineComponent({
  props: {
    src: {
      type: Object,
      required: true
    },
    alt: {
      type: Object,
      required: true
    },
    title: {
      type: Object,
      required: true
    },
    selected: {
      type: Object,
      required: true
    },
    readonly: {
      type: Object,
      required: true
    },
    setAttr: {
      type: Function,
      required: true
    },
    config: {
      type: Object,
      required: true
    }
  },
  setup(props) {
    const { src, alt, title } = props;
    return () => {
      var _a;
      if (!((_a = src.value) == null ? void 0 : _a.length)) {
        return /* @__PURE__ */ h(
          ImageInput,
          {
            src: props.src,
            selected: props.selected,
            readonly: props.readonly,
            setLink: (link) => props.setAttr("src", link),
            imageIcon: props.config.imageIcon,
            uploadButton: props.config.uploadButton,
            confirmButton: props.config.confirmButton,
            uploadPlaceholderText: props.config.uploadPlaceholderText,
            onUpload: props.config.onUpload,
            className: "empty-image-inline"
          }
        );
      }
      return /* @__PURE__ */ h(
        "img",
        {
          class: "image-inline",
          src: src.value,
          alt: alt.value,
          title: title.value
        }
      );
    };
  }
});

const inlineImageView = $view(
  imageSchema.node,
  (ctx) => {
    return (initialNode, view, getPos) => {
      const src = ref(initialNode.attrs.src);
      const alt = ref(initialNode.attrs.alt);
      const title = ref(initialNode.attrs.title);
      const selected = ref(false);
      const readonly = ref(!view.editable);
      const setAttr = (attr, value) => {
        if (!view.editable) return;
        const pos = getPos();
        if (pos == null) return;
        view.dispatch(
          view.state.tr.setNodeAttribute(
            pos,
            attr,
            attr === "src" ? DOMPurify.sanitize(value) : value
          )
        );
      };
      const config = ctx.get(inlineImageConfig.key);
      const app = createApp(MilkdownImageInline, {
        src,
        alt,
        title,
        selected,
        readonly,
        setAttr,
        config
      });
      const dom = document.createElement("span");
      dom.className = "milkdown-image-inline";
      app.mount(dom);
      const disposeSelectedWatcher = watchEffect(() => {
        const isSelected = selected.value;
        if (isSelected) {
          dom.classList.add("selected");
        } else {
          dom.classList.remove("selected");
        }
      });
      const proxyDomURL = config.proxyDomURL;
      const bindAttrs = (node) => {
        if (!proxyDomURL) {
          src.value = node.attrs.src;
        } else {
          const proxiedURL = proxyDomURL(node.attrs.src);
          if (typeof proxiedURL === "string") {
            src.value = proxiedURL;
          } else {
            proxiedURL.then((url) => {
              src.value = url;
            }).catch(console.error);
          }
        }
        alt.value = node.attrs.alt;
        title.value = node.attrs.title;
      };
      bindAttrs(initialNode);
      return {
        dom,
        update: (updatedNode) => {
          if (updatedNode.type !== initialNode.type) return false;
          bindAttrs(updatedNode);
          return true;
        },
        stopEvent: (e) => {
          if (e.target instanceof HTMLInputElement) return true;
          return false;
        },
        selectNode: () => {
          selected.value = true;
        },
        deselectNode: () => {
          selected.value = false;
        },
        destroy: () => {
          disposeSelectedWatcher();
          app.unmount();
          dom.remove();
        }
      };
    };
  }
);
withMeta(inlineImageView, {
  displayName: "NodeView<image-inline>",
  group: "ImageInline"
});

const imageInlineComponent = [
  inlineImageConfig,
  inlineImageView
];

export { defaultInlineImageConfig, imageInlineComponent, inlineImageConfig, inlineImageView };
//# sourceMappingURL=index.js.map
