import { $ctx, $view } from '@milkdown/utils';
import { codeBlockSchema } from '@milkdown/preset-commonmark';
import { Compartment, EditorState } from '@codemirror/state';
import { EditorView, drawSelection, keymap } from '@codemirror/view';
import { exitCode } from '@milkdown/prose/commands';
import { undo, redo } from '@milkdown/prose/history';
import { TextSelection } from '@milkdown/prose/state';
import { h, defineComponent, Fragment, ref, watch, computed, onMounted, onUnmounted, watchEffect, createApp } from 'vue';
import clsx from 'clsx';
import DOMPurify from 'dompurify';
import { computePosition } from '@floating-ui/dom';

var __defProp = Object.defineProperty;
var __getOwnPropSymbols = Object.getOwnPropertySymbols;
var __hasOwnProp = Object.prototype.hasOwnProperty;
var __propIsEnum = Object.prototype.propertyIsEnumerable;
var __defNormalProp = (obj, key, value) => key in obj ? __defProp(obj, key, { enumerable: true, configurable: true, writable: true, value }) : obj[key] = value;
var __spreadValues = (a, b) => {
  for (var prop in b || (b = {}))
    if (__hasOwnProp.call(b, prop))
      __defNormalProp(a, prop, b[prop]);
  if (__getOwnPropSymbols)
    for (var prop of __getOwnPropSymbols(b)) {
      if (__propIsEnum.call(b, prop))
        __defNormalProp(a, prop, b[prop]);
    }
  return a;
};
function withMeta(plugin, meta) {
  Object.assign(plugin, {
    meta: __spreadValues({
      package: "@milkdown/components"
    }, meta)
  });
  return plugin;
}

const defaultConfig = {
  extensions: [],
  languages: [],
  expandIcon: "\u2B07",
  searchIcon: "\u{1F50D}",
  clearSearchIcon: "\u232B",
  searchPlaceholder: "Search language",
  noResultText: "No result",
  copyText: "Copy",
  copyIcon: "\u{1F4CB}",
  onCopy: () => {
  },
  renderLanguage: (language) => language,
  renderPreview: () => null,
  previewToggleButton: (previewOnlyMode) => previewOnlyMode ? "Edit" : "Hide",
  previewLabel: "Preview"
};
const codeBlockConfig = $ctx(defaultConfig, "codeBlockConfigCtx");
withMeta(codeBlockConfig, {
  displayName: "Config<code-block>",
  group: "CodeBlock"
});

class LanguageLoader {
  constructor(languages) {
    this.languages = languages;
    this.map = {};
    languages.forEach((language) => {
      language.alias.forEach((alias) => {
        this.map[alias] = language;
      });
    });
  }
  getAll() {
    return this.languages.map((language) => {
      return {
        name: language.name,
        alias: language.alias
      };
    });
  }
  load(languageName) {
    const languageMap = this.map;
    const language = languageMap[languageName.toLowerCase()];
    if (!language) return Promise.resolve(void 0);
    if (language.support) return Promise.resolve(language.support);
    return language.load();
  }
}

function Icon({ icon, class: className, onClick }) {
  return /* @__PURE__ */ h(
    "span",
    {
      class: clsx("milkdown-icon", className),
      onPointerdown: onClick,
      ref: (el) => {
        if (el && icon) {
          el.innerHTML = DOMPurify.sanitize(icon.trim());
        }
      }
    }
  );
}
Icon.props = {
  icon: {
    type: String,
    required: false
  },
  class: {
    type: String,
    required: false
  },
  onClick: {
    type: Function,
    required: false
  }
};

var __async = (__this, __arguments, generator) => {
  return new Promise((resolve, reject) => {
    var fulfilled = (value) => {
      try {
        step(generator.next(value));
      } catch (e) {
        reject(e);
      }
    };
    var rejected = (value) => {
      try {
        step(generator.throw(value));
      } catch (e) {
        reject(e);
      }
    };
    var step = (x) => x.done ? resolve(x.value) : Promise.resolve(x.value).then(fulfilled, rejected);
    step((generator = generator.apply(__this, __arguments)).next());
  });
};
function copyToClipboard(text) {
  return __async(this, null, function* () {
    try {
      return navigator.clipboard.writeText(text);
    } catch (e) {
      const element = document.createElement("textarea");
      const previouslyFocusedElement = document.activeElement;
      element.value = text;
      element.setAttribute("readonly", "");
      element.style.contain = "strict";
      element.style.position = "absolute";
      element.style.left = "-9999px";
      element.style.fontSize = "12pt";
      const selection = document.getSelection();
      const originalRange = selection ? selection.rangeCount > 0 && selection.getRangeAt(0) : null;
      document.body.appendChild(element);
      element.select();
      element.selectionStart = 0;
      element.selectionEnd = text.length;
      document.execCommand("copy");
      document.body.removeChild(element);
      if (originalRange) {
        selection.removeAllRanges();
        selection.addRange(originalRange);
      }
      if (previouslyFocusedElement) {
        previouslyFocusedElement.focus();
      }
    }
  });
}
const CopyButton = defineComponent({
  props: {
    copyText: {
      type: String,
      required: true
    },
    copyIcon: {
      type: String,
      required: true
    },
    onCopy: {
      type: Function,
      required: true
    },
    text: {
      type: String,
      required: true
    }
  },
  setup(props) {
    const onCopyCode = () => {
      copyToClipboard(props.text).then(() => props.onCopy(props.text)).catch(console.error);
    };
    return () => {
      return /* @__PURE__ */ h(Fragment, null, /* @__PURE__ */ h("button", { type: "button", class: "copy-button", onClick: onCopyCode }, /* @__PURE__ */ h(Icon, { icon: props.copyIcon }), props.copyText));
    };
  }
});

const LanguagePicker = defineComponent({
  props: {
    language: {
      type: Object,
      required: true
    },
    getReadOnly: {
      type: Function,
      required: true
    },
    config: {
      type: Object,
      required: true
    },
    getAllLanguages: {
      type: Function,
      required: true
    },
    setLanguage: {
      type: Function,
      required: true
    }
  },
  setup({ language, config, setLanguage, getAllLanguages, getReadOnly }) {
    const triggerRef = ref();
    const showPicker = ref(false);
    const searchRef = ref();
    const pickerRef = ref();
    const filter = ref("");
    watch([showPicker, triggerRef, pickerRef], () => {
      filter.value = "";
      const picker = triggerRef.value;
      const languageList = pickerRef.value;
      if (!picker || !languageList) return;
      computePosition(picker, languageList, {
        placement: "bottom-start"
      }).then(({ x, y }) => {
        Object.assign(languageList.style, {
          left: `${x}px`,
          top: `${y}px`
        });
      }).catch(console.error);
    });
    const onTogglePicker = (e) => {
      e.preventDefault();
      e.stopPropagation();
      if (getReadOnly()) return;
      const next = !showPicker.value;
      showPicker.value = next;
      if (next) {
        setTimeout(() => {
          var _a;
          return (_a = searchRef.value) == null ? void 0 : _a.focus();
        }, 0);
      }
    };
    const changeFilter = (e) => {
      const target = e.target;
      filter.value = target.value;
    };
    const onSearchKeydown = (e) => {
      if (e.key === "Escape") filter.value = "";
    };
    const languages = computed(() => {
      var _a;
      if (!showPicker.value) return [];
      const all = (_a = getAllLanguages()) != null ? _a : [];
      const selected = all.find(
        (languageInfo) => languageInfo.name.toLowerCase() === language.value.toLowerCase()
      );
      const filtered = all.filter((languageInfo) => {
        const currentValue = filter.value.toLowerCase();
        return (languageInfo.name.toLowerCase().includes(currentValue) || languageInfo.alias.some(
          (alias) => alias.toLowerCase().includes(currentValue)
        )) && languageInfo !== selected;
      });
      if (filtered.length === 0) return [];
      if (!selected) return filtered;
      return [selected, ...filtered];
    });
    const clickHandler = (e) => {
      const target = e.target;
      if (triggerRef.value && triggerRef.value.contains(target)) return;
      const picker = pickerRef.value;
      const trigger = triggerRef.value;
      if (!trigger || !picker) return;
      if (trigger.dataset.expanded !== "true") return;
      if (!picker.contains(target)) showPicker.value = false;
    };
    onMounted(() => {
      window.addEventListener("click", clickHandler);
    });
    onUnmounted(() => {
      window.removeEventListener("click", clickHandler);
    });
    return () => {
      return /* @__PURE__ */ h(Fragment, null, /* @__PURE__ */ h(
        "button",
        {
          type: "button",
          ref: triggerRef,
          class: "language-button",
          onClick: onTogglePicker,
          "data-expanded": String(showPicker.value)
        },
        language.value || "Text",
        /* @__PURE__ */ h("div", { class: "expand-icon" }, /* @__PURE__ */ h(Icon, { icon: config.expandIcon }))
      ), /* @__PURE__ */ h("div", { ref: pickerRef, class: "language-picker" }, showPicker.value ? /* @__PURE__ */ h("div", { class: "list-wrapper" }, /* @__PURE__ */ h("div", { class: "search-box" }, /* @__PURE__ */ h("div", { class: "search-icon" }, /* @__PURE__ */ h(Icon, { icon: config.searchIcon })), /* @__PURE__ */ h(
        "input",
        {
          ref: searchRef,
          class: "search-input",
          placeholder: config.searchPlaceholder,
          value: filter.value,
          onInput: changeFilter,
          onKeydown: onSearchKeydown
        }
      ), /* @__PURE__ */ h(
        "div",
        {
          class: clsx(
            "clear-icon",
            filter.value.length === 0 && "hidden"
          ),
          onMousedown: (e) => {
            e.preventDefault();
            filter.value = "";
          }
        },
        /* @__PURE__ */ h(Icon, { icon: config.clearSearchIcon })
      )), /* @__PURE__ */ h(
        "ul",
        {
          class: "language-list",
          role: "listbox",
          onKeydown: (e) => {
            if (e.key === "Enter") {
              const active = document.activeElement;
              if (active instanceof HTMLElement && active.dataset.language)
                setLanguage(active.dataset.language);
            }
          }
        },
        !languages.value.length ? /* @__PURE__ */ h("li", { class: "language-list-item no-result" }, config.noResultText) : languages.value.map((languageInfo) => /* @__PURE__ */ h(
          "li",
          {
            role: "listitem",
            tabindex: "0",
            class: "language-list-item",
            "aria-selected": languageInfo.name.toLowerCase() === language.value.toLowerCase(),
            "data-language": languageInfo.name,
            onClick: () => {
              setLanguage(languageInfo.name);
              showPicker.value = false;
            }
          },
          config.renderLanguage(
            languageInfo.name,
            languageInfo.name.toLowerCase() === language.value.toLowerCase()
          )
        ))
      )) : null));
    };
  }
});

const PreviewPanel = defineComponent({
  props: {
    text: {
      type: Object,
      required: true
    },
    language: {
      type: Object,
      required: true
    },
    config: {
      type: Object,
      required: true
    },
    previewOnlyMode: {
      type: Object,
      required: true
    },
    preview: {
      type: Object,
      required: true
    }
  },
  setup(props) {
    const { previewOnlyMode, config, preview } = props;
    const previewRef = ref();
    watchEffect(() => {
      const previewContainer = previewRef.value;
      if (!previewContainer) return;
      while (previewContainer.firstChild) {
        previewContainer.removeChild(previewContainer.firstChild);
      }
      const previewContent = preview.value;
      if (typeof previewContent === "string" || previewContent instanceof Element) {
        previewContainer.innerHTML = DOMPurify.sanitize(previewContent);
      }
    });
    return () => {
      if (!preview.value) return null;
      return /* @__PURE__ */ h("div", { class: "preview-panel" }, !previewOnlyMode.value && /* @__PURE__ */ h(Fragment, null, /* @__PURE__ */ h("div", { class: "preview-divider" }), /* @__PURE__ */ h("div", { class: clsx("preview-label", previewOnlyMode && "hidden") }, config.previewLabel)), /* @__PURE__ */ h("div", { ref: previewRef, class: "preview" }));
    };
  }
});

const CodeBlock = defineComponent({
  props: {
    text: {
      type: Object,
      required: true
    },
    selected: {
      type: Object,
      required: true
    },
    getReadOnly: {
      type: Function,
      required: true
    },
    codemirror: {
      type: Object,
      required: true
    },
    language: {
      type: Object,
      required: true
    },
    getAllLanguages: {
      type: Function,
      required: true
    },
    setLanguage: {
      type: Function,
      required: true
    },
    config: {
      type: Object,
      required: true
    }
  },
  setup(props) {
    const previewOnlyMode = ref(false);
    const codemirrorHostRef = ref();
    onMounted(() => {
      var _a;
      while ((_a = codemirrorHostRef.value) == null ? void 0 : _a.firstChild) {
        codemirrorHostRef.value.removeChild(codemirrorHostRef.value.firstChild);
      }
      if (codemirrorHostRef.value) {
        codemirrorHostRef.value.appendChild(props.codemirror.dom);
      }
    });
    const preview = computed(() => {
      const text = props.text.value;
      const language = props.language.value;
      return props.config.renderPreview(language, text);
    });
    const empty = () => {
    };
    return () => {
      var _a;
      return /* @__PURE__ */ h(Fragment, null, /* @__PURE__ */ h("div", { class: "tools" }, /* @__PURE__ */ h(
        LanguagePicker,
        {
          language: props.language,
          config: props.config,
          setLanguage: props.setLanguage,
          getAllLanguages: props.getAllLanguages,
          getReadOnly: props.getReadOnly
        }
      ), /* @__PURE__ */ h("div", { class: "tools-button-group" }, /* @__PURE__ */ h(
        CopyButton,
        {
          copyIcon: props.config.copyIcon,
          copyText: props.config.copyText,
          onCopy: (_a = props.config.onCopy) != null ? _a : empty,
          text: props.text.value
        }
      ), preview.value ? /* @__PURE__ */ h(
        "button",
        {
          class: "preview-toggle-button",
          onClick: () => previewOnlyMode.value = !previewOnlyMode.value
        },
        /* @__PURE__ */ h(
          Icon,
          {
            icon: props.config.previewToggleButton(
              previewOnlyMode.value
            )
          }
        )
      ) : null)), /* @__PURE__ */ h(
        "div",
        {
          ref: codemirrorHostRef,
          class: clsx(
            "codemirror-host",
            preview.value && previewOnlyMode.value && "hidden"
          )
        }
      ), /* @__PURE__ */ h(
        PreviewPanel,
        {
          text: props.text,
          language: props.language,
          config: props.config,
          previewOnlyMode,
          preview
        }
      ));
    };
  }
});

class CodeMirrorBlock {
  constructor(node, view, getPos, loader, config) {
    this.node = node;
    this.view = view;
    this.getPos = getPos;
    this.loader = loader;
    this.config = config;
    this.selected = ref(false);
    this.language = ref("");
    this.text = ref("");
    this.updating = false;
    this.languageName = "";
    this.forwardUpdate = (update) => {
      var _a;
      if (this.updating || !this.cm.hasFocus) return;
      let offset = ((_a = this.getPos()) != null ? _a : 0) + 1;
      const { main } = update.state.selection;
      const selFrom = offset + main.from;
      const selTo = offset + main.to;
      const pmSel = this.view.state.selection;
      if (update.docChanged || pmSel.from !== selFrom || pmSel.to !== selTo) {
        const tr = this.view.state.tr;
        update.changes.iterChanges((fromA, toA, fromB, toB, text) => {
          if (text.length)
            tr.replaceWith(
              offset + fromA,
              offset + toA,
              this.view.state.schema.text(text.toString())
            );
          else tr.delete(offset + fromA, offset + toA);
          offset += toB - fromB - (toA - fromA);
        });
        tr.setSelection(TextSelection.create(tr.doc, selFrom, selTo));
        this.view.dispatch(tr);
      }
    };
    this.createApp = () => {
      return createApp(CodeBlock, {
        text: this.text,
        selected: this.selected,
        codemirror: this.cm,
        language: this.language,
        getAllLanguages: this.getAllLanguages,
        getReadOnly: () => !this.view.editable,
        setLanguage: this.setLanguage,
        config: this.config
      });
    };
    this.codeMirrorKeymap = () => {
      const view = this.view;
      return [
        { key: "ArrowUp", run: () => this.maybeEscape("line", -1) },
        { key: "ArrowLeft", run: () => this.maybeEscape("char", -1) },
        { key: "ArrowDown", run: () => this.maybeEscape("line", 1) },
        { key: "ArrowRight", run: () => this.maybeEscape("char", 1) },
        {
          key: "Mod-Enter",
          run: () => {
            if (!exitCode(view.state, view.dispatch)) return false;
            view.focus();
            return true;
          }
        },
        { key: "Mod-z", run: () => undo(view.state, view.dispatch) },
        { key: "Shift-Mod-z", run: () => redo(view.state, view.dispatch) },
        { key: "Mod-y", run: () => redo(view.state, view.dispatch) },
        {
          key: "Backspace",
          run: () => {
            var _a;
            const ranges = this.cm.state.selection.ranges;
            if (ranges.length > 1) return false;
            const selection = ranges[0];
            if (selection && (!selection.empty || selection.anchor > 0))
              return false;
            if (this.cm.state.doc.lines >= 2) return false;
            const state = this.view.state;
            const pos = (_a = this.getPos()) != null ? _a : 0;
            const tr = state.tr.replaceWith(
              pos,
              pos + this.node.nodeSize,
              state.schema.nodes.paragraph.createChecked({}, this.node.content)
            );
            tr.setSelection(TextSelection.near(tr.doc.resolve(pos)));
            this.view.dispatch(tr);
            this.view.focus();
            return true;
          }
        }
      ];
    };
    this.maybeEscape = (unit, dir) => {
      var _a;
      const { state } = this.cm;
      let main = state.selection.main;
      if (!main.empty) return false;
      if (unit === "line") main = state.doc.lineAt(main.head);
      if (dir < 0 ? main.from > 0 : main.to < state.doc.length) return false;
      const targetPos = ((_a = this.getPos()) != null ? _a : 0) + (dir < 0 ? 0 : this.node.nodeSize);
      const selection = TextSelection.near(
        this.view.state.doc.resolve(targetPos),
        dir
      );
      const tr = this.view.state.tr.setSelection(selection).scrollIntoView();
      this.view.dispatch(tr);
      this.view.focus();
      return true;
    };
    this.setLanguage = (language) => {
      var _a;
      this.view.dispatch(
        this.view.state.tr.setNodeAttribute(
          (_a = this.getPos()) != null ? _a : 0,
          "language",
          language
        )
      );
    };
    this.getAllLanguages = () => {
      return this.loader.getAll();
    };
    this.languageConf = new Compartment();
    this.readOnlyConf = new Compartment();
    this.cm = new EditorView({
      doc: this.node.textContent,
      root: this.view.root,
      extensions: [
        this.readOnlyConf.of(EditorState.readOnly.of(!this.view.editable)),
        drawSelection(),
        keymap.of(this.codeMirrorKeymap()),
        this.languageConf.of([]),
        EditorState.changeFilter.of(() => this.view.editable),
        ...config.extensions,
        EditorView.updateListener.of(this.forwardUpdate)
      ]
    });
    this.app = this.createApp();
    this.dom = this.createDom(this.app);
    this.disposeSelectedWatcher = watchEffect(() => {
      const isSelected = this.selected.value;
      if (isSelected) {
        this.dom.classList.add("selected");
      } else {
        this.dom.classList.remove("selected");
      }
    });
    this.updateLanguage();
  }
  createDom(app) {
    const dom = document.createElement("div");
    dom.className = "milkdown-code-block";
    this.text.value = this.node.textContent;
    app.mount(dom);
    return dom;
  }
  updateLanguage() {
    const languageName = this.node.attrs.language;
    if (languageName === this.languageName) return;
    this.language.value = languageName;
    const language = this.loader.load(languageName != null ? languageName : "");
    language.then((lang) => {
      if (lang) {
        this.cm.dispatch({
          effects: this.languageConf.reconfigure(lang)
        });
        this.languageName = languageName;
      }
    }).catch(console.error);
  }
  setSelection(anchor, head) {
    if (!this.cm.dom.isConnected) return;
    this.cm.focus();
    this.updating = true;
    this.cm.dispatch({ selection: { anchor, head } });
    this.updating = false;
  }
  update(node) {
    if (node.type !== this.node.type) return false;
    if (this.updating) return true;
    this.node = node;
    this.text.value = node.textContent;
    this.updateLanguage();
    if (this.view.editable === this.cm.state.readOnly) {
      this.cm.dispatch({
        effects: this.readOnlyConf.reconfigure(
          EditorState.readOnly.of(!this.view.editable)
        )
      });
    }
    const change = computeChange(this.cm.state.doc.toString(), node.textContent);
    if (change) {
      this.updating = true;
      this.cm.dispatch({
        changes: { from: change.from, to: change.to, insert: change.text },
        scrollIntoView: true
      });
      this.updating = false;
    }
    return true;
  }
  selectNode() {
    this.selected.value = true;
    this.cm.focus();
  }
  deselectNode() {
    this.selected.value = false;
  }
  stopEvent() {
    return true;
  }
  destroy() {
    this.app.unmount();
    this.cm.destroy();
    this.disposeSelectedWatcher();
  }
}
function computeChange(oldVal, newVal) {
  if (oldVal === newVal) return null;
  let start = 0;
  let oldEnd = oldVal.length;
  let newEnd = newVal.length;
  while (start < oldEnd && oldVal.charCodeAt(start) === newVal.charCodeAt(start))
    ++start;
  while (oldEnd > start && newEnd > start && oldVal.charCodeAt(oldEnd - 1) === newVal.charCodeAt(newEnd - 1)) {
    oldEnd--;
    newEnd--;
  }
  return { from: start, to: oldEnd, text: newVal.slice(start, newEnd) };
}

const codeBlockView = $view(
  codeBlockSchema.node,
  (ctx) => {
    const config = ctx.get(codeBlockConfig.key);
    const languageLoader = new LanguageLoader(config.languages);
    return (node, view, getPos) => new CodeMirrorBlock(node, view, getPos, languageLoader, config);
  }
);
withMeta(codeBlockView, {
  displayName: "NodeView<code-block>",
  group: "CodeBlock"
});

const codeBlockComponent = [
  codeBlockView,
  codeBlockConfig
];

export { codeBlockComponent, codeBlockConfig, codeBlockView, defaultConfig };
//# sourceMappingURL=index.js.map
