import { $ctx, $remark, $nodeSchema, $view } from '@milkdown/utils';
import { visit } from 'unist-util-visit';
import { expectDomTypeError } from '@milkdown/exception';
import DOMPurify from 'dompurify';
import { h, defineComponent, ref, Fragment, createApp, watchEffect } from 'vue';
import clsx from 'clsx';
import { customAlphabet } from 'nanoid';

var __defProp$2 = Object.defineProperty;
var __getOwnPropSymbols$2 = Object.getOwnPropertySymbols;
var __hasOwnProp$2 = Object.prototype.hasOwnProperty;
var __propIsEnum$2 = Object.prototype.propertyIsEnumerable;
var __defNormalProp$2 = (obj, key, value) => key in obj ? __defProp$2(obj, key, { enumerable: true, configurable: true, writable: true, value }) : obj[key] = value;
var __spreadValues$2 = (a, b) => {
  for (var prop in b || (b = {}))
    if (__hasOwnProp$2.call(b, prop))
      __defNormalProp$2(a, prop, b[prop]);
  if (__getOwnPropSymbols$2)
    for (var prop of __getOwnPropSymbols$2(b)) {
      if (__propIsEnum$2.call(b, prop))
        __defNormalProp$2(a, prop, b[prop]);
    }
  return a;
};
function withMeta(plugin, meta) {
  Object.assign(plugin, {
    meta: __spreadValues$2({
      package: "@milkdown/components"
    }, meta)
  });
  return plugin;
}

const defaultImageBlockConfig = {
  imageIcon: "\u{1F30C}",
  captionIcon: "\u{1F4AC}",
  uploadButton: "Upload file",
  confirmButton: "Confirm \u23CE",
  uploadPlaceholderText: "or paste the image link ...",
  captionPlaceholderText: "Image caption",
  onUpload: (file) => Promise.resolve(URL.createObjectURL(file))
};
const imageBlockConfig = $ctx(
  defaultImageBlockConfig,
  "imageBlockConfigCtx"
);
withMeta(imageBlockConfig, {
  displayName: "Config<image-block>",
  group: "ImageBlock"
});

function visitImage(ast) {
  return visit(
    ast,
    "paragraph",
    (node, index, parent) => {
      var _a, _b;
      if (((_a = node.children) == null ? void 0 : _a.length) !== 1) return;
      const firstChild = (_b = node.children) == null ? void 0 : _b[0];
      if (!firstChild || firstChild.type !== "image") return;
      const { url, alt, title } = firstChild;
      const newNode = {
        type: "image-block",
        url,
        alt,
        title
      };
      parent.children.splice(index, 1, newNode);
    }
  );
}
const remarkImageBlockPlugin = $remark(
  "remark-image-block",
  () => () => visitImage
);
withMeta(remarkImageBlockPlugin.plugin, {
  displayName: "Remark<remarkImageBlock>",
  group: "ImageBlock"
});
withMeta(remarkImageBlockPlugin.options, {
  displayName: "RemarkConfig<remarkImageBlock>",
  group: "ImageBlock"
});

var __defProp$1 = Object.defineProperty;
var __getOwnPropSymbols$1 = Object.getOwnPropertySymbols;
var __hasOwnProp$1 = Object.prototype.hasOwnProperty;
var __propIsEnum$1 = Object.prototype.propertyIsEnumerable;
var __defNormalProp$1 = (obj, key, value) => key in obj ? __defProp$1(obj, key, { enumerable: true, configurable: true, writable: true, value }) : obj[key] = value;
var __spreadValues$1 = (a, b) => {
  for (var prop in b || (b = {}))
    if (__hasOwnProp$1.call(b, prop))
      __defNormalProp$1(a, prop, b[prop]);
  if (__getOwnPropSymbols$1)
    for (var prop of __getOwnPropSymbols$1(b)) {
      if (__propIsEnum$1.call(b, prop))
        __defNormalProp$1(a, prop, b[prop]);
    }
  return a;
};
const IMAGE_DATA_TYPE = "image-block";
const imageBlockSchema = $nodeSchema("image-block", () => {
  return {
    inline: false,
    group: "block",
    selectable: true,
    draggable: true,
    isolating: true,
    marks: "",
    atom: true,
    priority: 100,
    attrs: {
      src: { default: "", validate: "string" },
      caption: { default: "", validate: "string" },
      ratio: { default: 1, validate: "number" }
    },
    parseDOM: [
      {
        tag: `img[data-type="${IMAGE_DATA_TYPE}"]`,
        getAttrs: (dom) => {
          var _a;
          if (!(dom instanceof HTMLElement)) throw expectDomTypeError(dom);
          return {
            src: dom.getAttribute("src") || "",
            caption: dom.getAttribute("caption") || "",
            ratio: Number((_a = dom.getAttribute("ratio")) != null ? _a : 1)
          };
        }
      }
    ],
    toDOM: (node) => ["img", __spreadValues$1({ "data-type": IMAGE_DATA_TYPE }, node.attrs)],
    parseMarkdown: {
      match: ({ type }) => type === "image-block",
      runner: (state, node, type) => {
        const src = node.url;
        const caption = node.title;
        let ratio = Number(node.alt || 1);
        if (Number.isNaN(ratio) || ratio === 0) ratio = 1;
        state.addNode(type, {
          src,
          caption,
          ratio
        });
      }
    },
    toMarkdown: {
      match: (node) => node.type.name === "image-block",
      runner: (state, node) => {
        state.openNode("paragraph");
        state.addNode("image", void 0, void 0, {
          title: node.attrs.caption,
          url: node.attrs.src,
          alt: `${Number.parseFloat(node.attrs.ratio).toFixed(2)}`
        });
        state.closeNode();
      }
    }
  };
});
withMeta(imageBlockSchema.node, {
  displayName: "NodeSchema<image-block>",
  group: "ImageBlock"
});

function Icon({ icon, class: className, onClick }) {
  return /* @__PURE__ */ h(
    "span",
    {
      class: clsx("milkdown-icon", className),
      onPointerdown: onClick,
      ref: (el) => {
        if (el && icon) {
          el.innerHTML = DOMPurify.sanitize(icon.trim());
        }
      }
    }
  );
}
Icon.props = {
  icon: {
    type: String,
    required: false
  },
  class: {
    type: String,
    required: false
  },
  onClick: {
    type: Function,
    required: false
  }
};

const nanoid = customAlphabet("abcdefg", 8);
const ImageInput = defineComponent({
  props: {
    src: {
      type: Object,
      required: true
    },
    selected: {
      type: Object,
      required: true
    },
    readonly: {
      type: Object,
      required: true
    },
    setLink: {
      type: Function,
      required: true
    },
    imageIcon: {
      type: String,
      required: false
    },
    uploadButton: {
      type: String,
      required: false
    },
    confirmButton: {
      type: String,
      required: false
    },
    uploadPlaceholderText: {
      type: String,
      required: false
    },
    onUpload: {
      type: Function,
      required: true
    }
  },
  setup({
    readonly,
    src,
    setLink,
    onUpload,
    imageIcon,
    uploadButton,
    confirmButton,
    uploadPlaceholderText,
    className
  }) {
    var _a, _b;
    const focusLinkInput = ref(false);
    const linkInputRef = ref();
    const currentLink = ref((_a = src.value) != null ? _a : "");
    const uuid = ref(nanoid());
    const hidePlaceholder = ref(((_b = src.value) == null ? void 0 : _b.length) !== 0);
    const onEditLink = (e) => {
      const target = e.target;
      const value = target.value;
      hidePlaceholder.value = value.length !== 0;
      currentLink.value = value;
    };
    const onKeydown = (e) => {
      var _a2, _b2;
      if (e.key === "Enter") {
        setLink((_b2 = (_a2 = linkInputRef.value) == null ? void 0 : _a2.value) != null ? _b2 : "");
      }
    };
    const onConfirmLinkInput = () => {
      var _a2, _b2;
      setLink((_b2 = (_a2 = linkInputRef.value) == null ? void 0 : _a2.value) != null ? _b2 : "");
    };
    const onUploadFile = (e) => {
      var _a2;
      const file = (_a2 = e.target.files) == null ? void 0 : _a2[0];
      if (!file) return;
      onUpload(file).then((url) => {
        if (!url) return;
        setLink(url);
        hidePlaceholder.value = true;
      }).catch((err) => {
        console.error("An error occurred while uploading image");
        console.error(err);
      });
    };
    return () => {
      return /* @__PURE__ */ h("div", { class: clsx("image-edit", className) }, /* @__PURE__ */ h(Icon, { icon: imageIcon, class: "image-icon" }), /* @__PURE__ */ h("div", { class: clsx("link-importer", focusLinkInput.value && "focus") }, /* @__PURE__ */ h(
        "input",
        {
          ref: linkInputRef,
          draggable: "true",
          onDragstart: (e) => {
            e.preventDefault();
            e.stopPropagation();
          },
          disabled: readonly.value,
          class: "link-input-area",
          value: currentLink.value,
          onInput: onEditLink,
          onKeydown,
          onFocus: () => focusLinkInput.value = true,
          onBlur: () => focusLinkInput.value = false
        }
      ), !hidePlaceholder.value && /* @__PURE__ */ h("div", { class: "placeholder" }, /* @__PURE__ */ h(
        "input",
        {
          disabled: readonly.value,
          class: "hidden",
          id: uuid.value,
          type: "file",
          accept: "image/*",
          onChange: onUploadFile
        }
      ), /* @__PURE__ */ h("label", { class: "uploader", for: uuid.value }, /* @__PURE__ */ h(Icon, { icon: uploadButton })), /* @__PURE__ */ h("span", { class: "text", onClick: () => {
        var _a2;
        return (_a2 = linkInputRef.value) == null ? void 0 : _a2.focus();
      } }, uploadPlaceholderText))), currentLink.value && /* @__PURE__ */ h("div", { class: "confirm", onClick: () => onConfirmLinkInput() }, /* @__PURE__ */ h(Icon, { icon: confirmButton })));
    };
  }
});

const ImageViewer = defineComponent({
  props: {
    src: {
      type: Object,
      required: true
    },
    caption: {
      type: Object,
      required: true
    },
    ratio: {
      type: Object,
      required: true
    },
    selected: {
      type: Object,
      required: true
    },
    readonly: {
      type: Object,
      required: true
    },
    setAttr: {
      type: Function,
      required: true
    },
    config: {
      type: Object,
      required: true
    }
  },
  setup({ src, caption, ratio, readonly, setAttr, config }) {
    var _a;
    const imageRef = ref();
    const resizeHandle = ref();
    const showCaption = ref(Boolean((_a = caption.value) == null ? void 0 : _a.length));
    const timer = ref(0);
    const onImageLoad = () => {
      var _a2;
      const image = imageRef.value;
      if (!image) return;
      const host = image.closest(".milkdown-image-block");
      if (!host) return;
      const maxWidth = host.getBoundingClientRect().width;
      if (!maxWidth) return;
      const height = image.height;
      const width = image.width;
      const transformedHeight = width < maxWidth ? height : maxWidth * (height / width);
      const h2 = (transformedHeight * ((_a2 = ratio.value) != null ? _a2 : 1)).toFixed(2);
      image.dataset.origin = transformedHeight.toFixed(2);
      image.dataset.height = h2;
      image.style.height = `${h2}px`;
    };
    const onToggleCaption = (e) => {
      e.preventDefault();
      e.stopPropagation();
      if (readonly.value) return;
      showCaption.value = !showCaption.value;
    };
    const onInputCaption = (e) => {
      const target = e.target;
      const value = target.value;
      if (timer.value) window.clearTimeout(timer.value);
      timer.value = window.setTimeout(() => {
        setAttr("caption", value);
      }, 1e3);
    };
    const onBlurCaption = (e) => {
      const target = e.target;
      const value = target.value;
      if (timer.value) {
        window.clearTimeout(timer.value);
        timer.value = 0;
      }
      setAttr("caption", value);
    };
    const onResizeHandlePointerMove = (e) => {
      e.preventDefault();
      const image = imageRef.value;
      if (!image) return;
      const top = image.getBoundingClientRect().top;
      const height = e.clientY - top;
      const h2 = Number(height < 100 ? 100 : height).toFixed(2);
      image.dataset.height = h2;
      image.style.height = `${h2}px`;
    };
    const onResizeHandlePointerUp = () => {
      window.removeEventListener("pointermove", onResizeHandlePointerMove);
      window.removeEventListener("pointerup", onResizeHandlePointerUp);
      const image = imageRef.value;
      if (!image) return;
      const originHeight = Number(image.dataset.origin);
      const currentHeight = Number(image.dataset.height);
      const ratio2 = Number.parseFloat(
        Number(currentHeight / originHeight).toFixed(2)
      );
      if (Number.isNaN(ratio2)) return;
      setAttr("ratio", ratio2);
    };
    const onResizeHandlePointerDown = (e) => {
      if (readonly.value) return;
      e.preventDefault();
      e.stopPropagation();
      window.addEventListener("pointermove", onResizeHandlePointerMove);
      window.addEventListener("pointerup", onResizeHandlePointerUp);
    };
    return () => {
      return /* @__PURE__ */ h(Fragment, null, /* @__PURE__ */ h("div", { class: "image-wrapper" }, /* @__PURE__ */ h("div", { class: "operation" }, /* @__PURE__ */ h("div", { class: "operation-item", onPointerdown: onToggleCaption }, /* @__PURE__ */ h(Icon, { icon: config.captionIcon }))), /* @__PURE__ */ h(
        "img",
        {
          ref: imageRef,
          "data-type": IMAGE_DATA_TYPE,
          onLoad: onImageLoad,
          src: src.value,
          alt: caption.value
        }
      ), /* @__PURE__ */ h(
        "div",
        {
          ref: resizeHandle,
          class: "image-resize-handle",
          onPointerdown: onResizeHandlePointerDown
        }
      )), showCaption.value && /* @__PURE__ */ h(
        "input",
        {
          draggable: "true",
          onDragstart: (e) => {
            e.preventDefault();
            e.stopPropagation();
          },
          class: "caption-input",
          placeholder: config == null ? void 0 : config.captionPlaceholderText,
          onInput: onInputCaption,
          onBlur: onBlurCaption,
          value: caption.value
        }
      ));
    };
  }
});

var __defProp = Object.defineProperty;
var __getOwnPropSymbols = Object.getOwnPropertySymbols;
var __hasOwnProp = Object.prototype.hasOwnProperty;
var __propIsEnum = Object.prototype.propertyIsEnumerable;
var __defNormalProp = (obj, key, value) => key in obj ? __defProp(obj, key, { enumerable: true, configurable: true, writable: true, value }) : obj[key] = value;
var __spreadValues = (a, b) => {
  for (var prop in b || (b = {}))
    if (__hasOwnProp.call(b, prop))
      __defNormalProp(a, prop, b[prop]);
  if (__getOwnPropSymbols)
    for (var prop of __getOwnPropSymbols(b)) {
      if (__propIsEnum.call(b, prop))
        __defNormalProp(a, prop, b[prop]);
    }
  return a;
};
const MilkdownImageBlock = defineComponent({
  props: {
    src: {
      type: Object,
      required: true
    },
    caption: {
      type: Object,
      required: true
    },
    ratio: {
      type: Object,
      required: true
    },
    selected: {
      type: Object,
      required: true
    },
    readonly: {
      type: Object,
      required: true
    },
    setAttr: {
      type: Function,
      required: true
    },
    config: {
      type: Object,
      required: true
    }
  },
  setup(props) {
    const { src } = props;
    return () => {
      var _a;
      if (!((_a = src.value) == null ? void 0 : _a.length)) {
        return /* @__PURE__ */ h(
          ImageInput,
          {
            src: props.src,
            selected: props.selected,
            readonly: props.readonly,
            setLink: (link) => props.setAttr("src", link),
            imageIcon: props.config.imageIcon,
            uploadButton: props.config.uploadButton,
            confirmButton: props.config.confirmButton,
            uploadPlaceholderText: props.config.uploadPlaceholderText,
            onUpload: props.config.onUpload
          }
        );
      }
      return /* @__PURE__ */ h(ImageViewer, __spreadValues({}, props));
    };
  }
});

const imageBlockView = $view(
  imageBlockSchema.node,
  (ctx) => {
    return (initialNode, view, getPos) => {
      const src = ref(initialNode.attrs.src);
      const caption = ref(initialNode.attrs.caption);
      const ratio = ref(initialNode.attrs.ratio);
      const selected = ref(false);
      const readonly = ref(!view.editable);
      const setAttr = (attr, value) => {
        if (!view.editable) return;
        const pos = getPos();
        if (pos == null) return;
        view.dispatch(
          view.state.tr.setNodeAttribute(
            pos,
            attr,
            attr === "src" ? DOMPurify.sanitize(value) : value
          )
        );
      };
      const config = ctx.get(imageBlockConfig.key);
      const app = createApp(MilkdownImageBlock, {
        src,
        caption,
        ratio,
        selected,
        readonly,
        setAttr,
        config
      });
      const dom = document.createElement("div");
      dom.className = "milkdown-image-block";
      app.mount(dom);
      const disposeSelectedWatcher = watchEffect(() => {
        const isSelected = selected.value;
        if (isSelected) {
          dom.classList.add("selected");
        } else {
          dom.classList.remove("selected");
        }
      });
      const proxyDomURL = config.proxyDomURL;
      const bindAttrs = (node) => {
        if (!proxyDomURL) {
          src.value = node.attrs.src;
        } else {
          const proxiedURL = proxyDomURL(node.attrs.src);
          if (typeof proxiedURL === "string") {
            src.value = proxiedURL;
          } else {
            proxiedURL.then((url) => {
              src.value = url;
            }).catch(console.error);
          }
        }
        ratio.value = node.attrs.ratio;
        caption.value = node.attrs.caption;
        readonly.value = !view.editable;
      };
      bindAttrs(initialNode);
      return {
        dom,
        update: (updatedNode) => {
          if (updatedNode.type !== initialNode.type) return false;
          bindAttrs(updatedNode);
          return true;
        },
        stopEvent: (e) => {
          if (e.target instanceof HTMLInputElement) return true;
          return false;
        },
        selectNode: () => {
          selected.value = true;
        },
        deselectNode: () => {
          selected.value = false;
        },
        destroy: () => {
          disposeSelectedWatcher();
          app.unmount();
          dom.remove();
        }
      };
    };
  }
);
withMeta(imageBlockView, {
  displayName: "NodeView<image-block>",
  group: "ImageBlock"
});

const imageBlockComponent = [
  remarkImageBlockPlugin,
  imageBlockSchema,
  imageBlockView,
  imageBlockConfig
].flat();

export { IMAGE_DATA_TYPE, defaultImageBlockConfig, imageBlockComponent, imageBlockConfig, imageBlockSchema, imageBlockView, remarkImageBlockPlugin };
//# sourceMappingURL=index.js.map
