import { codeBlockConfig } from '@milkdown/kit/component/code-block';
import katex from 'katex';
import { createSlice } from '@milkdown/kit/ctx';
import { codeBlockSchema } from '@milkdown/kit/preset/commonmark';
import { findNodeInSelection, nodeRule } from '@milkdown/kit/prose';
import { NodeSelection, TextSelection, EditorState } from '@milkdown/kit/prose/state';
import { $nodeSchema, $command, $inputRule, $remark } from '@milkdown/kit/utils';
import { tooltipFactory, TooltipProvider } from '@milkdown/kit/plugin/tooltip';
import { redo, undo } from '@milkdown/kit/prose/history';
import { keymap } from '@milkdown/kit/prose/keymap';
import { Schema } from '@milkdown/kit/prose/model';
import { EditorView } from '@milkdown/kit/prose/view';
import { defineComponent, h, shallowRef, createApp } from 'vue';
import { Icon } from '@milkdown/kit/component';
import { textblockTypeInputRule } from '@milkdown/kit/prose/inputrules';
import remarkMath from 'remark-math';
import { visit } from 'unist-util-visit';

createSlice([], "FeaturesCtx");
createSlice({}, "CrepeCtx");
function useCrepeFeatures(ctx) {
  return ctx.use("FeaturesCtx");
}
function crepeFeatureConfig(feature) {
  return (ctx) => {
    useCrepeFeatures(ctx).update((features) => {
      if (features.includes(feature)) {
        return features;
      }
      return [...features, feature];
    });
  };
}

var CrepeFeature = /* @__PURE__ */ ((CrepeFeature2) => {
  CrepeFeature2["CodeMirror"] = "code-mirror";
  CrepeFeature2["ListItem"] = "list-item";
  CrepeFeature2["LinkTooltip"] = "link-tooltip";
  CrepeFeature2["Cursor"] = "cursor";
  CrepeFeature2["ImageBlock"] = "image-block";
  CrepeFeature2["BlockEdit"] = "block-edit";
  CrepeFeature2["Toolbar"] = "toolbar";
  CrepeFeature2["Placeholder"] = "placeholder";
  CrepeFeature2["Table"] = "table";
  CrepeFeature2["Latex"] = "latex";
  return CrepeFeature2;
})(CrepeFeature || {});

const confirmIcon = `
  <svg
    xmlns="http://www.w3.org/2000/svg"
    width="24"
    height="24"
    viewBox="0 0 24 24"
    fill="none"
  >
    <g clip-path="url(#clip0_1013_1606)">
      <path
        d="M9.00012 16.1998L5.50012 12.6998C5.11012 12.3098 4.49012 12.3098 4.10012 12.6998C3.71012 13.0898 3.71012 13.7098 4.10012 14.0998L8.29012 18.2898C8.68012 18.6798 9.31012 18.6798 9.70012 18.2898L20.3001 7.69982C20.6901 7.30982 20.6901 6.68982 20.3001 6.29982C19.9101 5.90982 19.2901 5.90982 18.9001 6.29982L9.00012 16.1998Z"
        fill="#817567"
      />
    </g>
    <defs>
      <clipPath id="clip0_1013_1606">
        <rect width="24" height="24" />
      </clipPath>
    </defs>
  </svg>
`;

const blockLatexSchema = codeBlockSchema.extendSchema((prev) => {
  return (ctx) => {
    const baseSchema = prev(ctx);
    return {
      ...baseSchema,
      toMarkdown: {
        match: baseSchema.toMarkdown.match,
        runner: (state, node) => {
          var _a, _b;
          const language = (_a = node.attrs.language) != null ? _a : "";
          if (language.toLowerCase() === "latex") {
            state.addNode(
              "math",
              void 0,
              ((_b = node.content.firstChild) == null ? void 0 : _b.text) || ""
            );
          } else {
            return baseSchema.toMarkdown.runner(state, node);
          }
        }
      }
    };
  };
});

const mathInlineId = "math_inline";
const mathInlineSchema = $nodeSchema(mathInlineId, () => ({
  group: "inline",
  inline: true,
  draggable: true,
  atom: true,
  attrs: {
    value: {
      default: ""
    }
  },
  parseDOM: [
    {
      tag: `span[data-type="${mathInlineId}"]`,
      getAttrs: (dom) => {
        var _a;
        return {
          value: (_a = dom.dataset.value) != null ? _a : ""
        };
      }
    }
  ],
  toDOM: (node) => {
    const code = node.attrs.value;
    const dom = document.createElement("span");
    dom.dataset.type = mathInlineId;
    dom.dataset.value = code;
    katex.render(code, dom, {
      throwOnError: false
    });
    return dom;
  },
  parseMarkdown: {
    match: (node) => node.type === "inlineMath",
    runner: (state, node, type) => {
      state.addNode(type, { value: node.value });
    }
  },
  toMarkdown: {
    match: (node) => node.type.name === mathInlineId,
    runner: (state, node) => {
      state.addNode("inlineMath", void 0, node.attrs.value);
    }
  }
}));

const toggleLatexCommand = $command("ToggleLatex", (ctx) => {
  return () => (state, dispatch) => {
    const {
      hasNode: hasLatex,
      pos: latexPos,
      target: latexNode
    } = findNodeInSelection(state, mathInlineSchema.type(ctx));
    const { selection, doc, tr } = state;
    if (!hasLatex) {
      const text = doc.textBetween(selection.from, selection.to);
      let _tr2 = tr.replaceSelectionWith(
        mathInlineSchema.type(ctx).create({
          value: text
        })
      );
      if (dispatch) {
        dispatch(
          _tr2.setSelection(NodeSelection.create(_tr2.doc, selection.from))
        );
      }
      return true;
    }
    const { from, to } = selection;
    if (!latexNode || latexPos < 0) return false;
    let _tr = tr.delete(latexPos, latexPos + 1);
    const content = latexNode.attrs.value;
    _tr = _tr.insertText(content, latexPos);
    if (dispatch) {
      dispatch(
        _tr.setSelection(
          TextSelection.create(_tr.doc, from, to + content.length - 1)
        )
      );
    }
    return true;
  };
});

const inlineLatexTooltip = tooltipFactory("INLINE_LATEX");

const LatexTooltip = defineComponent({
  props: {
    config: {
      type: Object,
      required: true
    },
    innerView: {
      type: Object,
      required: true
    },
    updateValue: {
      type: Object,
      required: true
    }
  },
  setup(props) {
    const innerViewRef = (el) => {
      if (!el || !(el instanceof HTMLElement)) return;
      while (el.firstChild) {
        el.removeChild(el.firstChild);
      }
      if (props.innerView.value) {
        el.appendChild(props.innerView.value.dom);
      }
    };
    const onUpdate = (e) => {
      e.preventDefault();
      props.updateValue.value();
    };
    return () => {
      return /* @__PURE__ */ h("div", { class: "container" }, props.innerView && /* @__PURE__ */ h("div", { ref: innerViewRef }), /* @__PURE__ */ h("button", { onPointerdown: onUpdate }, /* @__PURE__ */ h(Icon, { icon: props.config.inlineEditConfirm })));
    };
  }
});

var __typeError = (msg) => {
  throw TypeError(msg);
};
var __accessCheck = (obj, member, msg) => member.has(obj) || __typeError("Cannot " + msg);
var __privateGet = (obj, member, getter) => (__accessCheck(obj, member, "read from private field"), getter ? getter.call(obj) : member.get(obj));
var __privateAdd = (obj, member, value) => member.has(obj) ? __typeError("Cannot add the same private member more than once") : member instanceof WeakSet ? member.add(obj) : member.set(obj, value);
var __privateSet = (obj, member, value, setter) => (__accessCheck(obj, member, "write to private field"), member.set(obj, value), value);
var _content, _provider, _dom, _innerView, _updateValue, _app, _onHide, _shouldShow;
class LatexInlineTooltip {
  constructor(ctx, view, config) {
    this.ctx = ctx;
    __privateAdd(this, _content);
    __privateAdd(this, _provider);
    __privateAdd(this, _dom);
    __privateAdd(this, _innerView, shallowRef(null));
    __privateAdd(this, _updateValue, shallowRef(() => {
    }));
    __privateAdd(this, _app);
    __privateAdd(this, _onHide, () => {
      if (__privateGet(this, _innerView).value) {
        __privateGet(this, _innerView).value.destroy();
        __privateGet(this, _innerView).value = null;
      }
    });
    __privateAdd(this, _shouldShow, (view) => {
      const shouldShow = () => {
        const { selection, schema } = view.state;
        if (selection.empty) return false;
        if (!(selection instanceof NodeSelection)) return false;
        const node = selection.node;
        if (node.type.name !== mathInlineId) return false;
        const textFrom = selection.from;
        const paragraph = schema.nodes.paragraph.create(
          null,
          schema.text(node.attrs.value)
        );
        const innerView = new EditorView(__privateGet(this, _dom), {
          state: EditorState.create({
            doc: paragraph,
            schema: new Schema({
              nodes: {
                doc: {
                  content: "block+"
                },
                paragraph: {
                  content: "inline*",
                  group: "block",
                  parseDOM: [{ tag: "p" }],
                  toDOM() {
                    return ["p", 0];
                  }
                },
                text: {
                  group: "inline"
                }
              }
            }),
            plugins: [
              keymap({
                "Mod-z": undo,
                "Mod-Z": redo,
                "Mod-y": redo,
                Enter: () => {
                  __privateGet(this, _updateValue).value();
                  return true;
                }
              })
            ]
          })
        });
        __privateGet(this, _innerView).value = innerView;
        __privateGet(this, _updateValue).value = () => {
          const { tr } = view.state;
          tr.setNodeAttribute(textFrom, "value", innerView.state.doc.textContent);
          view.dispatch(tr);
          requestAnimationFrame(() => {
            view.focus();
          });
        };
        return true;
      };
      const show = shouldShow();
      if (!show) __privateGet(this, _onHide).call(this);
      return show;
    });
    this.update = (view, prevState) => {
      __privateGet(this, _provider).update(view, prevState);
    };
    this.destroy = () => {
      __privateGet(this, _app).unmount();
      __privateGet(this, _provider).destroy();
      __privateGet(this, _content).remove();
    };
    const content = document.createElement("div");
    content.className = "milkdown-latex-inline-edit";
    __privateSet(this, _content, content);
    __privateSet(this, _app, createApp(LatexTooltip, {
      config,
      innerView: __privateGet(this, _innerView),
      updateValue: __privateGet(this, _updateValue)
    }));
    __privateGet(this, _app).mount(content);
    __privateSet(this, _provider, new TooltipProvider({
      debounce: 0,
      content: __privateGet(this, _content),
      shouldShow: __privateGet(this, _shouldShow),
      offset: 10,
      floatingUIOptions: {
        placement: "bottom"
      }
    }));
    __privateGet(this, _provider).update(view);
    __privateSet(this, _dom, document.createElement("div"));
  }
}
_content = new WeakMap();
_provider = new WeakMap();
_dom = new WeakMap();
_innerView = new WeakMap();
_updateValue = new WeakMap();
_app = new WeakMap();
_onHide = new WeakMap();
_shouldShow = new WeakMap();

const mathInlineInputRule = $inputRule(
  (ctx) => nodeRule(/(?:\$)([^$]+)(?:\$)$/, mathInlineSchema.type(ctx), {
    getAttr: (match) => {
      var _a;
      return {
        value: (_a = match[1]) != null ? _a : ""
      };
    }
  })
);
const mathBlockInputRule = $inputRule(
  (ctx) => textblockTypeInputRule(/^\$\$[\s\n]$/, codeBlockSchema.type(ctx), () => ({
    language: "LaTeX"
  }))
);

const remarkMathPlugin = $remark(
  "remarkMath",
  () => remarkMath
);
function visitMathBlock(ast) {
  return visit(
    ast,
    "math",
    (node, index, parent) => {
      const { value } = node;
      const newNode = {
        type: "code",
        lang: "LaTeX",
        value
      };
      parent.children.splice(index, 1, newNode);
    }
  );
}
const remarkMathBlockPlugin = $remark(
  "remarkMathBlock",
  () => () => visitMathBlock
);

const latex = (editor, config) => {
  editor.config(crepeFeatureConfig(CrepeFeature.Latex)).config((ctx) => {
    const flags = useCrepeFeatures(ctx).get();
    const isCodeMirrorEnabled = flags.includes(CrepeFeature.CodeMirror);
    if (!isCodeMirrorEnabled) {
      throw new Error("You need to enable CodeMirror to use LaTeX feature");
    }
    ctx.update(codeBlockConfig.key, (prev) => ({
      ...prev,
      renderPreview: (language, content) => {
        if (language.toLowerCase() === "latex" && content.length > 0) {
          return renderLatex(content, config == null ? void 0 : config.katexOptions);
        }
        const renderPreview = prev.renderPreview;
        return renderPreview(language, content);
      }
    }));
    ctx.set(inlineLatexTooltip.key, {
      view: (view) => {
        var _a;
        return new LatexInlineTooltip(ctx, view, {
          inlineEditConfirm: (_a = config == null ? void 0 : config.inlineEditConfirm) != null ? _a : confirmIcon,
          ...config
        });
      }
    });
  }).use(remarkMathPlugin).use(remarkMathBlockPlugin).use(mathInlineSchema).use(inlineLatexTooltip).use(mathInlineInputRule).use(mathBlockInputRule).use(blockLatexSchema).use(toggleLatexCommand);
};
function renderLatex(content, options) {
  const html = katex.renderToString(content, {
    ...options,
    throwOnError: false,
    displayMode: true
  });
  return html;
}

export { latex };
//# sourceMappingURL=index.js.map
