import { Plugin, PluginKey, TextSelection } from "@milkdown/prose/state";
import { $ctx, $prose } from "@milkdown/utils";
import { computePosition, flip, offset } from "@floating-ui/dom";
import { posToDOMRect, findParentNode } from "@milkdown/prose";
import debounce from "lodash.debounce";
function slashFactory(id) {
  const slashSpec = $ctx(
    {},
    `${id}_SLASH_SPEC`
  );
  const slashPlugin = $prose((ctx) => {
    const spec = ctx.get(slashSpec.key);
    return new Plugin({
      key: new PluginKey(`${id}_SLASH`),
      ...spec
    });
  });
  const result = [slashSpec, slashPlugin];
  result.key = slashSpec.key;
  result.pluginKey = slashPlugin.key;
  slashSpec.meta = {
    package: "@milkdown/plugin-slash",
    displayName: `Ctx<slashSpec>|${id}`
  };
  slashPlugin.meta = {
    package: "@milkdown/plugin-slash",
    displayName: `Prose<slash>|${id}`
  };
  return result;
}
class SlashProvider {
  constructor(options) {
    this.#initialized = false;
    this.onShow = () => {
    };
    this.onHide = () => {
    };
    this.#onUpdate = (view, prevState) => {
      const { state, composing } = view;
      const { selection, doc } = state;
      const { ranges } = selection;
      const from = Math.min(...ranges.map((range) => range.$from.pos));
      const to = Math.max(...ranges.map((range) => range.$to.pos));
      const isSame = prevState && prevState.doc.eq(doc) && prevState.selection.eq(selection);
      if (!this.#initialized) {
        const root = this.#root ?? view.dom.parentElement ?? document.body;
        root.appendChild(this.element);
        this.#initialized = true;
      }
      if (composing || isSame) return;
      if (!this.#shouldShow(view, prevState)) {
        this.hide();
        return;
      }
      const virtualEl = {
        getBoundingClientRect: () => posToDOMRect(view, from, to)
      };
      computePosition(virtualEl, this.element, {
        placement: "bottom-start",
        middleware: [flip(), offset(this.#offset), ...this.#middleware],
        ...this.#floatingUIOptions
      }).then(({ x, y }) => {
        Object.assign(this.element.style, {
          left: `${x}px`,
          top: `${y}px`
        });
      }).catch(console.error);
      this.show();
    };
    this.update = (view, prevState) => {
      this.#updater(view, prevState);
    };
    this.getContent = (view, matchNode = (node) => node.type.name === "paragraph") => {
      const { selection } = view.state;
      const { empty, $from } = selection;
      const isTextBlock = view.state.selection instanceof TextSelection;
      if (typeof document === "undefined") return;
      const isSlashChildren = this.element.contains(document.activeElement);
      const notHasFocus = !view.hasFocus() && !isSlashChildren;
      const isReadonly = !view.editable;
      const paragraph = findParentNode(matchNode)(view.state.selection);
      const isNotInParagraph = !paragraph;
      if (notHasFocus || isReadonly || !empty || !isTextBlock || isNotInParagraph)
        return;
      return $from.parent.textBetween(
        Math.max(0, $from.parentOffset - 500),
        $from.parentOffset,
        void 0,
        "￼"
      );
    };
    this.destroy = () => {
      this.#updater.cancel();
    };
    this.show = () => {
      this.element.dataset.show = "true";
      this.onShow();
    };
    this.hide = () => {
      this.element.dataset.show = "false";
      this.onHide();
    };
    this.element = options.content;
    this.#debounce = options.debounce ?? 200;
    this.#shouldShow = options.shouldShow ?? this.#_shouldShow;
    this.#trigger = options.trigger ?? "/";
    this.#offset = options.offset;
    this.#middleware = options.middleware ?? [];
    this.#floatingUIOptions = options.floatingUIOptions ?? {};
    this.#root = options.root;
    this.#updater = debounce(this.#onUpdate, this.#debounce);
  }
  #initialized;
  /// @internal
  #middleware;
  /// @internal
  #floatingUIOptions;
  /// @internal
  #root;
  /// @internal
  #debounce;
  /// @internal
  #trigger;
  /// @internal
  #shouldShow;
  /// @internal
  #updater;
  /// The offset to get the block. Default is 0.
  #offset;
  #onUpdate;
  /// @internal
  #_shouldShow(view) {
    const currentTextBlockContent = this.getContent(view);
    if (!currentTextBlockContent) return false;
    const target = currentTextBlockContent.at(-1);
    if (!target) return false;
    return Array.isArray(this.#trigger) ? this.#trigger.includes(target) : this.#trigger === target;
  }
}
export {
  SlashProvider,
  slashFactory
};
//# sourceMappingURL=index.js.map
