import { expectDomTypeError } from "@milkdown/exception";
import { paragraphSchema, listItemSchema } from "@milkdown/preset-commonmark";
import { InputRule } from "@milkdown/prose/inputrules";
import { $markAttr, $markSchema, $inputRule, $useKeymap, $command, $nodeSchema, $prose, $remark } from "@milkdown/utils";
import { tableNodes, TableMap, CellSelection, goToNextCell, isInTable, deleteTable, deleteColumn, deleteRow, selectedRect, addColumnBefore, addColumnAfter, setCellAttr, columnResizing, tableEditing } from "@milkdown/prose/tables";
import { commandsCtx } from "@milkdown/core";
import { markRule, findParentNodeClosestToPos, cloneTr, findParentNodeType } from "@milkdown/prose";
import { toggleMark } from "@milkdown/prose/commands";
import { Selection, TextSelection, Plugin, PluginKey } from "@milkdown/prose/state";
import { imeSpan } from "prosemirror-safari-ime-span";
import remarkGFM from "remark-gfm";
function withMeta(plugin, meta) {
  Object.assign(plugin, {
    meta: {
      package: "@milkdown/preset-gfm",
      ...meta
    }
  });
  return plugin;
}
const strikethroughAttr = $markAttr("strike_through");
withMeta(strikethroughAttr, {
  displayName: "Attr<strikethrough>",
  group: "Strikethrough"
});
const strikethroughSchema = $markSchema("strike_through", (ctx) => ({
  parseDOM: [
    { tag: "del" },
    {
      style: "text-decoration",
      getAttrs: (value) => value === "line-through"
    }
  ],
  toDOM: (mark) => ["del", ctx.get(strikethroughAttr.key)(mark)],
  parseMarkdown: {
    match: (node) => node.type === "delete",
    runner: (state, node, markType) => {
      state.openMark(markType);
      state.next(node.children);
      state.closeMark(markType);
    }
  },
  toMarkdown: {
    match: (mark) => mark.type.name === "strike_through",
    runner: (state, mark) => {
      state.withMark(mark, "delete");
    }
  }
}));
withMeta(strikethroughSchema.mark, {
  displayName: "MarkSchema<strikethrough>",
  group: "Strikethrough"
});
withMeta(strikethroughSchema.ctx, {
  displayName: "MarkSchemaCtx<strikethrough>",
  group: "Strikethrough"
});
const toggleStrikethroughCommand = $command(
  "ToggleStrikeThrough",
  (ctx) => () => {
    return toggleMark(strikethroughSchema.type(ctx));
  }
);
withMeta(toggleStrikethroughCommand, {
  displayName: "Command<ToggleStrikethrough>",
  group: "Strikethrough"
});
const strikethroughInputRule = $inputRule((ctx) => {
  return markRule(
    new RegExp("(?<![\\w:/])(~{1,2})(.+?)\\1(?!\\w|\\/)"),
    strikethroughSchema.type(ctx)
  );
});
withMeta(strikethroughInputRule, {
  displayName: "InputRule<strikethrough>",
  group: "Strikethrough"
});
const strikethroughKeymap = $useKeymap("strikeThroughKeymap", {
  ToggleStrikethrough: {
    shortcuts: "Mod-Alt-x",
    command: (ctx) => {
      const commands2 = ctx.get(commandsCtx);
      return () => commands2.call(toggleStrikethroughCommand.key);
    }
  }
});
withMeta(strikethroughKeymap.ctx, {
  displayName: "KeymapCtx<strikethrough>",
  group: "Strikethrough"
});
withMeta(strikethroughKeymap.shortcuts, {
  displayName: "Keymap<strikethrough>",
  group: "Strikethrough"
});
const originalSchema = tableNodes({
  tableGroup: "block",
  cellContent: "paragraph",
  cellAttributes: {
    alignment: {
      default: "left",
      getFromDOM: (dom) => dom.style.textAlign || "left",
      setDOMAttr: (value, attrs) => {
        attrs.style = `text-align: ${value || "left"}`;
      }
    }
  }
});
const tableSchema = $nodeSchema("table", () => ({
  ...originalSchema.table,
  content: "table_header_row table_row+",
  disableDropCursor: true,
  parseMarkdown: {
    match: (node) => node.type === "table",
    runner: (state, node, type) => {
      const align = node.align;
      const children = node.children.map((x, i) => ({
        ...x,
        align,
        isHeader: i === 0
      }));
      state.openNode(type);
      state.next(children);
      state.closeNode();
    }
  },
  toMarkdown: {
    match: (node) => node.type.name === "table",
    runner: (state, node) => {
      const firstLine = node.content.firstChild?.content;
      if (!firstLine) return;
      const align = [];
      firstLine.forEach((cell) => {
        align.push(cell.attrs.alignment);
      });
      state.openNode("table", void 0, { align });
      state.next(node.content);
      state.closeNode();
    }
  }
}));
withMeta(tableSchema.node, {
  displayName: "NodeSchema<table>",
  group: "Table"
});
withMeta(tableSchema.ctx, {
  displayName: "NodeSchemaCtx<table>",
  group: "Table"
});
const tableHeaderRowSchema = $nodeSchema("table_header_row", () => ({
  ...originalSchema.table_row,
  disableDropCursor: true,
  content: "(table_header)*",
  parseDOM: [{ tag: "tr[data-is-header]" }],
  toDOM() {
    return ["tr", { "data-is-header": true }, 0];
  },
  parseMarkdown: {
    match: (node) => Boolean(node.type === "tableRow" && node.isHeader),
    runner: (state, node, type) => {
      const align = node.align;
      const children = node.children.map((x, i) => ({
        ...x,
        align: align[i],
        isHeader: node.isHeader
      }));
      state.openNode(type);
      state.next(children);
      state.closeNode();
    }
  },
  toMarkdown: {
    match: (node) => node.type.name === "table_header_row",
    runner: (state, node) => {
      state.openNode("tableRow", void 0, { isHeader: true });
      state.next(node.content);
      state.closeNode();
    }
  }
}));
withMeta(tableHeaderRowSchema.node, {
  displayName: "NodeSchema<tableHeaderRow>",
  group: "Table"
});
withMeta(tableHeaderRowSchema.ctx, {
  displayName: "NodeSchemaCtx<tableHeaderRow>",
  group: "Table"
});
const tableRowSchema = $nodeSchema("table_row", () => ({
  ...originalSchema.table_row,
  disableDropCursor: true,
  content: "(table_cell)*",
  parseMarkdown: {
    match: (node) => node.type === "tableRow",
    runner: (state, node, type) => {
      const align = node.align;
      const children = node.children.map((x, i) => ({
        ...x,
        align: align[i]
      }));
      state.openNode(type);
      state.next(children);
      state.closeNode();
    }
  },
  toMarkdown: {
    match: (node) => node.type.name === "table_row",
    runner: (state, node) => {
      if (node.content.size === 0) {
        return;
      }
      state.openNode("tableRow");
      state.next(node.content);
      state.closeNode();
    }
  }
}));
withMeta(tableRowSchema.node, {
  displayName: "NodeSchema<tableRow>",
  group: "Table"
});
withMeta(tableRowSchema.ctx, {
  displayName: "NodeSchemaCtx<tableRow>",
  group: "Table"
});
const tableCellSchema = $nodeSchema("table_cell", () => ({
  ...originalSchema.table_cell,
  disableDropCursor: true,
  parseMarkdown: {
    match: (node) => node.type === "tableCell" && !node.isHeader,
    runner: (state, node, type) => {
      const align = node.align;
      state.openNode(type, { alignment: align }).openNode(state.schema.nodes.paragraph).next(node.children).closeNode().closeNode();
    }
  },
  toMarkdown: {
    match: (node) => node.type.name === "table_cell",
    runner: (state, node) => {
      state.openNode("tableCell").next(node.content).closeNode();
    }
  }
}));
withMeta(tableCellSchema.node, {
  displayName: "NodeSchema<tableCell>",
  group: "Table"
});
withMeta(tableCellSchema.ctx, {
  displayName: "NodeSchemaCtx<tableCell>",
  group: "Table"
});
const tableHeaderSchema = $nodeSchema("table_header", () => ({
  ...originalSchema.table_header,
  disableDropCursor: true,
  parseMarkdown: {
    match: (node) => node.type === "tableCell" && !!node.isHeader,
    runner: (state, node, type) => {
      const align = node.align;
      state.openNode(type, { alignment: align });
      state.openNode(state.schema.nodes.paragraph);
      state.next(node.children);
      state.closeNode();
      state.closeNode();
    }
  },
  toMarkdown: {
    match: (node) => node.type.name === "table_header",
    runner: (state, node) => {
      state.openNode("tableCell");
      state.next(node.content);
      state.closeNode();
    }
  }
}));
withMeta(tableHeaderSchema.node, {
  displayName: "NodeSchema<tableHeader>",
  group: "Table"
});
withMeta(tableHeaderSchema.ctx, {
  displayName: "NodeSchemaCtx<tableHeader>",
  group: "Table"
});
function createTable(ctx, rowsCount = 3, colsCount = 3) {
  const cells = Array(colsCount).fill(0).map(() => tableCellSchema.type(ctx).createAndFill());
  const headerCells = Array(colsCount).fill(0).map(() => tableHeaderSchema.type(ctx).createAndFill());
  const rows = Array(rowsCount).fill(0).map(
    (_, i) => i === 0 ? tableHeaderRowSchema.type(ctx).create(null, headerCells) : tableRowSchema.type(ctx).create(null, cells)
  );
  return tableSchema.type(ctx).create(null, rows);
}
function findTable($pos) {
  return findParentNodeClosestToPos(
    (node) => node.type.spec.tableRole === "table"
  )($pos);
}
function getCellsInCol(columnIndexes, selection) {
  const table = findTable(selection.$from);
  if (!table) return void 0;
  const map = TableMap.get(table.node);
  const indexes = Array.isArray(columnIndexes) ? columnIndexes : [columnIndexes];
  return indexes.filter((index) => index >= 0 && index <= map.width - 1).flatMap((index) => {
    const cells = map.cellsInRect({
      left: index,
      right: index + 1,
      top: 0,
      bottom: map.height
    });
    return cells.map((nodePos) => {
      const node = table.node.nodeAt(nodePos);
      const pos = nodePos + table.start;
      return { pos, start: pos + 1, node, depth: table.depth + 2 };
    });
  });
}
function getCellsInRow(rowIndex, selection) {
  const table = findTable(selection.$from);
  if (!table) {
    return;
  }
  const map = TableMap.get(table.node);
  const indexes = Array.isArray(rowIndex) ? rowIndex : [rowIndex];
  return indexes.filter((index) => index >= 0 && index <= map.height - 1).flatMap((index) => {
    const cells = map.cellsInRect({
      left: 0,
      right: map.width,
      top: index,
      bottom: index + 1
    });
    return cells.map((nodePos) => {
      const node = table.node.nodeAt(nodePos);
      const pos = nodePos + table.start;
      return { pos, start: pos + 1, node, depth: table.depth + 2 };
    });
  });
}
function selectLine(type) {
  return (index, pos) => (tr) => {
    pos = pos ?? tr.selection.from;
    const $pos = tr.doc.resolve(pos);
    const $node = findParentNodeClosestToPos(
      (node) => node.type.name === "table"
    )($pos);
    const table = $node ? {
      node: $node.node,
      from: $node.start
    } : void 0;
    const isRowSelection = type === "row";
    if (table) {
      const map = TableMap.get(table.node);
      if (index >= 0 && index < (isRowSelection ? map.height : map.width)) {
        const lastCell = map.positionAt(
          isRowSelection ? index : map.height - 1,
          isRowSelection ? map.width - 1 : index,
          table.node
        );
        const $lastCell = tr.doc.resolve(table.from + lastCell);
        const createCellSelection = isRowSelection ? CellSelection.rowSelection : CellSelection.colSelection;
        const firstCell = map.positionAt(
          isRowSelection ? index : 0,
          isRowSelection ? 0 : index,
          table.node
        );
        const $firstCell = tr.doc.resolve(table.from + firstCell);
        return cloneTr(
          tr.setSelection(
            createCellSelection($lastCell, $firstCell)
          )
        );
      }
    }
    return tr;
  };
}
const selectRow = selectLine("row");
const selectCol = selectLine("col");
function addRowWithAlignment(ctx, tr, { map, tableStart, table }, row) {
  const rowPos = Array(row).fill(0).reduce((acc, _, i) => {
    return acc + table.child(i).nodeSize;
  }, tableStart);
  const cells = Array(map.width).fill(0).map((_, col) => {
    const headerCol = table.nodeAt(map.map[col]);
    return tableCellSchema.type(ctx).createAndFill({ alignment: headerCol?.attrs.alignment });
  });
  tr.insert(rowPos, tableRowSchema.type(ctx).create(null, cells));
  return tr;
}
function convertRowsToTable(tableNode, arrayOfNodes) {
  const rowsPM = [];
  const map = TableMap.get(tableNode);
  for (let rowIndex = 0; rowIndex < map.height; rowIndex++) {
    const row = tableNode.child(rowIndex);
    const rowCells = [];
    for (let colIndex = 0; colIndex < map.width; colIndex++) {
      if (!arrayOfNodes[rowIndex][colIndex]) continue;
      const cellPos = map.map[rowIndex * map.width + colIndex];
      const cell = arrayOfNodes[rowIndex][colIndex];
      const oldCell = tableNode.nodeAt(cellPos);
      const newCell = oldCell.type.createChecked(
        Object.assign({}, cell.attrs),
        cell.content,
        cell.marks
      );
      rowCells.push(newCell);
    }
    rowsPM.push(row.type.createChecked(row.attrs, rowCells, row.marks));
  }
  const newTable = tableNode.type.createChecked(
    tableNode.attrs,
    rowsPM,
    tableNode.marks
  );
  return newTable;
}
function convertTableToRows(tableNode) {
  const map = TableMap.get(tableNode);
  const rows = [];
  const rowCount = map.height;
  const colCount = map.width;
  for (let rowIndex = 0; rowIndex < rowCount; rowIndex++) {
    const row = [];
    for (let colIndex = 0; colIndex < colCount; colIndex++) {
      let cellIndex = rowIndex * colCount + colIndex;
      let cellPos = map.map[cellIndex];
      if (rowIndex > 0) {
        const topCellIndex = cellIndex - colCount;
        const topCellPos = map.map[topCellIndex];
        if (cellPos === topCellPos) {
          row.push(null);
          continue;
        }
      }
      if (colIndex > 0) {
        const leftCellIndex = cellIndex - 1;
        const leftCellPos = map.map[leftCellIndex];
        if (cellPos === leftCellPos) {
          row.push(null);
          continue;
        }
      }
      if (!cellPos) {
        row.push(null);
      } else {
        row.push(tableNode.nodeAt(cellPos));
      }
    }
    rows.push(row);
  }
  return rows;
}
function getSelectionRangeInRow(tr, startRowIndex, endRowIndex = startRowIndex) {
  let startIndex = startRowIndex;
  let endIndex = endRowIndex;
  for (let i = startRowIndex; i >= 0; i--) {
    const cells = getCellsInRow(i, tr.selection);
    if (cells) {
      cells.forEach((cell) => {
        const maybeEndIndex = cell.node.attrs.rowspan + i - 1;
        if (maybeEndIndex >= startIndex) {
          startIndex = i;
        }
        if (maybeEndIndex > endIndex) {
          endIndex = maybeEndIndex;
        }
      });
    }
  }
  for (let i = startRowIndex; i <= endIndex; i++) {
    const cells = getCellsInRow(i, tr.selection);
    if (cells) {
      cells.forEach((cell) => {
        const maybeEndIndex = cell.node.attrs.rowspan + i - 1;
        if (cell.node.attrs.rowspan > 1 && maybeEndIndex > endIndex) {
          endIndex = maybeEndIndex;
        }
      });
    }
  }
  const indexes = [];
  for (let i = startIndex; i <= endIndex; i++) {
    const maybeCells = getCellsInRow(i, tr.selection);
    if (maybeCells && maybeCells.length > 0) {
      indexes.push(i);
    }
  }
  startIndex = indexes[0];
  endIndex = indexes[indexes.length - 1];
  const firstSelectedRowCells = getCellsInRow(startIndex, tr.selection);
  const firstColumnCells = getCellsInCol(0, tr.selection);
  if (!firstSelectedRowCells || !firstColumnCells) {
    return;
  }
  const $anchor = tr.doc.resolve(
    firstSelectedRowCells[firstSelectedRowCells.length - 1].pos
  );
  let headCell;
  for (let i = endIndex; i >= startIndex; i--) {
    const rowCells = getCellsInRow(i, tr.selection);
    if (rowCells && rowCells.length > 0) {
      for (let j = firstColumnCells.length - 1; j >= 0; j--) {
        if (firstColumnCells[j].pos === rowCells[0].pos) {
          headCell = rowCells[0];
          break;
        }
      }
      if (headCell) {
        break;
      }
    }
  }
  if (!headCell) {
    return;
  }
  const $head = tr.doc.resolve(headCell.pos);
  return { $anchor, $head, indexes };
}
function moveRowInArrayOfRows(rows, indexesOrigin, indexesTarget, directionOverride) {
  const direction = indexesOrigin[0] > indexesTarget[0] ? -1 : 1;
  const rowsExtracted = rows.splice(indexesOrigin[0], indexesOrigin.length);
  const positionOffset = rowsExtracted.length % 2 === 0 ? 1 : 0;
  let target;
  {
    target = direction === -1 ? indexesTarget[0] : indexesTarget[indexesTarget.length - 1] - positionOffset;
  }
  rows.splice(target, 0, ...rowsExtracted);
  return rows;
}
function moveRow(moveRowParams) {
  const { tr, origin, target, select, pos } = moveRowParams;
  const $pos = tr.doc.resolve(pos);
  const table = findTable($pos);
  if (!table) return false;
  const indexesOriginRow = getSelectionRangeInRow(tr, origin)?.indexes;
  const indexesTargetRow = getSelectionRangeInRow(tr, target)?.indexes;
  if (!indexesOriginRow || !indexesTargetRow) return false;
  if (indexesOriginRow.includes(target)) return false;
  const newTable = moveTableRow(
    table.node,
    indexesOriginRow,
    indexesTargetRow
  );
  tr.replaceWith(table.pos, table.pos + table.node.nodeSize, newTable);
  if (!select) return true;
  const map = TableMap.get(newTable);
  const start = table.start;
  const index = target;
  const lastCell = map.positionAt(index, map.width - 1, newTable);
  const $lastCell = tr.doc.resolve(start + lastCell);
  const firstCell = map.positionAt(index, 0, newTable);
  const $firstCell = tr.doc.resolve(start + firstCell);
  tr.setSelection(CellSelection.rowSelection($lastCell, $firstCell));
  return true;
}
function moveTableRow(table, indexesOrigin, indexesTarget, direction) {
  let rows = convertTableToRows(table);
  rows = moveRowInArrayOfRows(rows, indexesOrigin, indexesTarget);
  return convertRowsToTable(table, rows);
}
function getSelectionRangeInCol(tr, startColIndex, endColIndex = startColIndex) {
  let startIndex = startColIndex;
  let endIndex = endColIndex;
  for (let i = startColIndex; i >= 0; i--) {
    const cells = getCellsInCol(i, tr.selection);
    if (cells) {
      cells.forEach((cell) => {
        const maybeEndIndex = cell.node.attrs.colspan + i - 1;
        if (maybeEndIndex >= startIndex) {
          startIndex = i;
        }
        if (maybeEndIndex > endIndex) {
          endIndex = maybeEndIndex;
        }
      });
    }
  }
  for (let i = startColIndex; i <= endIndex; i++) {
    const cells = getCellsInCol(i, tr.selection);
    if (cells) {
      cells.forEach((cell) => {
        const maybeEndIndex = cell.node.attrs.colspan + i - 1;
        if (cell.node.attrs.colspan > 1 && maybeEndIndex > endIndex) {
          endIndex = maybeEndIndex;
        }
      });
    }
  }
  const indexes = [];
  for (let i = startIndex; i <= endIndex; i++) {
    const maybeCells = getCellsInCol(i, tr.selection);
    if (maybeCells && maybeCells.length > 0) {
      indexes.push(i);
    }
  }
  startIndex = indexes[0];
  endIndex = indexes[indexes.length - 1];
  const firstSelectedColumnCells = getCellsInCol(startIndex, tr.selection);
  const firstRowCells = getCellsInRow(0, tr.selection);
  if (!firstSelectedColumnCells || !firstRowCells) {
    return;
  }
  const $anchor = tr.doc.resolve(
    firstSelectedColumnCells[firstSelectedColumnCells.length - 1].pos
  );
  let headCell;
  for (let i = endIndex; i >= startIndex; i--) {
    const columnCells = getCellsInCol(i, tr.selection);
    if (columnCells && columnCells.length > 0) {
      for (let j = firstRowCells.length - 1; j >= 0; j--) {
        if (firstRowCells[j].pos === columnCells[0].pos) {
          headCell = columnCells[0];
          break;
        }
      }
      if (headCell) {
        break;
      }
    }
  }
  if (!headCell) {
    return;
  }
  const $head = tr.doc.resolve(headCell.pos);
  return { $anchor, $head, indexes };
}
function transpose(array) {
  return array[0].map((_, i) => array.map((column) => column[i]));
}
function moveCol(moveColParams) {
  const { tr, origin, target, select, pos } = moveColParams;
  const $pos = tr.doc.resolve(pos);
  const table = findTable($pos);
  if (!table) return false;
  const indexesOriginColumn = getSelectionRangeInCol(tr, origin)?.indexes;
  const indexesTargetColumn = getSelectionRangeInCol(tr, target)?.indexes;
  if (!indexesOriginColumn || !indexesTargetColumn) return false;
  if (indexesOriginColumn.includes(target)) return false;
  const newTable = moveTableCol(
    table.node,
    indexesOriginColumn,
    indexesTargetColumn
  );
  tr.replaceWith(table.pos, table.pos + table.node.nodeSize, newTable);
  if (!select) return true;
  const map = TableMap.get(newTable);
  const start = table.start;
  const index = target;
  const lastCell = map.positionAt(map.height - 1, index, newTable);
  const $lastCell = tr.doc.resolve(start + lastCell);
  const firstCell = map.positionAt(0, index, newTable);
  const $firstCell = tr.doc.resolve(start + firstCell);
  tr.setSelection(CellSelection.colSelection($lastCell, $firstCell));
  return true;
}
function moveTableCol(table, indexesOrigin, indexesTarget, direction) {
  let rows = transpose(convertTableToRows(table));
  rows = moveRowInArrayOfRows(rows, indexesOrigin, indexesTarget);
  rows = transpose(rows);
  return convertRowsToTable(table, rows);
}
function getAllCellsInTable(selection) {
  const table = findTable(selection.$from);
  if (!table) return;
  const map = TableMap.get(table.node);
  const cells = map.cellsInRect({
    left: 0,
    right: map.width,
    top: 0,
    bottom: map.height
  });
  return cells.map((nodePos) => {
    const node = table.node.nodeAt(nodePos);
    const pos = nodePos + table.start;
    return { pos, start: pos + 1, node };
  });
}
function selectTable(tr) {
  const cells = getAllCellsInTable(tr.selection);
  if (cells && cells[0]) {
    const $firstCell = tr.doc.resolve(cells[0].pos);
    const last = cells[cells.length - 1];
    if (last) {
      const $lastCell = tr.doc.resolve(last.pos);
      return cloneTr(tr.setSelection(new CellSelection($lastCell, $firstCell)));
    }
  }
  return tr;
}
const goToPrevTableCellCommand = $command(
  "GoToPrevTableCell",
  () => () => goToNextCell(-1)
);
withMeta(goToPrevTableCellCommand, {
  displayName: "Command<goToPrevTableCellCommand>",
  group: "Table"
});
const goToNextTableCellCommand = $command(
  "GoToNextTableCell",
  () => () => goToNextCell(1)
);
withMeta(goToNextTableCellCommand, {
  displayName: "Command<goToNextTableCellCommand>",
  group: "Table"
});
const exitTable = $command(
  "ExitTable",
  (ctx) => () => (state, dispatch) => {
    if (!isInTable(state)) return false;
    const { $head } = state.selection;
    const table = findParentNodeType($head, tableSchema.type(ctx));
    if (!table) return false;
    const { to } = table;
    const tr = state.tr.replaceWith(
      to,
      to,
      paragraphSchema.type(ctx).createAndFill()
    );
    tr.setSelection(Selection.near(tr.doc.resolve(to), 1)).scrollIntoView();
    dispatch?.(tr);
    return true;
  }
);
withMeta(exitTable, {
  displayName: "Command<breakTableCommand>",
  group: "Table"
});
const insertTableCommand = $command(
  "InsertTable",
  (ctx) => ({ row, col } = {}) => (state, dispatch) => {
    const { selection, tr } = state;
    const { from } = selection;
    const table = createTable(ctx, row, col);
    const _tr = tr.replaceSelectionWith(table);
    const sel = Selection.findFrom(_tr.doc.resolve(from), 1, true);
    if (sel) _tr.setSelection(sel);
    dispatch?.(_tr);
    return true;
  }
);
withMeta(insertTableCommand, {
  displayName: "Command<insertTableCommand>",
  group: "Table"
});
const moveRowCommand = $command(
  "MoveRow",
  () => ({ from, to, pos } = {}) => (state, dispatch) => {
    const { tr } = state;
    if (moveRow({
      tr,
      origin: from ?? 0,
      target: to ?? 0,
      pos: pos ?? state.selection.from,
      select: true
    })) {
      dispatch?.(tr);
      return true;
    }
    return false;
  }
);
withMeta(moveRowCommand, {
  displayName: "Command<moveRowCommand>",
  group: "Table"
});
const moveColCommand = $command(
  "MoveCol",
  () => ({ from, to, pos } = {}) => (state, dispatch) => {
    const { tr } = state;
    if (moveCol({
      tr,
      origin: from ?? 0,
      target: to ?? 0,
      pos: pos ?? state.selection.from,
      select: true
    })) {
      dispatch?.(tr);
      return true;
    }
    return false;
  }
);
withMeta(moveColCommand, {
  displayName: "Command<moveColCommand>",
  group: "Table"
});
const selectRowCommand = $command(
  "SelectRow",
  () => (payload = { index: 0 }) => (state, dispatch) => {
    const { tr } = state;
    const result = dispatch?.(selectRow(payload.index, payload.pos)(tr));
    return Boolean(result);
  }
);
withMeta(selectRowCommand, {
  displayName: "Command<selectRowCommand>",
  group: "Table"
});
const selectColCommand = $command(
  "SelectCol",
  () => (payload = { index: 0 }) => (state, dispatch) => {
    const { tr } = state;
    const result = dispatch?.(selectCol(payload.index, payload.pos)(tr));
    return Boolean(result);
  }
);
withMeta(selectColCommand, {
  displayName: "Command<selectColCommand>",
  group: "Table"
});
const selectTableCommand = $command(
  "SelectTable",
  () => () => (state, dispatch) => {
    const { tr } = state;
    const result = dispatch?.(selectTable(tr));
    return Boolean(result);
  }
);
withMeta(selectTableCommand, {
  displayName: "Command<selectTableCommand>",
  group: "Table"
});
const deleteSelectedCellsCommand = $command(
  "DeleteSelectedCells",
  () => () => (state, dispatch) => {
    const { selection } = state;
    if (!(selection instanceof CellSelection)) return false;
    const isRow = selection.isRowSelection();
    const isCol = selection.isColSelection();
    if (isRow && isCol) return deleteTable(state, dispatch);
    if (isCol) return deleteColumn(state, dispatch);
    else return deleteRow(state, dispatch);
  }
);
withMeta(deleteSelectedCellsCommand, {
  displayName: "Command<deleteSelectedCellsCommand>",
  group: "Table"
});
const addColBeforeCommand = $command(
  "AddColBefore",
  () => () => addColumnBefore
);
withMeta(addColBeforeCommand, {
  displayName: "Command<addColBeforeCommand>",
  group: "Table"
});
const addColAfterCommand = $command(
  "AddColAfter",
  () => () => addColumnAfter
);
withMeta(addColAfterCommand, {
  displayName: "Command<addColAfterCommand>",
  group: "Table"
});
const addRowBeforeCommand = $command(
  "AddRowBefore",
  (ctx) => () => (state, dispatch) => {
    if (!isInTable(state)) return false;
    if (dispatch) {
      const rect = selectedRect(state);
      dispatch(addRowWithAlignment(ctx, state.tr, rect, rect.top));
    }
    return true;
  }
);
withMeta(addRowBeforeCommand, {
  displayName: "Command<addRowBeforeCommand>",
  group: "Table"
});
const addRowAfterCommand = $command(
  "AddRowAfter",
  (ctx) => () => (state, dispatch) => {
    if (!isInTable(state)) return false;
    if (dispatch) {
      const rect = selectedRect(state);
      dispatch(addRowWithAlignment(ctx, state.tr, rect, rect.bottom));
    }
    return true;
  }
);
withMeta(addRowAfterCommand, {
  displayName: "Command<addRowAfterCommand>",
  group: "Table"
});
const setAlignCommand = $command(
  "SetAlign",
  () => (alignment = "left") => setCellAttr("alignment", alignment)
);
withMeta(setAlignCommand, {
  displayName: "Command<setAlignCommand>",
  group: "Table"
});
const insertTableInputRule = $inputRule(
  (ctx) => new InputRule(
    /^\|(?<col>\d+)[xX](?<row>\d+)\|\s$/,
    (state, match, start, end) => {
      const $start = state.doc.resolve(start);
      if (!$start.node(-1).canReplaceWith(
        $start.index(-1),
        $start.indexAfter(-1),
        tableSchema.type(ctx)
      ))
        return null;
      const row = Math.max(Number(match.groups?.row ?? 0), 2);
      const tableNode = createTable(ctx, row, Number(match.groups?.col));
      const tr = state.tr.replaceRangeWith(start, end, tableNode);
      return tr.setSelection(TextSelection.create(tr.doc, start + 3)).scrollIntoView();
    }
  )
);
withMeta(insertTableInputRule, {
  displayName: "InputRule<insertTableInputRule>",
  group: "Table"
});
const tableKeymap = $useKeymap("tableKeymap", {
  NextCell: {
    priority: 100,
    shortcuts: ["Mod-]", "Tab"],
    command: (ctx) => {
      const commands2 = ctx.get(commandsCtx);
      return () => commands2.call(goToNextTableCellCommand.key);
    }
  },
  PrevCell: {
    shortcuts: ["Mod-[", "Shift-Tab"],
    command: (ctx) => {
      const commands2 = ctx.get(commandsCtx);
      return () => commands2.call(goToPrevTableCellCommand.key);
    }
  },
  ExitTable: {
    shortcuts: ["Mod-Enter", "Enter"],
    command: (ctx) => {
      const commands2 = ctx.get(commandsCtx);
      return () => commands2.call(exitTable.key);
    }
  }
});
withMeta(tableKeymap.ctx, {
  displayName: "KeymapCtx<table>",
  group: "Table"
});
withMeta(tableKeymap.shortcuts, {
  displayName: "Keymap<table>",
  group: "Table"
});
const id$1 = "footnote_definition";
const markdownId = "footnoteDefinition";
const footnoteDefinitionSchema = $nodeSchema(
  "footnote_definition",
  () => ({
    group: "block",
    content: "block+",
    defining: true,
    attrs: {
      label: {
        default: "",
        validate: "string"
      }
    },
    parseDOM: [
      {
        tag: `dl[data-type="${id$1}"]`,
        getAttrs: (dom) => {
          if (!(dom instanceof HTMLElement)) throw expectDomTypeError(dom);
          return {
            label: dom.dataset.label
          };
        },
        contentElement: "dd"
      }
    ],
    toDOM: (node) => {
      const label = node.attrs.label;
      return [
        "dl",
        {
          // TODO: add a prosemirror plugin to sync label on change
          "data-label": label,
          "data-type": id$1
        },
        ["dt", label],
        ["dd", 0]
      ];
    },
    parseMarkdown: {
      match: ({ type }) => type === markdownId,
      runner: (state, node, type) => {
        state.openNode(type, {
          label: node.label
        }).next(node.children).closeNode();
      }
    },
    toMarkdown: {
      match: (node) => node.type.name === id$1,
      runner: (state, node) => {
        state.openNode(markdownId, void 0, {
          label: node.attrs.label,
          identifier: node.attrs.label
        }).next(node.content).closeNode();
      }
    }
  })
);
withMeta(footnoteDefinitionSchema.ctx, {
  displayName: "NodeSchemaCtx<footnodeDef>",
  group: "footnote"
});
withMeta(footnoteDefinitionSchema.node, {
  displayName: "NodeSchema<footnodeDef>",
  group: "footnote"
});
const id = "footnote_reference";
const footnoteReferenceSchema = $nodeSchema(
  "footnote_reference",
  () => ({
    group: "inline",
    inline: true,
    atom: true,
    attrs: {
      label: {
        default: "",
        validate: "string"
      }
    },
    parseDOM: [
      {
        tag: `sup[data-type="${id}"]`,
        getAttrs: (dom) => {
          if (!(dom instanceof HTMLElement)) throw expectDomTypeError(dom);
          return {
            label: dom.dataset.label
          };
        }
      }
    ],
    toDOM: (node) => {
      const label = node.attrs.label;
      return [
        "sup",
        {
          // TODO: add a prosemirror plugin to sync label on change
          "data-label": label,
          "data-type": id
        },
        label
      ];
    },
    parseMarkdown: {
      match: ({ type }) => type === "footnoteReference",
      runner: (state, node, type) => {
        state.addNode(type, {
          label: node.label
        });
      }
    },
    toMarkdown: {
      match: (node) => node.type.name === id,
      runner: (state, node) => {
        state.addNode("footnoteReference", void 0, void 0, {
          label: node.attrs.label,
          identifier: node.attrs.label
        });
      }
    }
  })
);
withMeta(footnoteReferenceSchema.ctx, {
  displayName: "NodeSchemaCtx<footnodeRef>",
  group: "footnote"
});
withMeta(footnoteReferenceSchema.node, {
  displayName: "NodeSchema<footnodeRef>",
  group: "footnote"
});
const extendListItemSchemaForTask = listItemSchema.extendSchema(
  (prev) => {
    return (ctx) => {
      const baseSchema = prev(ctx);
      return {
        ...baseSchema,
        attrs: {
          ...baseSchema.attrs,
          checked: {
            default: null,
            validate: "boolean|null"
          }
        },
        parseDOM: [
          {
            tag: 'li[data-item-type="task"]',
            getAttrs: (dom) => {
              if (!(dom instanceof HTMLElement)) throw expectDomTypeError(dom);
              return {
                label: dom.dataset.label,
                listType: dom.dataset.listType,
                spread: dom.dataset.spread,
                checked: dom.dataset.checked ? dom.dataset.checked === "true" : null
              };
            }
          },
          ...baseSchema?.parseDOM || []
        ],
        toDOM: (node) => {
          if (baseSchema.toDOM && node.attrs.checked == null)
            return baseSchema.toDOM(node);
          return [
            "li",
            {
              "data-item-type": "task",
              "data-label": node.attrs.label,
              "data-list-type": node.attrs.listType,
              "data-spread": node.attrs.spread,
              "data-checked": node.attrs.checked
            },
            0
          ];
        },
        parseMarkdown: {
          match: ({ type }) => type === "listItem",
          runner: (state, node, type) => {
            if (node.checked == null) {
              baseSchema.parseMarkdown.runner(state, node, type);
              return;
            }
            const label = node.label != null ? `${node.label}.` : "•";
            const checked = node.checked != null ? Boolean(node.checked) : null;
            const listType = node.label != null ? "ordered" : "bullet";
            const spread = node.spread != null ? `${node.spread}` : "true";
            state.openNode(type, { label, listType, spread, checked });
            state.next(node.children);
            state.closeNode();
          }
        },
        toMarkdown: {
          match: (node) => node.type.name === "list_item",
          runner: (state, node) => {
            if (node.attrs.checked == null) {
              baseSchema.toMarkdown.runner(state, node);
              return;
            }
            const label = node.attrs.label;
            const listType = node.attrs.listType;
            const spread = node.attrs.spread === "true";
            const checked = node.attrs.checked;
            state.openNode("listItem", void 0, {
              label,
              listType,
              spread,
              checked
            });
            state.next(node.content);
            state.closeNode();
          }
        }
      };
    };
  }
);
withMeta(extendListItemSchemaForTask.node, {
  displayName: "NodeSchema<taskListItem>",
  group: "ListItem"
});
withMeta(extendListItemSchemaForTask.ctx, {
  displayName: "NodeSchemaCtx<taskListItem>",
  group: "ListItem"
});
const wrapInTaskListInputRule = $inputRule(() => {
  return new InputRule(
    /^\[(?<checked>\s|x)\]\s$/,
    (state, match, start, end) => {
      const pos = state.doc.resolve(start);
      let depth = 0;
      let node = pos.node(depth);
      while (node && node.type.name !== "list_item") {
        depth--;
        node = pos.node(depth);
      }
      if (!node || node.attrs.checked != null) return null;
      const checked = Boolean(match.groups?.checked === "x");
      const finPos = pos.before(depth);
      const tr = state.tr;
      tr.deleteRange(start, end).setNodeMarkup(finPos, void 0, {
        ...node.attrs,
        checked
      });
      return tr;
    }
  );
});
withMeta(wrapInTaskListInputRule, {
  displayName: "InputRule<wrapInTaskListInputRule>",
  group: "ListItem"
});
const keymap = [
  strikethroughKeymap,
  tableKeymap
].flat();
const inputRules = [
  insertTableInputRule,
  wrapInTaskListInputRule
];
const markInputRules = [strikethroughInputRule];
const autoInsertSpanPlugin = $prose(() => imeSpan);
withMeta(autoInsertSpanPlugin, {
  displayName: "Prose<autoInsertSpanPlugin>",
  group: "Prose"
});
const columnResizingPlugin = $prose(() => columnResizing({}));
withMeta(columnResizingPlugin, {
  displayName: "Prose<columnResizingPlugin>",
  group: "Prose"
});
const tableEditingPlugin = $prose(
  () => tableEditing({ allowTableNodeSelection: true })
);
withMeta(tableEditingPlugin, {
  displayName: "Prose<tableEditingPlugin>",
  group: "Prose"
});
const remarkGFMPlugin = $remark("remarkGFM", () => remarkGFM);
withMeta(remarkGFMPlugin.plugin, {
  displayName: "Remark<remarkGFMPlugin>",
  group: "Remark"
});
withMeta(remarkGFMPlugin.options, {
  displayName: "RemarkConfig<remarkGFMPlugin>",
  group: "Remark"
});
const pluginKey = new PluginKey("MILKDOWN_KEEP_TABLE_ALIGN_PLUGIN");
function getChildIndex(node, parent) {
  let index = 0;
  parent.forEach((child, _offset, i) => {
    if (child === node) index = i;
  });
  return index;
}
const keepTableAlignPlugin = $prose(() => {
  return new Plugin({
    key: pluginKey,
    appendTransaction: (_tr, oldState, state) => {
      let tr;
      const check = (node, pos) => {
        if (!tr) tr = state.tr;
        if (node.type.name !== "table_cell") return;
        const $pos = state.doc.resolve(pos);
        const tableRow = $pos.node($pos.depth);
        const table = $pos.node($pos.depth - 1);
        const tableHeaderRow = table.firstChild;
        if (!tableHeaderRow) return;
        const index = getChildIndex(node, tableRow);
        const headerCell = tableHeaderRow.maybeChild(index);
        if (!headerCell) return;
        const align = headerCell.attrs.alignment;
        const currentAlign = node.attrs.alignment;
        if (align === currentAlign) return;
        tr.setNodeMarkup(pos, void 0, { ...node.attrs, alignment: align });
      };
      if (oldState.doc !== state.doc) state.doc.descendants(check);
      return tr;
    }
  });
});
withMeta(keepTableAlignPlugin, {
  displayName: "Prose<keepTableAlignPlugin>",
  group: "Prose"
});
const plugins = [
  keepTableAlignPlugin,
  autoInsertSpanPlugin,
  remarkGFMPlugin,
  tableEditingPlugin
].flat();
const schema = [
  extendListItemSchemaForTask,
  tableSchema,
  tableHeaderRowSchema,
  tableRowSchema,
  tableHeaderSchema,
  tableCellSchema,
  footnoteDefinitionSchema,
  footnoteReferenceSchema,
  strikethroughAttr,
  strikethroughSchema
].flat();
const commands = [
  goToNextTableCellCommand,
  goToPrevTableCellCommand,
  exitTable,
  insertTableCommand,
  moveRowCommand,
  moveColCommand,
  selectRowCommand,
  selectColCommand,
  selectTableCommand,
  deleteSelectedCellsCommand,
  addRowBeforeCommand,
  addRowAfterCommand,
  addColBeforeCommand,
  addColAfterCommand,
  setAlignCommand,
  toggleStrikethroughCommand
];
const gfm = [
  schema,
  inputRules,
  markInputRules,
  keymap,
  commands,
  plugins
].flat();
export {
  addColAfterCommand,
  addColBeforeCommand,
  addRowAfterCommand,
  addRowBeforeCommand,
  addRowWithAlignment,
  autoInsertSpanPlugin,
  columnResizingPlugin,
  commands,
  createTable,
  deleteSelectedCellsCommand,
  exitTable,
  extendListItemSchemaForTask,
  findTable,
  footnoteDefinitionSchema,
  footnoteReferenceSchema,
  getAllCellsInTable,
  getCellsInCol,
  getCellsInRow,
  gfm,
  goToNextTableCellCommand,
  goToPrevTableCellCommand,
  inputRules,
  insertTableCommand,
  insertTableInputRule,
  keepTableAlignPlugin,
  keymap,
  markInputRules,
  moveCol,
  moveColCommand,
  moveRow,
  moveRowCommand,
  plugins,
  remarkGFMPlugin,
  schema,
  selectCol,
  selectColCommand,
  selectLine,
  selectRow,
  selectRowCommand,
  selectTable,
  selectTableCommand,
  setAlignCommand,
  strikethroughAttr,
  strikethroughInputRule,
  strikethroughKeymap,
  strikethroughSchema,
  tableCellSchema,
  tableEditingPlugin,
  tableHeaderRowSchema,
  tableHeaderSchema,
  tableKeymap,
  tableRowSchema,
  tableSchema,
  toggleStrikethroughCommand,
  wrapInTaskListInputRule
};
//# sourceMappingURL=index.js.map
