(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
  typeof define === 'function' && define.amd ? define(['exports'], factory) :
  (factory((global.MinDash = {})));
}(this, (function (exports) { 'use strict';

  /**
   * Flatten array, one level deep.
   *
   * @param {Array<?>} arr
   *
   * @return {Array<?>}
   */
  function flatten(arr) {
    return Array.prototype.concat.apply([], arr);
  }

  var nativeToString = Object.prototype.toString;
  var nativeHasOwnProperty = Object.prototype.hasOwnProperty;

  function isUndefined(obj) {
    return obj === undefined;
  }

  function isDefined(obj) {
    return obj !== undefined;
  }

  function isNil(obj) {
    return obj == null;
  }

  function isArray(obj) {
    return nativeToString.call(obj) === '[object Array]';
  }

  function isObject(obj) {
    return nativeToString.call(obj) === '[object Object]';
  }

  function isNumber(obj) {
    return nativeToString.call(obj) === '[object Number]';
  }

  function isFunction(obj) {
    return nativeToString.call(obj) === '[object Function]';
  }

  function isString(obj) {
    return nativeToString.call(obj) === '[object String]';
  }

  /**
   * Ensure collection is an array.
   *
   * @param {Object} obj
   */
  function ensureArray(obj) {

    if (isArray(obj)) {
      return;
    }

    throw new Error('must supply array');
  }

  /**
   * Return true, if target owns a property with the given key.
   *
   * @param {Object} target
   * @param {String} key
   *
   * @return {Boolean}
   */
  function has(target, key) {
    return nativeHasOwnProperty.call(target, key);
  }

  /**
   * Find element in collection.
   *
   * @param  {Array|Object} collection
   * @param  {Function|Object} matcher
   *
   * @return {Object}
   */
  function find(collection, matcher) {

    matcher = toMatcher(matcher);

    var match;

    forEach(collection, function (val, key) {
      if (matcher(val, key)) {
        match = val;

        return false;
      }
    });

    return match;
  }

  /**
   * Find element index in collection.
   *
   * @param  {Array|Object} collection
   * @param  {Function} matcher
   *
   * @return {Object}
   */
  function findIndex(collection, matcher) {

    matcher = toMatcher(matcher);

    var idx = isArray(collection) ? -1 : undefined;

    forEach(collection, function (val, key) {
      if (matcher(val, key)) {
        idx = key;

        return false;
      }
    });

    return idx;
  }

  /**
   * Find element in collection.
   *
   * @param  {Array|Object} collection
   * @param  {Function} matcher
   *
   * @return {Array} result
   */
  function filter(collection, matcher) {

    var result = [];

    forEach(collection, function (val, key) {
      if (matcher(val, key)) {
        result.push(val);
      }
    });

    return result;
  }

  /**
   * Iterate over collection; returning something
   * (non-undefined) will stop iteration.
   *
   * @param  {Array|Object} collection
   * @param  {Function} iterator
   *
   * @return {Object} return result that stopped the iteration
   */
  function forEach(collection, iterator) {

    if (isUndefined(collection)) {
      return;
    }

    var convertKey = isArray(collection) ? toNum : identity;

    for (var key in collection) {

      if (has(collection, key)) {
        var val = collection[key];

        var result = iterator(val, convertKey(key));

        if (result === false) {
          return;
        }
      }
    }
  }

  /**
   * Return collection without element.
   *
   * @param  {Array} arr
   * @param  {Function} matcher
   *
   * @return {Array}
   */
  function without(arr, matcher) {

    if (isUndefined(arr)) {
      return [];
    }

    ensureArray(arr);

    matcher = toMatcher(matcher);

    return arr.filter(function (el, idx) {
      return !matcher(el, idx);
    });
  }

  /**
   * Reduce collection, returning a single result.
   *
   * @param  {Object|Array} collection
   * @param  {Function} iterator
   * @param  {Any} result
   *
   * @return {Any} result returned from last iterator
   */
  function reduce(collection, iterator, result) {

    forEach(collection, function (value, idx) {
      result = iterator(result, value, idx);
    });

    return result;
  }

  /**
   * Return true if every element in the collection
   * matches the criteria.
   *
   * @param  {Object|Array} collection
   * @param  {Function} matcher
   *
   * @return {Boolean}
   */
  function every(collection, matcher) {

    return reduce(collection, function (matches, val, key) {
      return matches && matcher(val, key);
    }, true);
  }

  /**
   * Return true if some elements in the collection
   * match the criteria.
   *
   * @param  {Object|Array} collection
   * @param  {Function} matcher
   *
   * @return {Boolean}
   */
  function some(collection, matcher) {

    return !!find(collection, matcher);
  }

  /**
   * Transform a collection into another collection
   * by piping each member through the given fn.
   *
   * @param  {Object|Array}   collection
   * @param  {Function} fn
   *
   * @return {Array} transformed collection
   */
  function map(collection, fn) {

    var result = [];

    forEach(collection, function (val, key) {
      result.push(fn(val, key));
    });

    return result;
  }

  /**
   * Get the collections keys.
   *
   * @param  {Object|Array} collection
   *
   * @return {Array}
   */
  function keys(collection) {
    return collection && Object.keys(collection) || [];
  }

  /**
   * Shorthand for `keys(o).length`.
   *
   * @param  {Object|Array} collection
   *
   * @return {Number}
   */
  function size(collection) {
    return keys(collection).length;
  }

  /**
   * Get the values in the collection.
   *
   * @param  {Object|Array} collection
   *
   * @return {Array}
   */
  function values(collection) {
    return map(collection, function (val) {
      return val;
    });
  }

  /**
   * Group collection members by attribute.
   *
   * @param  {Object|Array} collection
   * @param  {Function} extractor
   *
   * @return {Object} map with { attrValue => [ a, b, c ] }
   */
  function groupBy(collection, extractor) {
    var grouped = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};


    extractor = toExtractor(extractor);

    forEach(collection, function (val) {
      var discriminator = extractor(val) || '_';

      var group = grouped[discriminator];

      if (!group) {
        group = grouped[discriminator] = [];
      }

      group.push(val);
    });

    return grouped;
  }

  function uniqueBy(extractor) {

    extractor = toExtractor(extractor);

    var grouped = {};

    for (var _len = arguments.length, collections = Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
      collections[_key - 1] = arguments[_key];
    }

    forEach(collections, function (c) {
      return groupBy(c, extractor, grouped);
    });

    var result = map(grouped, function (val, key) {
      return val[0];
    });

    return result;
  }

  var unionBy = uniqueBy;

  /**
   * Sort collection by criteria.
   *
   * @param  {Object|Array} collection
   * @param  {String|Function} extractor
   *
   * @return {Array}
   */
  function sortBy(collection, extractor) {

    extractor = toExtractor(extractor);

    var sorted = [];

    forEach(collection, function (value, key) {
      var disc = extractor(value, key);

      var entry = {
        d: disc,
        v: value
      };

      for (var idx = 0; idx < sorted.length; idx++) {
        var d = sorted[idx].d;


        if (disc < d) {
          sorted.splice(idx, 0, entry);
          return;
        }
      }

      // not inserted, append (!)
      sorted.push(entry);
    });

    return map(sorted, function (e) {
      return e.v;
    });
  }

  /**
   * Create an object pattern matcher.
   *
   * @example
   *
   * const matcher = matchPattern({ id: 1 });
   *
   * var element = find(elements, matcher);
   *
   * @param  {Object} pattern
   *
   * @return {Function} matcherFn
   */
  function matchPattern(pattern) {

    return function (el) {

      return every(pattern, function (val, key) {
        return el[key] === val;
      });
    };
  }

  function toExtractor(extractor) {
    return isFunction(extractor) ? extractor : function (e) {
      return e[extractor];
    };
  }

  function toMatcher(matcher) {
    return isFunction(matcher) ? matcher : function (e) {
      return e === matcher;
    };
  }

  function identity(arg) {
    return arg;
  }

  function toNum(arg) {
    return Number(arg);
  }

  function _toConsumableArray(arr) { if (Array.isArray(arr)) { for (var i = 0, arr2 = Array(arr.length); i < arr.length; i++) { arr2[i] = arr[i]; } return arr2; } else { return Array.from(arr); } }

  var slice = Array.prototype.slice;

  /**
   * Debounce fn, calling it only once if
   * the given time elapsed between calls.
   *
   * @param  {Function} fn
   * @param  {Number} timeout
   *
   * @return {Function} debounced function
   */
  function debounce(fn, timeout) {

    var timer;

    return function () {

      var args = slice.call(arguments);

      if (timer) {
        clearTimeout(timer);
      }

      timer = setTimeout(function () {
        fn.apply(undefined, _toConsumableArray(args));
      }, timeout);
    };
  }

  /**
   * Bind function against target <this>.
   *
   * @param  {Function} fn
   * @param  {Object}   target
   *
   * @return {Function} bound function
   */
  function bind(fn, target) {
    return fn.bind(target);
  }

  var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

  /**
   * Convenience wrapper for `Object.assign`.
   *
   * @param {Object} target
   * @param {...Object} others
   *
   * @return {Object} the target
   */
  function assign(target) {
    for (var _len = arguments.length, others = Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
      others[_key - 1] = arguments[_key];
    }

    return _extends.apply(undefined, [target].concat(others));
  }

  /**
   * Pick given properties from the target object.
   *
   * @param {Object} target
   * @param {Array} properties
   *
   * @return {Object} target
   */
  function pick(target, properties) {

    var result = {};

    var obj = Object(target);

    forEach(properties, function (prop) {

      if (prop in obj) {
        result[prop] = target[prop];
      }
    });

    return result;
  }

  /**
   * Pick all target properties, excluding the given ones.
   *
   * @param {Object} target
   * @param {Array} properties
   *
   * @return {Object} target
   */
  function omit(target, properties) {

    var result = {};

    var obj = Object(target);

    forEach(obj, function (prop, key) {

      if (properties.indexOf(key) === -1) {
        result[key] = prop;
      }
    });

    return result;
  }

  /**
   * Recursively merge `...sources` into given target.
   *
   * Does support merging objects; does not support merging arrays.
   *
   * @param {Object} target
   * @param {...Object} sources
   *
   * @return {Object} the target
   */
  function merge(target) {
    for (var _len2 = arguments.length, sources = Array(_len2 > 1 ? _len2 - 1 : 0), _key2 = 1; _key2 < _len2; _key2++) {
      sources[_key2 - 1] = arguments[_key2];
    }

    if (!sources.length) {
      return target;
    }

    forEach(sources, function (source) {

      // skip non-obj sources, i.e. null
      if (!source || !isObject(source)) {
        return;
      }

      forEach(source, function (sourceVal, key) {

        var targetVal = target[key];

        if (isObject(sourceVal)) {

          if (!isObject(targetVal)) {
            // override target[key] with object
            targetVal = {};
          }

          target[key] = merge(targetVal, sourceVal);
        } else {
          target[key] = sourceVal;
        }
      });
    });

    return target;
  }

  exports.flatten = flatten;
  exports.find = find;
  exports.findIndex = findIndex;
  exports.filter = filter;
  exports.forEach = forEach;
  exports.without = without;
  exports.reduce = reduce;
  exports.every = every;
  exports.some = some;
  exports.map = map;
  exports.keys = keys;
  exports.size = size;
  exports.values = values;
  exports.groupBy = groupBy;
  exports.uniqueBy = uniqueBy;
  exports.unionBy = unionBy;
  exports.sortBy = sortBy;
  exports.matchPattern = matchPattern;
  exports.debounce = debounce;
  exports.bind = bind;
  exports.isUndefined = isUndefined;
  exports.isDefined = isDefined;
  exports.isNil = isNil;
  exports.isArray = isArray;
  exports.isObject = isObject;
  exports.isNumber = isNumber;
  exports.isFunction = isFunction;
  exports.isString = isString;
  exports.ensureArray = ensureArray;
  exports.has = has;
  exports.assign = assign;
  exports.pick = pick;
  exports.omit = omit;
  exports.merge = merge;

  Object.defineProperty(exports, '__esModule', { value: true });

})));
