/*
 * MinIO Javascript Library for Amazon S3 Compatible Cloud Storage, (C) 2015, 2016 MinIO, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import * as Crypto from "crypto";
import Through2 from 'through2';
import { isFunction } from "./internal/helper.mjs";
import * as xmlParsers from "./xml-parsers.mjs";

// getConcater returns a stream that concatenates the input and emits
// the concatenated output when 'end' has reached. If an optional
// parser function is passed upon reaching the 'end' of the stream,
// `parser(concatenated_data)` will be emitted.
export function getConcater(parser, emitError) {
  var objectMode = false;
  var bufs = [];
  if (parser && !isFunction(parser)) {
    throw new TypeError('parser should be of type "function"');
  }
  if (parser) {
    objectMode = true;
  }
  return Through2({
    objectMode
  }, function (chunk, enc, cb) {
    bufs.push(chunk);
    cb();
  }, function (cb) {
    if (emitError) {
      cb(parser(Buffer.concat(bufs).toString()));
      // cb(e) would mean we have to emit 'end' by explicitly calling this.push(null)
      this.push(null);
      return;
    }
    if (bufs.length) {
      if (parser) {
        this.push(parser(Buffer.concat(bufs).toString()));
      } else {
        this.push(Buffer.concat(bufs));
      }
    }
    cb();
  });
}

// A through stream that calculates md5sum and sha256sum
export function getHashSummer(enableSHA256) {
  var md5 = Crypto.createHash('md5');
  var sha256 = Crypto.createHash('sha256');
  return Through2.obj(function (chunk, enc, cb) {
    if (enableSHA256) {
      sha256.update(chunk);
    } else {
      md5.update(chunk);
    }
    cb();
  }, function (cb) {
    var md5sum = '';
    var sha256sum = '';
    if (enableSHA256) {
      sha256sum = sha256.digest('hex');
    } else {
      md5sum = md5.digest('base64');
    }
    var hashData = {
      md5sum,
      sha256sum
    };
    this.push(hashData);
    this.push(null);
    cb();
  });
}

// Following functions return a stream object that parses XML
// and emits suitable Javascript objects.

// Parses listObjects response.
export function getListObjectsTransformer() {
  return getConcater(xmlParsers.parseListObjects);
}

// Parses listObjects response.
export function getListObjectsV2Transformer() {
  return getConcater(xmlParsers.parseListObjectsV2);
}

// Parses listObjects with metadata response.
export function getListObjectsV2WithMetadataTransformer() {
  return getConcater(xmlParsers.parseListObjectsV2WithMetadata);
}

// Parses GET/SET BucketNotification response
export function getBucketNotificationTransformer() {
  return getConcater(xmlParsers.parseBucketNotification);
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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