/*
 * MinIO Javascript Library for Amazon S3 Compatible Cloud Storage, (C) 2015 MinIO, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import { XMLParser } from 'fast-xml-parser';
import * as errors from "./errors.mjs";
import { isObject, parseXml, sanitizeETag, sanitizeObjectKey, sanitizeSize, toArray } from "./internal/helper.mjs";
const fxpWithoutNumParser = new XMLParser({
  numberParseOptions: {
    skipLike: /./
  }
});

// parse XML response for bucket notification
export function parseBucketNotification(xml) {
  var result = {
    TopicConfiguration: [],
    QueueConfiguration: [],
    CloudFunctionConfiguration: []
  };
  // Parse the events list
  var genEvents = function (events) {
    var result = [];
    if (events) {
      toArray(events).forEach(s3event => {
        result.push(s3event);
      });
    }
    return result;
  };
  // Parse all filter rules
  var genFilterRules = function (filters) {
    var result = [];
    if (filters) {
      filters = toArray(filters);
      if (filters[0].S3Key) {
        filters[0].S3Key = toArray(filters[0].S3Key);
        if (filters[0].S3Key[0].FilterRule) {
          toArray(filters[0].S3Key[0].FilterRule).forEach(rule => {
            var Name = toArray(rule.Name)[0];
            var Value = toArray(rule.Value)[0];
            result.push({
              Name,
              Value
            });
          });
        }
      }
    }
    return result;
  };
  var xmlobj = parseXml(xml);
  xmlobj = xmlobj.NotificationConfiguration;

  // Parse all topic configurations in the xml
  if (xmlobj.TopicConfiguration) {
    toArray(xmlobj.TopicConfiguration).forEach(config => {
      var Id = toArray(config.Id)[0];
      var Topic = toArray(config.Topic)[0];
      var Event = genEvents(config.Event);
      var Filter = genFilterRules(config.Filter);
      result.TopicConfiguration.push({
        Id,
        Topic,
        Event,
        Filter
      });
    });
  }
  // Parse all topic configurations in the xml
  if (xmlobj.QueueConfiguration) {
    toArray(xmlobj.QueueConfiguration).forEach(config => {
      var Id = toArray(config.Id)[0];
      var Queue = toArray(config.Queue)[0];
      var Event = genEvents(config.Event);
      var Filter = genFilterRules(config.Filter);
      result.QueueConfiguration.push({
        Id,
        Queue,
        Event,
        Filter
      });
    });
  }
  // Parse all QueueConfiguration arrays
  if (xmlobj.CloudFunctionConfiguration) {
    toArray(xmlobj.CloudFunctionConfiguration).forEach(config => {
      var Id = toArray(config.Id)[0];
      var CloudFunction = toArray(config.CloudFunction)[0];
      var Event = genEvents(config.Event);
      var Filter = genFilterRules(config.Filter);
      result.CloudFunctionConfiguration.push({
        Id,
        CloudFunction,
        Event,
        Filter
      });
    });
  }
  return result;
}
const formatObjInfo = (content, opts = {}) => {
  let {
    Key,
    LastModified,
    ETag,
    Size,
    VersionId,
    IsLatest
  } = content;
  if (!isObject(opts)) {
    opts = {};
  }
  const name = sanitizeObjectKey(toArray(Key)[0]);
  const lastModified = new Date(toArray(LastModified)[0]);
  const etag = sanitizeETag(toArray(ETag)[0]);
  const size = sanitizeSize(Size);
  return {
    name,
    lastModified,
    etag,
    size,
    versionId: VersionId,
    isLatest: IsLatest,
    isDeleteMarker: opts.IsDeleteMarker ? opts.IsDeleteMarker : false
  };
};

// parse XML response for list objects in a bucket
export function parseListObjects(xml) {
  var result = {
    objects: [],
    isTruncated: false
  };
  let isTruncated = false;
  let nextMarker, nextVersionKeyMarker;
  const xmlobj = fxpWithoutNumParser.parse(xml);
  const parseCommonPrefixesEntity = responseEntity => {
    if (responseEntity) {
      toArray(responseEntity).forEach(commonPrefix => {
        result.objects.push({
          prefix: sanitizeObjectKey(toArray(commonPrefix.Prefix)[0]),
          size: 0
        });
      });
    }
  };
  const listBucketResult = xmlobj.ListBucketResult;
  const listVersionsResult = xmlobj.ListVersionsResult;
  if (listBucketResult) {
    if (listBucketResult.IsTruncated) {
      isTruncated = listBucketResult.IsTruncated;
    }
    if (listBucketResult.Contents) {
      toArray(listBucketResult.Contents).forEach(content => {
        const name = sanitizeObjectKey(toArray(content.Key)[0]);
        const lastModified = new Date(toArray(content.LastModified)[0]);
        const etag = sanitizeETag(toArray(content.ETag)[0]);
        const size = sanitizeSize(content.Size);
        result.objects.push({
          name,
          lastModified,
          etag,
          size
        });
      });
    }
    if (listBucketResult.NextMarker) {
      nextMarker = listBucketResult.NextMarker;
    } else if (isTruncated && result.objects.length > 0) {
      nextMarker = result.objects[result.objects.length - 1].name;
    }
    parseCommonPrefixesEntity(listBucketResult.CommonPrefixes);
  }
  if (listVersionsResult) {
    if (listVersionsResult.IsTruncated) {
      isTruncated = listVersionsResult.IsTruncated;
    }
    if (listVersionsResult.Version) {
      toArray(listVersionsResult.Version).forEach(content => {
        result.objects.push(formatObjInfo(content));
      });
    }
    if (listVersionsResult.DeleteMarker) {
      toArray(listVersionsResult.DeleteMarker).forEach(content => {
        result.objects.push(formatObjInfo(content, {
          IsDeleteMarker: true
        }));
      });
    }
    if (listVersionsResult.NextKeyMarker) {
      nextVersionKeyMarker = listVersionsResult.NextKeyMarker;
    }
    if (listVersionsResult.NextVersionIdMarker) {
      result.versionIdMarker = listVersionsResult.NextVersionIdMarker;
    }
    parseCommonPrefixesEntity(listVersionsResult.CommonPrefixes);
  }
  result.isTruncated = isTruncated;
  if (isTruncated) {
    result.nextMarker = nextVersionKeyMarker || nextMarker;
  }
  return result;
}

// parse XML response for list objects v2 in a bucket
export function parseListObjectsV2(xml) {
  var result = {
    objects: [],
    isTruncated: false
  };
  var xmlobj = parseXml(xml);
  if (!xmlobj.ListBucketResult) {
    throw new errors.InvalidXMLError('Missing tag: "ListBucketResult"');
  }
  xmlobj = xmlobj.ListBucketResult;
  if (xmlobj.IsTruncated) {
    result.isTruncated = xmlobj.IsTruncated;
  }
  if (xmlobj.NextContinuationToken) {
    result.nextContinuationToken = xmlobj.NextContinuationToken;
  }
  if (xmlobj.Contents) {
    toArray(xmlobj.Contents).forEach(content => {
      var name = sanitizeObjectKey(toArray(content.Key)[0]);
      var lastModified = new Date(content.LastModified);
      var etag = sanitizeETag(content.ETag);
      var size = content.Size;
      result.objects.push({
        name,
        lastModified,
        etag,
        size
      });
    });
  }
  if (xmlobj.CommonPrefixes) {
    toArray(xmlobj.CommonPrefixes).forEach(commonPrefix => {
      result.objects.push({
        prefix: sanitizeObjectKey(toArray(commonPrefix.Prefix)[0]),
        size: 0
      });
    });
  }
  return result;
}

// parse XML response for list objects v2 with metadata in a bucket
export function parseListObjectsV2WithMetadata(xml) {
  var result = {
    objects: [],
    isTruncated: false
  };
  var xmlobj = parseXml(xml);
  if (!xmlobj.ListBucketResult) {
    throw new errors.InvalidXMLError('Missing tag: "ListBucketResult"');
  }
  xmlobj = xmlobj.ListBucketResult;
  if (xmlobj.IsTruncated) {
    result.isTruncated = xmlobj.IsTruncated;
  }
  if (xmlobj.NextContinuationToken) {
    result.nextContinuationToken = xmlobj.NextContinuationToken;
  }
  if (xmlobj.Contents) {
    toArray(xmlobj.Contents).forEach(content => {
      var name = sanitizeObjectKey(content.Key);
      var lastModified = new Date(content.LastModified);
      var etag = sanitizeETag(content.ETag);
      var size = content.Size;
      var metadata;
      if (content.UserMetadata != null) {
        metadata = toArray(content.UserMetadata)[0];
      } else {
        metadata = null;
      }
      result.objects.push({
        name,
        lastModified,
        etag,
        size,
        metadata
      });
    });
  }
  if (xmlobj.CommonPrefixes) {
    toArray(xmlobj.CommonPrefixes).forEach(commonPrefix => {
      result.objects.push({
        prefix: sanitizeObjectKey(toArray(commonPrefix.Prefix)[0]),
        size: 0
      });
    });
  }
  return result;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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