"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.removeDirAndFiles = removeDirAndFiles;
var fs = _interopRequireWildcard(require("fs"), true);
var path = _interopRequireWildcard(require("path"), true);
var querystring = _interopRequireWildcard(require("query-string"), true);
var errors = _interopRequireWildcard(require("./errors.js"), true);
var _helper = require("./internal/helper.js");
var _type = require("./internal/type.js");
exports.RETENTION_MODES = _type.RETENTION_MODES;
exports.ENCRYPTION_TYPES = _type.ENCRYPTION_TYPES;
exports.LEGAL_HOLD_STATUS = _type.LEGAL_HOLD_STATUS;
exports.RETENTION_VALIDITY_UNITS = _type.RETENTION_VALIDITY_UNITS;
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
const DEFAULT_REGION = 'us-east-1';
exports.DEFAULT_REGION = DEFAULT_REGION;
const PRESIGN_EXPIRY_DAYS_MAX = 24 * 60 * 60 * 7; // 7 days in seconds
exports.PRESIGN_EXPIRY_DAYS_MAX = PRESIGN_EXPIRY_DAYS_MAX;
class CopySourceOptions {
  constructor({
    Bucket,
    Object,
    VersionID = '',
    MatchETag = '',
    NoMatchETag = '',
    MatchModifiedSince = null,
    MatchUnmodifiedSince = null,
    MatchRange = false,
    Start = 0,
    End = 0,
    Encryption = undefined
  }) {
    this.Bucket = Bucket;
    this.Object = Object;
    this.VersionID = VersionID;
    this.MatchETag = MatchETag;
    this.NoMatchETag = NoMatchETag;
    this.MatchModifiedSince = MatchModifiedSince;
    this.MatchUnmodifiedSince = MatchUnmodifiedSince;
    this.MatchRange = MatchRange;
    this.Start = Start;
    this.End = End;
    this.Encryption = Encryption;
  }
  validate() {
    if (!(0, _helper.isValidBucketName)(this.Bucket)) {
      throw new errors.InvalidBucketNameError('Invalid Source bucket name: ' + this.Bucket);
    }
    if (!(0, _helper.isValidObjectName)(this.Object)) {
      throw new errors.InvalidObjectNameError(`Invalid Source object name: ${this.Object}`);
    }
    if (this.MatchRange && this.Start !== -1 && this.End !== -1 && this.Start > this.End || this.Start < 0) {
      throw new errors.InvalidObjectNameError('Source start must be non-negative, and start must be at most end.');
    } else if (this.MatchRange && !(0, _helper.isNumber)(this.Start) || !(0, _helper.isNumber)(this.End)) {
      throw new errors.InvalidObjectNameError('MatchRange is specified. But Invalid Start and End values are specified.');
    }
    return true;
  }
  getHeaders() {
    const headerOptions = {};
    headerOptions['x-amz-copy-source'] = encodeURI(this.Bucket + '/' + this.Object);
    if (!(0, _helper.isEmpty)(this.VersionID)) {
      headerOptions['x-amz-copy-source'] = `${encodeURI(this.Bucket + '/' + this.Object)}?versionId=${this.VersionID}`;
    }
    if (!(0, _helper.isEmpty)(this.MatchETag)) {
      headerOptions['x-amz-copy-source-if-match'] = this.MatchETag;
    }
    if (!(0, _helper.isEmpty)(this.NoMatchETag)) {
      headerOptions['x-amz-copy-source-if-none-match'] = this.NoMatchETag;
    }
    if (!(0, _helper.isEmpty)(this.MatchModifiedSince)) {
      headerOptions['x-amz-copy-source-if-modified-since'] = this.MatchModifiedSince;
    }
    if (!(0, _helper.isEmpty)(this.MatchUnmodifiedSince)) {
      headerOptions['x-amz-copy-source-if-unmodified-since'] = this.MatchUnmodifiedSince;
    }
    return headerOptions;
  }
}

/**
 * @deprecated use nodejs fs module
 */
exports.CopySourceOptions = CopySourceOptions;
function removeDirAndFiles(dirPath, removeSelf = true) {
  if (removeSelf) {
    return fs.rmSync(dirPath, {
      recursive: true,
      force: true
    });
  }
  fs.readdirSync(dirPath).forEach(item => {
    fs.rmSync(path.join(dirPath, item), {
      recursive: true,
      force: true
    });
  });
}
class CopyDestinationOptions {
  constructor({
    Bucket,
    Object,
    Encryption,
    UserMetadata,
    UserTags,
    LegalHold,
    RetainUntilDate,
    Mode,
    MetadataDirective
  }) {
    this.Bucket = Bucket;
    this.Object = Object;
    this.Encryption = Encryption ?? undefined; // null input will become undefined, easy for runtime assert
    this.UserMetadata = UserMetadata;
    this.UserTags = UserTags;
    this.LegalHold = LegalHold;
    this.Mode = Mode; // retention mode
    this.RetainUntilDate = RetainUntilDate;
    this.MetadataDirective = MetadataDirective;
  }
  getHeaders() {
    const replaceDirective = 'REPLACE';
    const headerOptions = {};
    const userTags = this.UserTags;
    if (!(0, _helper.isEmpty)(userTags)) {
      headerOptions['X-Amz-Tagging-Directive'] = replaceDirective;
      headerOptions['X-Amz-Tagging'] = (0, _helper.isObject)(userTags) ? querystring.stringify(userTags) : (0, _helper.isString)(userTags) ? userTags : '';
    }
    if (this.Mode) {
      headerOptions['X-Amz-Object-Lock-Mode'] = this.Mode; // GOVERNANCE or COMPLIANCE
    }

    if (this.RetainUntilDate) {
      headerOptions['X-Amz-Object-Lock-Retain-Until-Date'] = this.RetainUntilDate; // needs to be UTC.
    }

    if (this.LegalHold) {
      headerOptions['X-Amz-Object-Lock-Legal-Hold'] = this.LegalHold; // ON or OFF
    }

    if (this.UserMetadata) {
      for (const [key, value] of Object.entries(this.UserMetadata)) {
        headerOptions[`X-Amz-Meta-${key}`] = value.toString();
      }
    }
    if (this.MetadataDirective) {
      headerOptions[`X-Amz-Metadata-Directive`] = this.MetadataDirective;
    }
    if (this.Encryption) {
      const encryptionHeaders = (0, _helper.getEncryptionHeaders)(this.Encryption);
      for (const [key, value] of Object.entries(encryptionHeaders)) {
        headerOptions[key] = value;
      }
    }
    return headerOptions;
  }
  validate() {
    if (!(0, _helper.isValidBucketName)(this.Bucket)) {
      throw new errors.InvalidBucketNameError('Invalid Destination bucket name: ' + this.Bucket);
    }
    if (!(0, _helper.isValidObjectName)(this.Object)) {
      throw new errors.InvalidObjectNameError(`Invalid Destination object name: ${this.Object}`);
    }
    if (!(0, _helper.isEmpty)(this.UserMetadata) && !(0, _helper.isObject)(this.UserMetadata)) {
      throw new errors.InvalidObjectNameError(`Destination UserMetadata should be an object with key value pairs`);
    }
    if (!(0, _helper.isEmpty)(this.Mode) && ![_type.RETENTION_MODES.GOVERNANCE, _type.RETENTION_MODES.COMPLIANCE].includes(this.Mode)) {
      throw new errors.InvalidObjectNameError(`Invalid Mode specified for destination object it should be one of [GOVERNANCE,COMPLIANCE]`);
    }
    if (this.Encryption !== undefined && (0, _helper.isEmptyObject)(this.Encryption)) {
      throw new errors.InvalidObjectNameError(`Invalid Encryption configuration for destination object `);
    }
    return true;
  }
}

/**
 * maybe this should be a generic type for Records, leave it for later refactor
 */
exports.CopyDestinationOptions = CopyDestinationOptions;
class SelectResults {
  constructor({
    records,
    // parsed data as stream
    response,
    // original response stream
    stats,
    // stats as xml
    progress // stats as xml
  }) {
    this.records = records;
    this.response = response;
    this.stats = stats;
    this.progress = progress;
  }
  setStats(stats) {
    this.stats = stats;
  }
  getStats() {
    return this.stats;
  }
  setProgress(progress) {
    this.progress = progress;
  }
  getProgress() {
    return this.progress;
  }
  setResponse(response) {
    this.response = response;
  }
  getResponse() {
    return this.response;
  }
  setRecords(records) {
    this.records = records;
  }
  getRecords() {
    return this.records;
  }
}
exports.SelectResults = SelectResults;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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