"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseBucketNotification = parseBucketNotification;
exports.parseListObjects = parseListObjects;
exports.parseListObjectsV2 = parseListObjectsV2;
exports.parseListObjectsV2WithMetadata = parseListObjectsV2WithMetadata;
var _fastXmlParser = require("fast-xml-parser");
var errors = _interopRequireWildcard(require("./errors.js"), true);
var _helper = require("./internal/helper.js");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * MinIO Javascript Library for Amazon S3 Compatible Cloud Storage, (C) 2015 MinIO, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

const fxpWithoutNumParser = new _fastXmlParser.XMLParser({
  numberParseOptions: {
    skipLike: /./
  }
});

// parse XML response for bucket notification
function parseBucketNotification(xml) {
  var result = {
    TopicConfiguration: [],
    QueueConfiguration: [],
    CloudFunctionConfiguration: []
  };
  // Parse the events list
  var genEvents = function (events) {
    var result = [];
    if (events) {
      (0, _helper.toArray)(events).forEach(s3event => {
        result.push(s3event);
      });
    }
    return result;
  };
  // Parse all filter rules
  var genFilterRules = function (filters) {
    var result = [];
    if (filters) {
      filters = (0, _helper.toArray)(filters);
      if (filters[0].S3Key) {
        filters[0].S3Key = (0, _helper.toArray)(filters[0].S3Key);
        if (filters[0].S3Key[0].FilterRule) {
          (0, _helper.toArray)(filters[0].S3Key[0].FilterRule).forEach(rule => {
            var Name = (0, _helper.toArray)(rule.Name)[0];
            var Value = (0, _helper.toArray)(rule.Value)[0];
            result.push({
              Name,
              Value
            });
          });
        }
      }
    }
    return result;
  };
  var xmlobj = (0, _helper.parseXml)(xml);
  xmlobj = xmlobj.NotificationConfiguration;

  // Parse all topic configurations in the xml
  if (xmlobj.TopicConfiguration) {
    (0, _helper.toArray)(xmlobj.TopicConfiguration).forEach(config => {
      var Id = (0, _helper.toArray)(config.Id)[0];
      var Topic = (0, _helper.toArray)(config.Topic)[0];
      var Event = genEvents(config.Event);
      var Filter = genFilterRules(config.Filter);
      result.TopicConfiguration.push({
        Id,
        Topic,
        Event,
        Filter
      });
    });
  }
  // Parse all topic configurations in the xml
  if (xmlobj.QueueConfiguration) {
    (0, _helper.toArray)(xmlobj.QueueConfiguration).forEach(config => {
      var Id = (0, _helper.toArray)(config.Id)[0];
      var Queue = (0, _helper.toArray)(config.Queue)[0];
      var Event = genEvents(config.Event);
      var Filter = genFilterRules(config.Filter);
      result.QueueConfiguration.push({
        Id,
        Queue,
        Event,
        Filter
      });
    });
  }
  // Parse all QueueConfiguration arrays
  if (xmlobj.CloudFunctionConfiguration) {
    (0, _helper.toArray)(xmlobj.CloudFunctionConfiguration).forEach(config => {
      var Id = (0, _helper.toArray)(config.Id)[0];
      var CloudFunction = (0, _helper.toArray)(config.CloudFunction)[0];
      var Event = genEvents(config.Event);
      var Filter = genFilterRules(config.Filter);
      result.CloudFunctionConfiguration.push({
        Id,
        CloudFunction,
        Event,
        Filter
      });
    });
  }
  return result;
}
const formatObjInfo = (content, opts = {}) => {
  let {
    Key,
    LastModified,
    ETag,
    Size,
    VersionId,
    IsLatest
  } = content;
  if (!(0, _helper.isObject)(opts)) {
    opts = {};
  }
  const name = (0, _helper.sanitizeObjectKey)((0, _helper.toArray)(Key)[0]);
  const lastModified = new Date((0, _helper.toArray)(LastModified)[0]);
  const etag = (0, _helper.sanitizeETag)((0, _helper.toArray)(ETag)[0]);
  const size = (0, _helper.sanitizeSize)(Size);
  return {
    name,
    lastModified,
    etag,
    size,
    versionId: VersionId,
    isLatest: IsLatest,
    isDeleteMarker: opts.IsDeleteMarker ? opts.IsDeleteMarker : false
  };
};

// parse XML response for list objects in a bucket
function parseListObjects(xml) {
  var result = {
    objects: [],
    isTruncated: false
  };
  let isTruncated = false;
  let nextMarker, nextVersionKeyMarker;
  const xmlobj = fxpWithoutNumParser.parse(xml);
  const parseCommonPrefixesEntity = responseEntity => {
    if (responseEntity) {
      (0, _helper.toArray)(responseEntity).forEach(commonPrefix => {
        result.objects.push({
          prefix: (0, _helper.sanitizeObjectKey)((0, _helper.toArray)(commonPrefix.Prefix)[0]),
          size: 0
        });
      });
    }
  };
  const listBucketResult = xmlobj.ListBucketResult;
  const listVersionsResult = xmlobj.ListVersionsResult;
  if (listBucketResult) {
    if (listBucketResult.IsTruncated) {
      isTruncated = listBucketResult.IsTruncated;
    }
    if (listBucketResult.Contents) {
      (0, _helper.toArray)(listBucketResult.Contents).forEach(content => {
        const name = (0, _helper.sanitizeObjectKey)((0, _helper.toArray)(content.Key)[0]);
        const lastModified = new Date((0, _helper.toArray)(content.LastModified)[0]);
        const etag = (0, _helper.sanitizeETag)((0, _helper.toArray)(content.ETag)[0]);
        const size = (0, _helper.sanitizeSize)(content.Size);
        result.objects.push({
          name,
          lastModified,
          etag,
          size
        });
      });
    }
    if (listBucketResult.NextMarker) {
      nextMarker = listBucketResult.NextMarker;
    } else if (isTruncated && result.objects.length > 0) {
      nextMarker = result.objects[result.objects.length - 1].name;
    }
    parseCommonPrefixesEntity(listBucketResult.CommonPrefixes);
  }
  if (listVersionsResult) {
    if (listVersionsResult.IsTruncated) {
      isTruncated = listVersionsResult.IsTruncated;
    }
    if (listVersionsResult.Version) {
      (0, _helper.toArray)(listVersionsResult.Version).forEach(content => {
        result.objects.push(formatObjInfo(content));
      });
    }
    if (listVersionsResult.DeleteMarker) {
      (0, _helper.toArray)(listVersionsResult.DeleteMarker).forEach(content => {
        result.objects.push(formatObjInfo(content, {
          IsDeleteMarker: true
        }));
      });
    }
    if (listVersionsResult.NextKeyMarker) {
      nextVersionKeyMarker = listVersionsResult.NextKeyMarker;
    }
    if (listVersionsResult.NextVersionIdMarker) {
      result.versionIdMarker = listVersionsResult.NextVersionIdMarker;
    }
    parseCommonPrefixesEntity(listVersionsResult.CommonPrefixes);
  }
  result.isTruncated = isTruncated;
  if (isTruncated) {
    result.nextMarker = nextVersionKeyMarker || nextMarker;
  }
  return result;
}

// parse XML response for list objects v2 in a bucket
function parseListObjectsV2(xml) {
  var result = {
    objects: [],
    isTruncated: false
  };
  var xmlobj = (0, _helper.parseXml)(xml);
  if (!xmlobj.ListBucketResult) {
    throw new errors.InvalidXMLError('Missing tag: "ListBucketResult"');
  }
  xmlobj = xmlobj.ListBucketResult;
  if (xmlobj.IsTruncated) {
    result.isTruncated = xmlobj.IsTruncated;
  }
  if (xmlobj.NextContinuationToken) {
    result.nextContinuationToken = xmlobj.NextContinuationToken;
  }
  if (xmlobj.Contents) {
    (0, _helper.toArray)(xmlobj.Contents).forEach(content => {
      var name = (0, _helper.sanitizeObjectKey)((0, _helper.toArray)(content.Key)[0]);
      var lastModified = new Date(content.LastModified);
      var etag = (0, _helper.sanitizeETag)(content.ETag);
      var size = content.Size;
      result.objects.push({
        name,
        lastModified,
        etag,
        size
      });
    });
  }
  if (xmlobj.CommonPrefixes) {
    (0, _helper.toArray)(xmlobj.CommonPrefixes).forEach(commonPrefix => {
      result.objects.push({
        prefix: (0, _helper.sanitizeObjectKey)((0, _helper.toArray)(commonPrefix.Prefix)[0]),
        size: 0
      });
    });
  }
  return result;
}

// parse XML response for list objects v2 with metadata in a bucket
function parseListObjectsV2WithMetadata(xml) {
  var result = {
    objects: [],
    isTruncated: false
  };
  var xmlobj = (0, _helper.parseXml)(xml);
  if (!xmlobj.ListBucketResult) {
    throw new errors.InvalidXMLError('Missing tag: "ListBucketResult"');
  }
  xmlobj = xmlobj.ListBucketResult;
  if (xmlobj.IsTruncated) {
    result.isTruncated = xmlobj.IsTruncated;
  }
  if (xmlobj.NextContinuationToken) {
    result.nextContinuationToken = xmlobj.NextContinuationToken;
  }
  if (xmlobj.Contents) {
    (0, _helper.toArray)(xmlobj.Contents).forEach(content => {
      var name = (0, _helper.sanitizeObjectKey)(content.Key);
      var lastModified = new Date(content.LastModified);
      var etag = (0, _helper.sanitizeETag)(content.ETag);
      var size = content.Size;
      var metadata;
      if (content.UserMetadata != null) {
        metadata = (0, _helper.toArray)(content.UserMetadata)[0];
      } else {
        metadata = null;
      }
      result.objects.push({
        name,
        lastModified,
        etag,
        size,
        metadata
      });
    });
  }
  if (xmlobj.CommonPrefixes) {
    (0, _helper.toArray)(xmlobj.CommonPrefixes).forEach(commonPrefix => {
      result.objects.push({
        prefix: (0, _helper.sanitizeObjectKey)((0, _helper.toArray)(commonPrefix.Prefix)[0]),
        size: 0
      });
    });
  }
  return result;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfZmFzdFhtbFBhcnNlciIsInJlcXVpcmUiLCJlcnJvcnMiLCJfaW50ZXJvcFJlcXVpcmVXaWxkY2FyZCIsIl9oZWxwZXIiLCJfZ2V0UmVxdWlyZVdpbGRjYXJkQ2FjaGUiLCJub2RlSW50ZXJvcCIsIldlYWtNYXAiLCJjYWNoZUJhYmVsSW50ZXJvcCIsImNhY2hlTm9kZUludGVyb3AiLCJvYmoiLCJfX2VzTW9kdWxlIiwiZGVmYXVsdCIsImNhY2hlIiwiaGFzIiwiZ2V0IiwibmV3T2JqIiwiaGFzUHJvcGVydHlEZXNjcmlwdG9yIiwiT2JqZWN0IiwiZGVmaW5lUHJvcGVydHkiLCJnZXRPd25Qcm9wZXJ0eURlc2NyaXB0b3IiLCJrZXkiLCJwcm90b3R5cGUiLCJoYXNPd25Qcm9wZXJ0eSIsImNhbGwiLCJkZXNjIiwic2V0IiwiZnhwV2l0aG91dE51bVBhcnNlciIsIlhNTFBhcnNlciIsIm51bWJlclBhcnNlT3B0aW9ucyIsInNraXBMaWtlIiwicGFyc2VCdWNrZXROb3RpZmljYXRpb24iLCJ4bWwiLCJyZXN1bHQiLCJUb3BpY0NvbmZpZ3VyYXRpb24iLCJRdWV1ZUNvbmZpZ3VyYXRpb24iLCJDbG91ZEZ1bmN0aW9uQ29uZmlndXJhdGlvbiIsImdlbkV2ZW50cyIsImV2ZW50cyIsInRvQXJyYXkiLCJmb3JFYWNoIiwiczNldmVudCIsInB1c2giLCJnZW5GaWx0ZXJSdWxlcyIsImZpbHRlcnMiLCJTM0tleSIsIkZpbHRlclJ1bGUiLCJydWxlIiwiTmFtZSIsIlZhbHVlIiwieG1sb2JqIiwicGFyc2VYbWwiLCJOb3RpZmljYXRpb25Db25maWd1cmF0aW9uIiwiY29uZmlnIiwiSWQiLCJUb3BpYyIsIkV2ZW50IiwiRmlsdGVyIiwiUXVldWUiLCJDbG91ZEZ1bmN0aW9uIiwiZm9ybWF0T2JqSW5mbyIsImNvbnRlbnQiLCJvcHRzIiwiS2V5IiwiTGFzdE1vZGlmaWVkIiwiRVRhZyIsIlNpemUiLCJWZXJzaW9uSWQiLCJJc0xhdGVzdCIsImlzT2JqZWN0IiwibmFtZSIsInNhbml0aXplT2JqZWN0S2V5IiwibGFzdE1vZGlmaWVkIiwiRGF0ZSIsImV0YWciLCJzYW5pdGl6ZUVUYWciLCJzaXplIiwic2FuaXRpemVTaXplIiwidmVyc2lvbklkIiwiaXNMYXRlc3QiLCJpc0RlbGV0ZU1hcmtlciIsIklzRGVsZXRlTWFya2VyIiwicGFyc2VMaXN0T2JqZWN0cyIsIm9iamVjdHMiLCJpc1RydW5jYXRlZCIsIm5leHRNYXJrZXIiLCJuZXh0VmVyc2lvbktleU1hcmtlciIsInBhcnNlIiwicGFyc2VDb21tb25QcmVmaXhlc0VudGl0eSIsInJlc3BvbnNlRW50aXR5IiwiY29tbW9uUHJlZml4IiwicHJlZml4IiwiUHJlZml4IiwibGlzdEJ1Y2tldFJlc3VsdCIsIkxpc3RCdWNrZXRSZXN1bHQiLCJsaXN0VmVyc2lvbnNSZXN1bHQiLCJMaXN0VmVyc2lvbnNSZXN1bHQiLCJJc1RydW5jYXRlZCIsIkNvbnRlbnRzIiwiTmV4dE1hcmtlciIsImxlbmd0aCIsIkNvbW1vblByZWZpeGVzIiwiVmVyc2lvbiIsIkRlbGV0ZU1hcmtlciIsIk5leHRLZXlNYXJrZXIiLCJOZXh0VmVyc2lvbklkTWFya2VyIiwidmVyc2lvbklkTWFya2VyIiwicGFyc2VMaXN0T2JqZWN0c1YyIiwiSW52YWxpZFhNTEVycm9yIiwiTmV4dENvbnRpbnVhdGlvblRva2VuIiwibmV4dENvbnRpbnVhdGlvblRva2VuIiwicGFyc2VMaXN0T2JqZWN0c1YyV2l0aE1ldGFkYXRhIiwibWV0YWRhdGEiLCJVc2VyTWV0YWRhdGEiXSwic291cmNlcyI6WyJ4bWwtcGFyc2Vycy5qcyJdLCJzb3VyY2VzQ29udGVudCI6WyIvKlxuICogTWluSU8gSmF2YXNjcmlwdCBMaWJyYXJ5IGZvciBBbWF6b24gUzMgQ29tcGF0aWJsZSBDbG91ZCBTdG9yYWdlLCAoQykgMjAxNSBNaW5JTywgSW5jLlxuICpcbiAqIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIik7XG4gKiB5b3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlIHdpdGggdGhlIExpY2Vuc2UuXG4gKiBZb3UgbWF5IG9idGFpbiBhIGNvcHkgb2YgdGhlIExpY2Vuc2UgYXRcbiAqXG4gKiAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogVW5sZXNzIHJlcXVpcmVkIGJ5IGFwcGxpY2FibGUgbGF3IG9yIGFncmVlZCB0byBpbiB3cml0aW5nLCBzb2Z0d2FyZVxuICogZGlzdHJpYnV0ZWQgdW5kZXIgdGhlIExpY2Vuc2UgaXMgZGlzdHJpYnV0ZWQgb24gYW4gXCJBUyBJU1wiIEJBU0lTLFxuICogV0lUSE9VVCBXQVJSQU5USUVTIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGVpdGhlciBleHByZXNzIG9yIGltcGxpZWQuXG4gKiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnMgYW5kXG4gKiBsaW1pdGF0aW9ucyB1bmRlciB0aGUgTGljZW5zZS5cbiAqL1xuXG5pbXBvcnQgeyBYTUxQYXJzZXIgfSBmcm9tICdmYXN0LXhtbC1wYXJzZXInXG5cbmltcG9ydCAqIGFzIGVycm9ycyBmcm9tICcuL2Vycm9ycy50cydcbmltcG9ydCB7IGlzT2JqZWN0LCBwYXJzZVhtbCwgc2FuaXRpemVFVGFnLCBzYW5pdGl6ZU9iamVjdEtleSwgc2FuaXRpemVTaXplLCB0b0FycmF5IH0gZnJvbSAnLi9pbnRlcm5hbC9oZWxwZXIudHMnXG5cbmNvbnN0IGZ4cFdpdGhvdXROdW1QYXJzZXIgPSBuZXcgWE1MUGFyc2VyKHtcbiAgbnVtYmVyUGFyc2VPcHRpb25zOiB7XG4gICAgc2tpcExpa2U6IC8uLyxcbiAgfSxcbn0pXG5cbi8vIHBhcnNlIFhNTCByZXNwb25zZSBmb3IgYnVja2V0IG5vdGlmaWNhdGlvblxuZXhwb3J0IGZ1bmN0aW9uIHBhcnNlQnVja2V0Tm90aWZpY2F0aW9uKHhtbCkge1xuICB2YXIgcmVzdWx0ID0ge1xuICAgIFRvcGljQ29uZmlndXJhdGlvbjogW10sXG4gICAgUXVldWVDb25maWd1cmF0aW9uOiBbXSxcbiAgICBDbG91ZEZ1bmN0aW9uQ29uZmlndXJhdGlvbjogW10sXG4gIH1cbiAgLy8gUGFyc2UgdGhlIGV2ZW50cyBsaXN0XG4gIHZhciBnZW5FdmVudHMgPSBmdW5jdGlvbiAoZXZlbnRzKSB7XG4gICAgdmFyIHJlc3VsdCA9IFtdXG4gICAgaWYgKGV2ZW50cykge1xuICAgICAgdG9BcnJheShldmVudHMpLmZvckVhY2goKHMzZXZlbnQpID0+IHtcbiAgICAgICAgcmVzdWx0LnB1c2goczNldmVudClcbiAgICAgIH0pXG4gICAgfVxuICAgIHJldHVybiByZXN1bHRcbiAgfVxuICAvLyBQYXJzZSBhbGwgZmlsdGVyIHJ1bGVzXG4gIHZhciBnZW5GaWx0ZXJSdWxlcyA9IGZ1bmN0aW9uIChmaWx0ZXJzKSB7XG4gICAgdmFyIHJlc3VsdCA9IFtdXG4gICAgaWYgKGZpbHRlcnMpIHtcbiAgICAgIGZpbHRlcnMgPSB0b0FycmF5KGZpbHRlcnMpXG4gICAgICBpZiAoZmlsdGVyc1swXS5TM0tleSkge1xuICAgICAgICBmaWx0ZXJzWzBdLlMzS2V5ID0gdG9BcnJheShmaWx0ZXJzWzBdLlMzS2V5KVxuICAgICAgICBpZiAoZmlsdGVyc1swXS5TM0tleVswXS5GaWx0ZXJSdWxlKSB7XG4gICAgICAgICAgdG9BcnJheShmaWx0ZXJzWzBdLlMzS2V5WzBdLkZpbHRlclJ1bGUpLmZvckVhY2goKHJ1bGUpID0+IHtcbiAgICAgICAgICAgIHZhciBOYW1lID0gdG9BcnJheShydWxlLk5hbWUpWzBdXG4gICAgICAgICAgICB2YXIgVmFsdWUgPSB0b0FycmF5KHJ1bGUuVmFsdWUpWzBdXG4gICAgICAgICAgICByZXN1bHQucHVzaCh7IE5hbWUsIFZhbHVlIH0pXG4gICAgICAgICAgfSlcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgICByZXR1cm4gcmVzdWx0XG4gIH1cblxuICB2YXIgeG1sb2JqID0gcGFyc2VYbWwoeG1sKVxuICB4bWxvYmogPSB4bWxvYmouTm90aWZpY2F0aW9uQ29uZmlndXJhdGlvblxuXG4gIC8vIFBhcnNlIGFsbCB0b3BpYyBjb25maWd1cmF0aW9ucyBpbiB0aGUgeG1sXG4gIGlmICh4bWxvYmouVG9waWNDb25maWd1cmF0aW9uKSB7XG4gICAgdG9BcnJheSh4bWxvYmouVG9waWNDb25maWd1cmF0aW9uKS5mb3JFYWNoKChjb25maWcpID0+IHtcbiAgICAgIHZhciBJZCA9IHRvQXJyYXkoY29uZmlnLklkKVswXVxuICAgICAgdmFyIFRvcGljID0gdG9BcnJheShjb25maWcuVG9waWMpWzBdXG4gICAgICB2YXIgRXZlbnQgPSBnZW5FdmVudHMoY29uZmlnLkV2ZW50KVxuICAgICAgdmFyIEZpbHRlciA9IGdlbkZpbHRlclJ1bGVzKGNvbmZpZy5GaWx0ZXIpXG4gICAgICByZXN1bHQuVG9waWNDb25maWd1cmF0aW9uLnB1c2goeyBJZCwgVG9waWMsIEV2ZW50LCBGaWx0ZXIgfSlcbiAgICB9KVxuICB9XG4gIC8vIFBhcnNlIGFsbCB0b3BpYyBjb25maWd1cmF0aW9ucyBpbiB0aGUgeG1sXG4gIGlmICh4bWxvYmouUXVldWVDb25maWd1cmF0aW9uKSB7XG4gICAgdG9BcnJheSh4bWxvYmouUXVldWVDb25maWd1cmF0aW9uKS5mb3JFYWNoKChjb25maWcpID0+IHtcbiAgICAgIHZhciBJZCA9IHRvQXJyYXkoY29uZmlnLklkKVswXVxuICAgICAgdmFyIFF1ZXVlID0gdG9BcnJheShjb25maWcuUXVldWUpWzBdXG4gICAgICB2YXIgRXZlbnQgPSBnZW5FdmVudHMoY29uZmlnLkV2ZW50KVxuICAgICAgdmFyIEZpbHRlciA9IGdlbkZpbHRlclJ1bGVzKGNvbmZpZy5GaWx0ZXIpXG4gICAgICByZXN1bHQuUXVldWVDb25maWd1cmF0aW9uLnB1c2goeyBJZCwgUXVldWUsIEV2ZW50LCBGaWx0ZXIgfSlcbiAgICB9KVxuICB9XG4gIC8vIFBhcnNlIGFsbCBRdWV1ZUNvbmZpZ3VyYXRpb24gYXJyYXlzXG4gIGlmICh4bWxvYmouQ2xvdWRGdW5jdGlvbkNvbmZpZ3VyYXRpb24pIHtcbiAgICB0b0FycmF5KHhtbG9iai5DbG91ZEZ1bmN0aW9uQ29uZmlndXJhdGlvbikuZm9yRWFjaCgoY29uZmlnKSA9PiB7XG4gICAgICB2YXIgSWQgPSB0b0FycmF5KGNvbmZpZy5JZClbMF1cbiAgICAgIHZhciBDbG91ZEZ1bmN0aW9uID0gdG9BcnJheShjb25maWcuQ2xvdWRGdW5jdGlvbilbMF1cbiAgICAgIHZhciBFdmVudCA9IGdlbkV2ZW50cyhjb25maWcuRXZlbnQpXG4gICAgICB2YXIgRmlsdGVyID0gZ2VuRmlsdGVyUnVsZXMoY29uZmlnLkZpbHRlcilcbiAgICAgIHJlc3VsdC5DbG91ZEZ1bmN0aW9uQ29uZmlndXJhdGlvbi5wdXNoKHsgSWQsIENsb3VkRnVuY3Rpb24sIEV2ZW50LCBGaWx0ZXIgfSlcbiAgICB9KVxuICB9XG5cbiAgcmV0dXJuIHJlc3VsdFxufVxuXG5jb25zdCBmb3JtYXRPYmpJbmZvID0gKGNvbnRlbnQsIG9wdHMgPSB7fSkgPT4ge1xuICBsZXQgeyBLZXksIExhc3RNb2RpZmllZCwgRVRhZywgU2l6ZSwgVmVyc2lvbklkLCBJc0xhdGVzdCB9ID0gY29udGVudFxuXG4gIGlmICghaXNPYmplY3Qob3B0cykpIHtcbiAgICBvcHRzID0ge31cbiAgfVxuXG4gIGNvbnN0IG5hbWUgPSBzYW5pdGl6ZU9iamVjdEtleSh0b0FycmF5KEtleSlbMF0pXG4gIGNvbnN0IGxhc3RNb2RpZmllZCA9IG5ldyBEYXRlKHRvQXJyYXkoTGFzdE1vZGlmaWVkKVswXSlcbiAgY29uc3QgZXRhZyA9IHNhbml0aXplRVRhZyh0b0FycmF5KEVUYWcpWzBdKVxuICBjb25zdCBzaXplID0gc2FuaXRpemVTaXplKFNpemUpXG5cbiAgcmV0dXJuIHtcbiAgICBuYW1lLFxuICAgIGxhc3RNb2RpZmllZCxcbiAgICBldGFnLFxuICAgIHNpemUsXG4gICAgdmVyc2lvbklkOiBWZXJzaW9uSWQsXG4gICAgaXNMYXRlc3Q6IElzTGF0ZXN0LFxuICAgIGlzRGVsZXRlTWFya2VyOiBvcHRzLklzRGVsZXRlTWFya2VyID8gb3B0cy5Jc0RlbGV0ZU1hcmtlciA6IGZhbHNlLFxuICB9XG59XG5cbi8vIHBhcnNlIFhNTCByZXNwb25zZSBmb3IgbGlzdCBvYmplY3RzIGluIGEgYnVja2V0XG5leHBvcnQgZnVuY3Rpb24gcGFyc2VMaXN0T2JqZWN0cyh4bWwpIHtcbiAgdmFyIHJlc3VsdCA9IHtcbiAgICBvYmplY3RzOiBbXSxcbiAgICBpc1RydW5jYXRlZDogZmFsc2UsXG4gIH1cbiAgbGV0IGlzVHJ1bmNhdGVkID0gZmFsc2VcbiAgbGV0IG5leHRNYXJrZXIsIG5leHRWZXJzaW9uS2V5TWFya2VyXG4gIGNvbnN0IHhtbG9iaiA9IGZ4cFdpdGhvdXROdW1QYXJzZXIucGFyc2UoeG1sKVxuXG4gIGNvbnN0IHBhcnNlQ29tbW9uUHJlZml4ZXNFbnRpdHkgPSAocmVzcG9uc2VFbnRpdHkpID0+IHtcbiAgICBpZiAocmVzcG9uc2VFbnRpdHkpIHtcbiAgICAgIHRvQXJyYXkocmVzcG9uc2VFbnRpdHkpLmZvckVhY2goKGNvbW1vblByZWZpeCkgPT4ge1xuICAgICAgICByZXN1bHQub2JqZWN0cy5wdXNoKHsgcHJlZml4OiBzYW5pdGl6ZU9iamVjdEtleSh0b0FycmF5KGNvbW1vblByZWZpeC5QcmVmaXgpWzBdKSwgc2l6ZTogMCB9KVxuICAgICAgfSlcbiAgICB9XG4gIH1cblxuICBjb25zdCBsaXN0QnVja2V0UmVzdWx0ID0geG1sb2JqLkxpc3RCdWNrZXRSZXN1bHRcbiAgY29uc3QgbGlzdFZlcnNpb25zUmVzdWx0ID0geG1sb2JqLkxpc3RWZXJzaW9uc1Jlc3VsdFxuXG4gIGlmIChsaXN0QnVja2V0UmVzdWx0KSB7XG4gICAgaWYgKGxpc3RCdWNrZXRSZXN1bHQuSXNUcnVuY2F0ZWQpIHtcbiAgICAgIGlzVHJ1bmNhdGVkID0gbGlzdEJ1Y2tldFJlc3VsdC5Jc1RydW5jYXRlZFxuICAgIH1cbiAgICBpZiAobGlzdEJ1Y2tldFJlc3VsdC5Db250ZW50cykge1xuICAgICAgdG9BcnJheShsaXN0QnVja2V0UmVzdWx0LkNvbnRlbnRzKS5mb3JFYWNoKChjb250ZW50KSA9PiB7XG4gICAgICAgIGNvbnN0IG5hbWUgPSBzYW5pdGl6ZU9iamVjdEtleSh0b0FycmF5KGNvbnRlbnQuS2V5KVswXSlcbiAgICAgICAgY29uc3QgbGFzdE1vZGlmaWVkID0gbmV3IERhdGUodG9BcnJheShjb250ZW50Lkxhc3RNb2RpZmllZClbMF0pXG4gICAgICAgIGNvbnN0IGV0YWcgPSBzYW5pdGl6ZUVUYWcodG9BcnJheShjb250ZW50LkVUYWcpWzBdKVxuICAgICAgICBjb25zdCBzaXplID0gc2FuaXRpemVTaXplKGNvbnRlbnQuU2l6ZSlcbiAgICAgICAgcmVzdWx0Lm9iamVjdHMucHVzaCh7IG5hbWUsIGxhc3RNb2RpZmllZCwgZXRhZywgc2l6ZSB9KVxuICAgICAgfSlcbiAgICB9XG5cbiAgICBpZiAobGlzdEJ1Y2tldFJlc3VsdC5OZXh0TWFya2VyKSB7XG4gICAgICBuZXh0TWFya2VyID0gbGlzdEJ1Y2tldFJlc3VsdC5OZXh0TWFya2VyXG4gICAgfSBlbHNlIGlmIChpc1RydW5jYXRlZCAmJiByZXN1bHQub2JqZWN0cy5sZW5ndGggPiAwKSB7XG4gICAgICBuZXh0TWFya2VyID0gcmVzdWx0Lm9iamVjdHNbcmVzdWx0Lm9iamVjdHMubGVuZ3RoIC0gMV0ubmFtZVxuICAgIH1cbiAgICBwYXJzZUNvbW1vblByZWZpeGVzRW50aXR5KGxpc3RCdWNrZXRSZXN1bHQuQ29tbW9uUHJlZml4ZXMpXG4gIH1cblxuICBpZiAobGlzdFZlcnNpb25zUmVzdWx0KSB7XG4gICAgaWYgKGxpc3RWZXJzaW9uc1Jlc3VsdC5Jc1RydW5jYXRlZCkge1xuICAgICAgaXNUcnVuY2F0ZWQgPSBsaXN0VmVyc2lvbnNSZXN1bHQuSXNUcnVuY2F0ZWRcbiAgICB9XG5cbiAgICBpZiAobGlzdFZlcnNpb25zUmVzdWx0LlZlcnNpb24pIHtcbiAgICAgIHRvQXJyYXkobGlzdFZlcnNpb25zUmVzdWx0LlZlcnNpb24pLmZvckVhY2goKGNvbnRlbnQpID0+IHtcbiAgICAgICAgcmVzdWx0Lm9iamVjdHMucHVzaChmb3JtYXRPYmpJbmZvKGNvbnRlbnQpKVxuICAgICAgfSlcbiAgICB9XG4gICAgaWYgKGxpc3RWZXJzaW9uc1Jlc3VsdC5EZWxldGVNYXJrZXIpIHtcbiAgICAgIHRvQXJyYXkobGlzdFZlcnNpb25zUmVzdWx0LkRlbGV0ZU1hcmtlcikuZm9yRWFjaCgoY29udGVudCkgPT4ge1xuICAgICAgICByZXN1bHQub2JqZWN0cy5wdXNoKGZvcm1hdE9iakluZm8oY29udGVudCwgeyBJc0RlbGV0ZU1hcmtlcjogdHJ1ZSB9KSlcbiAgICAgIH0pXG4gICAgfVxuXG4gICAgaWYgKGxpc3RWZXJzaW9uc1Jlc3VsdC5OZXh0S2V5TWFya2VyKSB7XG4gICAgICBuZXh0VmVyc2lvbktleU1hcmtlciA9IGxpc3RWZXJzaW9uc1Jlc3VsdC5OZXh0S2V5TWFya2VyXG4gICAgfVxuICAgIGlmIChsaXN0VmVyc2lvbnNSZXN1bHQuTmV4dFZlcnNpb25JZE1hcmtlcikge1xuICAgICAgcmVzdWx0LnZlcnNpb25JZE1hcmtlciA9IGxpc3RWZXJzaW9uc1Jlc3VsdC5OZXh0VmVyc2lvbklkTWFya2VyXG4gICAgfVxuICAgIHBhcnNlQ29tbW9uUHJlZml4ZXNFbnRpdHkobGlzdFZlcnNpb25zUmVzdWx0LkNvbW1vblByZWZpeGVzKVxuICB9XG5cbiAgcmVzdWx0LmlzVHJ1bmNhdGVkID0gaXNUcnVuY2F0ZWRcbiAgaWYgKGlzVHJ1bmNhdGVkKSB7XG4gICAgcmVzdWx0Lm5leHRNYXJrZXIgPSBuZXh0VmVyc2lvbktleU1hcmtlciB8fCBuZXh0TWFya2VyXG4gIH1cbiAgcmV0dXJuIHJlc3VsdFxufVxuXG4vLyBwYXJzZSBYTUwgcmVzcG9uc2UgZm9yIGxpc3Qgb2JqZWN0cyB2MiBpbiBhIGJ1Y2tldFxuZXhwb3J0IGZ1bmN0aW9uIHBhcnNlTGlzdE9iamVjdHNWMih4bWwpIHtcbiAgdmFyIHJlc3VsdCA9IHtcbiAgICBvYmplY3RzOiBbXSxcbiAgICBpc1RydW5jYXRlZDogZmFsc2UsXG4gIH1cbiAgdmFyIHhtbG9iaiA9IHBhcnNlWG1sKHhtbClcbiAgaWYgKCF4bWxvYmouTGlzdEJ1Y2tldFJlc3VsdCkge1xuICAgIHRocm93IG5ldyBlcnJvcnMuSW52YWxpZFhNTEVycm9yKCdNaXNzaW5nIHRhZzogXCJMaXN0QnVja2V0UmVzdWx0XCInKVxuICB9XG4gIHhtbG9iaiA9IHhtbG9iai5MaXN0QnVja2V0UmVzdWx0XG4gIGlmICh4bWxvYmouSXNUcnVuY2F0ZWQpIHtcbiAgICByZXN1bHQuaXNUcnVuY2F0ZWQgPSB4bWxvYmouSXNUcnVuY2F0ZWRcbiAgfVxuICBpZiAoeG1sb2JqLk5leHRDb250aW51YXRpb25Ub2tlbikge1xuICAgIHJlc3VsdC5uZXh0Q29udGludWF0aW9uVG9rZW4gPSB4bWxvYmouTmV4dENvbnRpbnVhdGlvblRva2VuXG4gIH1cbiAgaWYgKHhtbG9iai5Db250ZW50cykge1xuICAgIHRvQXJyYXkoeG1sb2JqLkNvbnRlbnRzKS5mb3JFYWNoKChjb250ZW50KSA9PiB7XG4gICAgICB2YXIgbmFtZSA9IHNhbml0aXplT2JqZWN0S2V5KHRvQXJyYXkoY29udGVudC5LZXkpWzBdKVxuICAgICAgdmFyIGxhc3RNb2RpZmllZCA9IG5ldyBEYXRlKGNvbnRlbnQuTGFzdE1vZGlmaWVkKVxuICAgICAgdmFyIGV0YWcgPSBzYW5pdGl6ZUVUYWcoY29udGVudC5FVGFnKVxuICAgICAgdmFyIHNpemUgPSBjb250ZW50LlNpemVcbiAgICAgIHJlc3VsdC5vYmplY3RzLnB1c2goeyBuYW1lLCBsYXN0TW9kaWZpZWQsIGV0YWcsIHNpemUgfSlcbiAgICB9KVxuICB9XG4gIGlmICh4bWxvYmouQ29tbW9uUHJlZml4ZXMpIHtcbiAgICB0b0FycmF5KHhtbG9iai5Db21tb25QcmVmaXhlcykuZm9yRWFjaCgoY29tbW9uUHJlZml4KSA9PiB7XG4gICAgICByZXN1bHQub2JqZWN0cy5wdXNoKHsgcHJlZml4OiBzYW5pdGl6ZU9iamVjdEtleSh0b0FycmF5KGNvbW1vblByZWZpeC5QcmVmaXgpWzBdKSwgc2l6ZTogMCB9KVxuICAgIH0pXG4gIH1cbiAgcmV0dXJuIHJlc3VsdFxufVxuXG4vLyBwYXJzZSBYTUwgcmVzcG9uc2UgZm9yIGxpc3Qgb2JqZWN0cyB2MiB3aXRoIG1ldGFkYXRhIGluIGEgYnVja2V0XG5leHBvcnQgZnVuY3Rpb24gcGFyc2VMaXN0T2JqZWN0c1YyV2l0aE1ldGFkYXRhKHhtbCkge1xuICB2YXIgcmVzdWx0ID0ge1xuICAgIG9iamVjdHM6IFtdLFxuICAgIGlzVHJ1bmNhdGVkOiBmYWxzZSxcbiAgfVxuICB2YXIgeG1sb2JqID0gcGFyc2VYbWwoeG1sKVxuICBpZiAoIXhtbG9iai5MaXN0QnVja2V0UmVzdWx0KSB7XG4gICAgdGhyb3cgbmV3IGVycm9ycy5JbnZhbGlkWE1MRXJyb3IoJ01pc3NpbmcgdGFnOiBcIkxpc3RCdWNrZXRSZXN1bHRcIicpXG4gIH1cbiAgeG1sb2JqID0geG1sb2JqLkxpc3RCdWNrZXRSZXN1bHRcbiAgaWYgKHhtbG9iai5Jc1RydW5jYXRlZCkge1xuICAgIHJlc3VsdC5pc1RydW5jYXRlZCA9IHhtbG9iai5Jc1RydW5jYXRlZFxuICB9XG4gIGlmICh4bWxvYmouTmV4dENvbnRpbnVhdGlvblRva2VuKSB7XG4gICAgcmVzdWx0Lm5leHRDb250aW51YXRpb25Ub2tlbiA9IHhtbG9iai5OZXh0Q29udGludWF0aW9uVG9rZW5cbiAgfVxuXG4gIGlmICh4bWxvYmouQ29udGVudHMpIHtcbiAgICB0b0FycmF5KHhtbG9iai5Db250ZW50cykuZm9yRWFjaCgoY29udGVudCkgPT4ge1xuICAgICAgdmFyIG5hbWUgPSBzYW5pdGl6ZU9iamVjdEtleShjb250ZW50LktleSlcbiAgICAgIHZhciBsYXN0TW9kaWZpZWQgPSBuZXcgRGF0ZShjb250ZW50Lkxhc3RNb2RpZmllZClcbiAgICAgIHZhciBldGFnID0gc2FuaXRpemVFVGFnKGNvbnRlbnQuRVRhZylcbiAgICAgIHZhciBzaXplID0gY29udGVudC5TaXplXG4gICAgICB2YXIgbWV0YWRhdGFcbiAgICAgIGlmIChjb250ZW50LlVzZXJNZXRhZGF0YSAhPSBudWxsKSB7XG4gICAgICAgIG1ldGFkYXRhID0gdG9BcnJheShjb250ZW50LlVzZXJNZXRhZGF0YSlbMF1cbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIG1ldGFkYXRhID0gbnVsbFxuICAgICAgfVxuICAgICAgcmVzdWx0Lm9iamVjdHMucHVzaCh7IG5hbWUsIGxhc3RNb2RpZmllZCwgZXRhZywgc2l6ZSwgbWV0YWRhdGEgfSlcbiAgICB9KVxuICB9XG5cbiAgaWYgKHhtbG9iai5Db21tb25QcmVmaXhlcykge1xuICAgIHRvQXJyYXkoeG1sb2JqLkNvbW1vblByZWZpeGVzKS5mb3JFYWNoKChjb21tb25QcmVmaXgpID0+IHtcbiAgICAgIHJlc3VsdC5vYmplY3RzLnB1c2goeyBwcmVmaXg6IHNhbml0aXplT2JqZWN0S2V5KHRvQXJyYXkoY29tbW9uUHJlZml4LlByZWZpeClbMF0pLCBzaXplOiAwIH0pXG4gICAgfSlcbiAgfVxuICByZXR1cm4gcmVzdWx0XG59XG4iXSwibWFwcGluZ3MiOiI7Ozs7Ozs7OztBQWdCQSxJQUFBQSxjQUFBLEdBQUFDLE9BQUE7QUFFQSxJQUFBQyxNQUFBLEdBQUFDLHVCQUFBLENBQUFGLE9BQUE7QUFDQSxJQUFBRyxPQUFBLEdBQUFILE9BQUE7QUFBaUgsU0FBQUkseUJBQUFDLFdBQUEsZUFBQUMsT0FBQSxrQ0FBQUMsaUJBQUEsT0FBQUQsT0FBQSxRQUFBRSxnQkFBQSxPQUFBRixPQUFBLFlBQUFGLHdCQUFBLFlBQUFBLENBQUFDLFdBQUEsV0FBQUEsV0FBQSxHQUFBRyxnQkFBQSxHQUFBRCxpQkFBQSxLQUFBRixXQUFBO0FBQUEsU0FBQUgsd0JBQUFPLEdBQUEsRUFBQUosV0FBQSxTQUFBQSxXQUFBLElBQUFJLEdBQUEsSUFBQUEsR0FBQSxDQUFBQyxVQUFBLFdBQUFELEdBQUEsUUFBQUEsR0FBQSxvQkFBQUEsR0FBQSx3QkFBQUEsR0FBQSw0QkFBQUUsT0FBQSxFQUFBRixHQUFBLFVBQUFHLEtBQUEsR0FBQVIsd0JBQUEsQ0FBQUMsV0FBQSxPQUFBTyxLQUFBLElBQUFBLEtBQUEsQ0FBQUMsR0FBQSxDQUFBSixHQUFBLFlBQUFHLEtBQUEsQ0FBQUUsR0FBQSxDQUFBTCxHQUFBLFNBQUFNLE1BQUEsV0FBQUMscUJBQUEsR0FBQUMsTUFBQSxDQUFBQyxjQUFBLElBQUFELE1BQUEsQ0FBQUUsd0JBQUEsV0FBQUMsR0FBQSxJQUFBWCxHQUFBLFFBQUFXLEdBQUEsa0JBQUFILE1BQUEsQ0FBQUksU0FBQSxDQUFBQyxjQUFBLENBQUFDLElBQUEsQ0FBQWQsR0FBQSxFQUFBVyxHQUFBLFNBQUFJLElBQUEsR0FBQVIscUJBQUEsR0FBQUMsTUFBQSxDQUFBRSx3QkFBQSxDQUFBVixHQUFBLEVBQUFXLEdBQUEsY0FBQUksSUFBQSxLQUFBQSxJQUFBLENBQUFWLEdBQUEsSUFBQVUsSUFBQSxDQUFBQyxHQUFBLEtBQUFSLE1BQUEsQ0FBQUMsY0FBQSxDQUFBSCxNQUFBLEVBQUFLLEdBQUEsRUFBQUksSUFBQSxZQUFBVCxNQUFBLENBQUFLLEdBQUEsSUFBQVgsR0FBQSxDQUFBVyxHQUFBLFNBQUFMLE1BQUEsQ0FBQUosT0FBQSxHQUFBRixHQUFBLE1BQUFHLEtBQUEsSUFBQUEsS0FBQSxDQUFBYSxHQUFBLENBQUFoQixHQUFBLEVBQUFNLE1BQUEsWUFBQUEsTUFBQTtBQW5Cakg7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQU9BLE1BQU1XLG1CQUFtQixHQUFHLElBQUlDLHdCQUFTLENBQUM7RUFDeENDLGtCQUFrQixFQUFFO0lBQ2xCQyxRQUFRLEVBQUU7RUFDWjtBQUNGLENBQUMsQ0FBQzs7QUFFRjtBQUNPLFNBQVNDLHVCQUF1QkEsQ0FBQ0MsR0FBRyxFQUFFO0VBQzNDLElBQUlDLE1BQU0sR0FBRztJQUNYQyxrQkFBa0IsRUFBRSxFQUFFO0lBQ3RCQyxrQkFBa0IsRUFBRSxFQUFFO0lBQ3RCQywwQkFBMEIsRUFBRTtFQUM5QixDQUFDO0VBQ0Q7RUFDQSxJQUFJQyxTQUFTLEdBQUcsU0FBQUEsQ0FBVUMsTUFBTSxFQUFFO0lBQ2hDLElBQUlMLE1BQU0sR0FBRyxFQUFFO0lBQ2YsSUFBSUssTUFBTSxFQUFFO01BQ1YsSUFBQUMsZUFBTyxFQUFDRCxNQUFNLENBQUMsQ0FBQ0UsT0FBTyxDQUFFQyxPQUFPLElBQUs7UUFDbkNSLE1BQU0sQ0FBQ1MsSUFBSSxDQUFDRCxPQUFPLENBQUM7TUFDdEIsQ0FBQyxDQUFDO0lBQ0o7SUFDQSxPQUFPUixNQUFNO0VBQ2YsQ0FBQztFQUNEO0VBQ0EsSUFBSVUsY0FBYyxHQUFHLFNBQUFBLENBQVVDLE9BQU8sRUFBRTtJQUN0QyxJQUFJWCxNQUFNLEdBQUcsRUFBRTtJQUNmLElBQUlXLE9BQU8sRUFBRTtNQUNYQSxPQUFPLEdBQUcsSUFBQUwsZUFBTyxFQUFDSyxPQUFPLENBQUM7TUFDMUIsSUFBSUEsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDQyxLQUFLLEVBQUU7UUFDcEJELE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQ0MsS0FBSyxHQUFHLElBQUFOLGVBQU8sRUFBQ0ssT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDQyxLQUFLLENBQUM7UUFDNUMsSUFBSUQsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUNDLFVBQVUsRUFBRTtVQUNsQyxJQUFBUCxlQUFPLEVBQUNLLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQ0MsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDQyxVQUFVLENBQUMsQ0FBQ04sT0FBTyxDQUFFTyxJQUFJLElBQUs7WUFDeEQsSUFBSUMsSUFBSSxHQUFHLElBQUFULGVBQU8sRUFBQ1EsSUFBSSxDQUFDQyxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDaEMsSUFBSUMsS0FBSyxHQUFHLElBQUFWLGVBQU8sRUFBQ1EsSUFBSSxDQUFDRSxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDbENoQixNQUFNLENBQUNTLElBQUksQ0FBQztjQUFFTSxJQUFJO2NBQUVDO1lBQU0sQ0FBQyxDQUFDO1VBQzlCLENBQUMsQ0FBQztRQUNKO01BQ0Y7SUFDRjtJQUNBLE9BQU9oQixNQUFNO0VBQ2YsQ0FBQztFQUVELElBQUlpQixNQUFNLEdBQUcsSUFBQUMsZ0JBQVEsRUFBQ25CLEdBQUcsQ0FBQztFQUMxQmtCLE1BQU0sR0FBR0EsTUFBTSxDQUFDRSx5QkFBeUI7O0VBRXpDO0VBQ0EsSUFBSUYsTUFBTSxDQUFDaEIsa0JBQWtCLEVBQUU7SUFDN0IsSUFBQUssZUFBTyxFQUFDVyxNQUFNLENBQUNoQixrQkFBa0IsQ0FBQyxDQUFDTSxPQUFPLENBQUVhLE1BQU0sSUFBSztNQUNyRCxJQUFJQyxFQUFFLEdBQUcsSUFBQWYsZUFBTyxFQUFDYyxNQUFNLENBQUNDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztNQUM5QixJQUFJQyxLQUFLLEdBQUcsSUFBQWhCLGVBQU8sRUFBQ2MsTUFBTSxDQUFDRSxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7TUFDcEMsSUFBSUMsS0FBSyxHQUFHbkIsU0FBUyxDQUFDZ0IsTUFBTSxDQUFDRyxLQUFLLENBQUM7TUFDbkMsSUFBSUMsTUFBTSxHQUFHZCxjQUFjLENBQUNVLE1BQU0sQ0FBQ0ksTUFBTSxDQUFDO01BQzFDeEIsTUFBTSxDQUFDQyxrQkFBa0IsQ0FBQ1EsSUFBSSxDQUFDO1FBQUVZLEVBQUU7UUFBRUMsS0FBSztRQUFFQyxLQUFLO1FBQUVDO01BQU8sQ0FBQyxDQUFDO0lBQzlELENBQUMsQ0FBQztFQUNKO0VBQ0E7RUFDQSxJQUFJUCxNQUFNLENBQUNmLGtCQUFrQixFQUFFO0lBQzdCLElBQUFJLGVBQU8sRUFBQ1csTUFBTSxDQUFDZixrQkFBa0IsQ0FBQyxDQUFDSyxPQUFPLENBQUVhLE1BQU0sSUFBSztNQUNyRCxJQUFJQyxFQUFFLEdBQUcsSUFBQWYsZUFBTyxFQUFDYyxNQUFNLENBQUNDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztNQUM5QixJQUFJSSxLQUFLLEdBQUcsSUFBQW5CLGVBQU8sRUFBQ2MsTUFBTSxDQUFDSyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7TUFDcEMsSUFBSUYsS0FBSyxHQUFHbkIsU0FBUyxDQUFDZ0IsTUFBTSxDQUFDRyxLQUFLLENBQUM7TUFDbkMsSUFBSUMsTUFBTSxHQUFHZCxjQUFjLENBQUNVLE1BQU0sQ0FBQ0ksTUFBTSxDQUFDO01BQzFDeEIsTUFBTSxDQUFDRSxrQkFBa0IsQ0FBQ08sSUFBSSxDQUFDO1FBQUVZLEVBQUU7UUFBRUksS0FBSztRQUFFRixLQUFLO1FBQUVDO01BQU8sQ0FBQyxDQUFDO0lBQzlELENBQUMsQ0FBQztFQUNKO0VBQ0E7RUFDQSxJQUFJUCxNQUFNLENBQUNkLDBCQUEwQixFQUFFO0lBQ3JDLElBQUFHLGVBQU8sRUFBQ1csTUFBTSxDQUFDZCwwQkFBMEIsQ0FBQyxDQUFDSSxPQUFPLENBQUVhLE1BQU0sSUFBSztNQUM3RCxJQUFJQyxFQUFFLEdBQUcsSUFBQWYsZUFBTyxFQUFDYyxNQUFNLENBQUNDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztNQUM5QixJQUFJSyxhQUFhLEdBQUcsSUFBQXBCLGVBQU8sRUFBQ2MsTUFBTSxDQUFDTSxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUM7TUFDcEQsSUFBSUgsS0FBSyxHQUFHbkIsU0FBUyxDQUFDZ0IsTUFBTSxDQUFDRyxLQUFLLENBQUM7TUFDbkMsSUFBSUMsTUFBTSxHQUFHZCxjQUFjLENBQUNVLE1BQU0sQ0FBQ0ksTUFBTSxDQUFDO01BQzFDeEIsTUFBTSxDQUFDRywwQkFBMEIsQ0FBQ00sSUFBSSxDQUFDO1FBQUVZLEVBQUU7UUFBRUssYUFBYTtRQUFFSCxLQUFLO1FBQUVDO01BQU8sQ0FBQyxDQUFDO0lBQzlFLENBQUMsQ0FBQztFQUNKO0VBRUEsT0FBT3hCLE1BQU07QUFDZjtBQUVBLE1BQU0yQixhQUFhLEdBQUdBLENBQUNDLE9BQU8sRUFBRUMsSUFBSSxHQUFHLENBQUMsQ0FBQyxLQUFLO0VBQzVDLElBQUk7SUFBRUMsR0FBRztJQUFFQyxZQUFZO0lBQUVDLElBQUk7SUFBRUMsSUFBSTtJQUFFQyxTQUFTO0lBQUVDO0VBQVMsQ0FBQyxHQUFHUCxPQUFPO0VBRXBFLElBQUksQ0FBQyxJQUFBUSxnQkFBUSxFQUFDUCxJQUFJLENBQUMsRUFBRTtJQUNuQkEsSUFBSSxHQUFHLENBQUMsQ0FBQztFQUNYO0VBRUEsTUFBTVEsSUFBSSxHQUFHLElBQUFDLHlCQUFpQixFQUFDLElBQUFoQyxlQUFPLEVBQUN3QixHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztFQUMvQyxNQUFNUyxZQUFZLEdBQUcsSUFBSUMsSUFBSSxDQUFDLElBQUFsQyxlQUFPLEVBQUN5QixZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztFQUN2RCxNQUFNVSxJQUFJLEdBQUcsSUFBQUMsb0JBQVksRUFBQyxJQUFBcEMsZUFBTyxFQUFDMEIsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7RUFDM0MsTUFBTVcsSUFBSSxHQUFHLElBQUFDLG9CQUFZLEVBQUNYLElBQUksQ0FBQztFQUUvQixPQUFPO0lBQ0xJLElBQUk7SUFDSkUsWUFBWTtJQUNaRSxJQUFJO0lBQ0pFLElBQUk7SUFDSkUsU0FBUyxFQUFFWCxTQUFTO0lBQ3BCWSxRQUFRLEVBQUVYLFFBQVE7SUFDbEJZLGNBQWMsRUFBRWxCLElBQUksQ0FBQ21CLGNBQWMsR0FBR25CLElBQUksQ0FBQ21CLGNBQWMsR0FBRztFQUM5RCxDQUFDO0FBQ0gsQ0FBQzs7QUFFRDtBQUNPLFNBQVNDLGdCQUFnQkEsQ0FBQ2xELEdBQUcsRUFBRTtFQUNwQyxJQUFJQyxNQUFNLEdBQUc7SUFDWGtELE9BQU8sRUFBRSxFQUFFO0lBQ1hDLFdBQVcsRUFBRTtFQUNmLENBQUM7RUFDRCxJQUFJQSxXQUFXLEdBQUcsS0FBSztFQUN2QixJQUFJQyxVQUFVLEVBQUVDLG9CQUFvQjtFQUNwQyxNQUFNcEMsTUFBTSxHQUFHdkIsbUJBQW1CLENBQUM0RCxLQUFLLENBQUN2RCxHQUFHLENBQUM7RUFFN0MsTUFBTXdELHlCQUF5QixHQUFJQyxjQUFjLElBQUs7SUFDcEQsSUFBSUEsY0FBYyxFQUFFO01BQ2xCLElBQUFsRCxlQUFPLEVBQUNrRCxjQUFjLENBQUMsQ0FBQ2pELE9BQU8sQ0FBRWtELFlBQVksSUFBSztRQUNoRHpELE1BQU0sQ0FBQ2tELE9BQU8sQ0FBQ3pDLElBQUksQ0FBQztVQUFFaUQsTUFBTSxFQUFFLElBQUFwQix5QkFBaUIsRUFBQyxJQUFBaEMsZUFBTyxFQUFDbUQsWUFBWSxDQUFDRSxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztVQUFFaEIsSUFBSSxFQUFFO1FBQUUsQ0FBQyxDQUFDO01BQzlGLENBQUMsQ0FBQztJQUNKO0VBQ0YsQ0FBQztFQUVELE1BQU1pQixnQkFBZ0IsR0FBRzNDLE1BQU0sQ0FBQzRDLGdCQUFnQjtFQUNoRCxNQUFNQyxrQkFBa0IsR0FBRzdDLE1BQU0sQ0FBQzhDLGtCQUFrQjtFQUVwRCxJQUFJSCxnQkFBZ0IsRUFBRTtJQUNwQixJQUFJQSxnQkFBZ0IsQ0FBQ0ksV0FBVyxFQUFFO01BQ2hDYixXQUFXLEdBQUdTLGdCQUFnQixDQUFDSSxXQUFXO0lBQzVDO0lBQ0EsSUFBSUosZ0JBQWdCLENBQUNLLFFBQVEsRUFBRTtNQUM3QixJQUFBM0QsZUFBTyxFQUFDc0QsZ0JBQWdCLENBQUNLLFFBQVEsQ0FBQyxDQUFDMUQsT0FBTyxDQUFFcUIsT0FBTyxJQUFLO1FBQ3RELE1BQU1TLElBQUksR0FBRyxJQUFBQyx5QkFBaUIsRUFBQyxJQUFBaEMsZUFBTyxFQUFDc0IsT0FBTyxDQUFDRSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN2RCxNQUFNUyxZQUFZLEdBQUcsSUFBSUMsSUFBSSxDQUFDLElBQUFsQyxlQUFPLEVBQUNzQixPQUFPLENBQUNHLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQy9ELE1BQU1VLElBQUksR0FBRyxJQUFBQyxvQkFBWSxFQUFDLElBQUFwQyxlQUFPLEVBQUNzQixPQUFPLENBQUNJLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ25ELE1BQU1XLElBQUksR0FBRyxJQUFBQyxvQkFBWSxFQUFDaEIsT0FBTyxDQUFDSyxJQUFJLENBQUM7UUFDdkNqQyxNQUFNLENBQUNrRCxPQUFPLENBQUN6QyxJQUFJLENBQUM7VUFBRTRCLElBQUk7VUFBRUUsWUFBWTtVQUFFRSxJQUFJO1VBQUVFO1FBQUssQ0FBQyxDQUFDO01BQ3pELENBQUMsQ0FBQztJQUNKO0lBRUEsSUFBSWlCLGdCQUFnQixDQUFDTSxVQUFVLEVBQUU7TUFDL0JkLFVBQVUsR0FBR1EsZ0JBQWdCLENBQUNNLFVBQVU7SUFDMUMsQ0FBQyxNQUFNLElBQUlmLFdBQVcsSUFBSW5ELE1BQU0sQ0FBQ2tELE9BQU8sQ0FBQ2lCLE1BQU0sR0FBRyxDQUFDLEVBQUU7TUFDbkRmLFVBQVUsR0FBR3BELE1BQU0sQ0FBQ2tELE9BQU8sQ0FBQ2xELE1BQU0sQ0FBQ2tELE9BQU8sQ0FBQ2lCLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQzlCLElBQUk7SUFDN0Q7SUFDQWtCLHlCQUF5QixDQUFDSyxnQkFBZ0IsQ0FBQ1EsY0FBYyxDQUFDO0VBQzVEO0VBRUEsSUFBSU4sa0JBQWtCLEVBQUU7SUFDdEIsSUFBSUEsa0JBQWtCLENBQUNFLFdBQVcsRUFBRTtNQUNsQ2IsV0FBVyxHQUFHVyxrQkFBa0IsQ0FBQ0UsV0FBVztJQUM5QztJQUVBLElBQUlGLGtCQUFrQixDQUFDTyxPQUFPLEVBQUU7TUFDOUIsSUFBQS9ELGVBQU8sRUFBQ3dELGtCQUFrQixDQUFDTyxPQUFPLENBQUMsQ0FBQzlELE9BQU8sQ0FBRXFCLE9BQU8sSUFBSztRQUN2RDVCLE1BQU0sQ0FBQ2tELE9BQU8sQ0FBQ3pDLElBQUksQ0FBQ2tCLGFBQWEsQ0FBQ0MsT0FBTyxDQUFDLENBQUM7TUFDN0MsQ0FBQyxDQUFDO0lBQ0o7SUFDQSxJQUFJa0Msa0JBQWtCLENBQUNRLFlBQVksRUFBRTtNQUNuQyxJQUFBaEUsZUFBTyxFQUFDd0Qsa0JBQWtCLENBQUNRLFlBQVksQ0FBQyxDQUFDL0QsT0FBTyxDQUFFcUIsT0FBTyxJQUFLO1FBQzVENUIsTUFBTSxDQUFDa0QsT0FBTyxDQUFDekMsSUFBSSxDQUFDa0IsYUFBYSxDQUFDQyxPQUFPLEVBQUU7VUFBRW9CLGNBQWMsRUFBRTtRQUFLLENBQUMsQ0FBQyxDQUFDO01BQ3ZFLENBQUMsQ0FBQztJQUNKO0lBRUEsSUFBSWMsa0JBQWtCLENBQUNTLGFBQWEsRUFBRTtNQUNwQ2xCLG9CQUFvQixHQUFHUyxrQkFBa0IsQ0FBQ1MsYUFBYTtJQUN6RDtJQUNBLElBQUlULGtCQUFrQixDQUFDVSxtQkFBbUIsRUFBRTtNQUMxQ3hFLE1BQU0sQ0FBQ3lFLGVBQWUsR0FBR1gsa0JBQWtCLENBQUNVLG1CQUFtQjtJQUNqRTtJQUNBakIseUJBQXlCLENBQUNPLGtCQUFrQixDQUFDTSxjQUFjLENBQUM7RUFDOUQ7RUFFQXBFLE1BQU0sQ0FBQ21ELFdBQVcsR0FBR0EsV0FBVztFQUNoQyxJQUFJQSxXQUFXLEVBQUU7SUFDZm5ELE1BQU0sQ0FBQ29ELFVBQVUsR0FBR0Msb0JBQW9CLElBQUlELFVBQVU7RUFDeEQ7RUFDQSxPQUFPcEQsTUFBTTtBQUNmOztBQUVBO0FBQ08sU0FBUzBFLGtCQUFrQkEsQ0FBQzNFLEdBQUcsRUFBRTtFQUN0QyxJQUFJQyxNQUFNLEdBQUc7SUFDWGtELE9BQU8sRUFBRSxFQUFFO0lBQ1hDLFdBQVcsRUFBRTtFQUNmLENBQUM7RUFDRCxJQUFJbEMsTUFBTSxHQUFHLElBQUFDLGdCQUFRLEVBQUNuQixHQUFHLENBQUM7RUFDMUIsSUFBSSxDQUFDa0IsTUFBTSxDQUFDNEMsZ0JBQWdCLEVBQUU7SUFDNUIsTUFBTSxJQUFJNUYsTUFBTSxDQUFDMEcsZUFBZSxDQUFDLGlDQUFpQyxDQUFDO0VBQ3JFO0VBQ0ExRCxNQUFNLEdBQUdBLE1BQU0sQ0FBQzRDLGdCQUFnQjtFQUNoQyxJQUFJNUMsTUFBTSxDQUFDK0MsV0FBVyxFQUFFO0lBQ3RCaEUsTUFBTSxDQUFDbUQsV0FBVyxHQUFHbEMsTUFBTSxDQUFDK0MsV0FBVztFQUN6QztFQUNBLElBQUkvQyxNQUFNLENBQUMyRCxxQkFBcUIsRUFBRTtJQUNoQzVFLE1BQU0sQ0FBQzZFLHFCQUFxQixHQUFHNUQsTUFBTSxDQUFDMkQscUJBQXFCO0VBQzdEO0VBQ0EsSUFBSTNELE1BQU0sQ0FBQ2dELFFBQVEsRUFBRTtJQUNuQixJQUFBM0QsZUFBTyxFQUFDVyxNQUFNLENBQUNnRCxRQUFRLENBQUMsQ0FBQzFELE9BQU8sQ0FBRXFCLE9BQU8sSUFBSztNQUM1QyxJQUFJUyxJQUFJLEdBQUcsSUFBQUMseUJBQWlCLEVBQUMsSUFBQWhDLGVBQU8sRUFBQ3NCLE9BQU8sQ0FBQ0UsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7TUFDckQsSUFBSVMsWUFBWSxHQUFHLElBQUlDLElBQUksQ0FBQ1osT0FBTyxDQUFDRyxZQUFZLENBQUM7TUFDakQsSUFBSVUsSUFBSSxHQUFHLElBQUFDLG9CQUFZLEVBQUNkLE9BQU8sQ0FBQ0ksSUFBSSxDQUFDO01BQ3JDLElBQUlXLElBQUksR0FBR2YsT0FBTyxDQUFDSyxJQUFJO01BQ3ZCakMsTUFBTSxDQUFDa0QsT0FBTyxDQUFDekMsSUFBSSxDQUFDO1FBQUU0QixJQUFJO1FBQUVFLFlBQVk7UUFBRUUsSUFBSTtRQUFFRTtNQUFLLENBQUMsQ0FBQztJQUN6RCxDQUFDLENBQUM7RUFDSjtFQUNBLElBQUkxQixNQUFNLENBQUNtRCxjQUFjLEVBQUU7SUFDekIsSUFBQTlELGVBQU8sRUFBQ1csTUFBTSxDQUFDbUQsY0FBYyxDQUFDLENBQUM3RCxPQUFPLENBQUVrRCxZQUFZLElBQUs7TUFDdkR6RCxNQUFNLENBQUNrRCxPQUFPLENBQUN6QyxJQUFJLENBQUM7UUFBRWlELE1BQU0sRUFBRSxJQUFBcEIseUJBQWlCLEVBQUMsSUFBQWhDLGVBQU8sRUFBQ21ELFlBQVksQ0FBQ0UsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFBRWhCLElBQUksRUFBRTtNQUFFLENBQUMsQ0FBQztJQUM5RixDQUFDLENBQUM7RUFDSjtFQUNBLE9BQU8zQyxNQUFNO0FBQ2Y7O0FBRUE7QUFDTyxTQUFTOEUsOEJBQThCQSxDQUFDL0UsR0FBRyxFQUFFO0VBQ2xELElBQUlDLE1BQU0sR0FBRztJQUNYa0QsT0FBTyxFQUFFLEVBQUU7SUFDWEMsV0FBVyxFQUFFO0VBQ2YsQ0FBQztFQUNELElBQUlsQyxNQUFNLEdBQUcsSUFBQUMsZ0JBQVEsRUFBQ25CLEdBQUcsQ0FBQztFQUMxQixJQUFJLENBQUNrQixNQUFNLENBQUM0QyxnQkFBZ0IsRUFBRTtJQUM1QixNQUFNLElBQUk1RixNQUFNLENBQUMwRyxlQUFlLENBQUMsaUNBQWlDLENBQUM7RUFDckU7RUFDQTFELE1BQU0sR0FBR0EsTUFBTSxDQUFDNEMsZ0JBQWdCO0VBQ2hDLElBQUk1QyxNQUFNLENBQUMrQyxXQUFXLEVBQUU7SUFDdEJoRSxNQUFNLENBQUNtRCxXQUFXLEdBQUdsQyxNQUFNLENBQUMrQyxXQUFXO0VBQ3pDO0VBQ0EsSUFBSS9DLE1BQU0sQ0FBQzJELHFCQUFxQixFQUFFO0lBQ2hDNUUsTUFBTSxDQUFDNkUscUJBQXFCLEdBQUc1RCxNQUFNLENBQUMyRCxxQkFBcUI7RUFDN0Q7RUFFQSxJQUFJM0QsTUFBTSxDQUFDZ0QsUUFBUSxFQUFFO0lBQ25CLElBQUEzRCxlQUFPLEVBQUNXLE1BQU0sQ0FBQ2dELFFBQVEsQ0FBQyxDQUFDMUQsT0FBTyxDQUFFcUIsT0FBTyxJQUFLO01BQzVDLElBQUlTLElBQUksR0FBRyxJQUFBQyx5QkFBaUIsRUFBQ1YsT0FBTyxDQUFDRSxHQUFHLENBQUM7TUFDekMsSUFBSVMsWUFBWSxHQUFHLElBQUlDLElBQUksQ0FBQ1osT0FBTyxDQUFDRyxZQUFZLENBQUM7TUFDakQsSUFBSVUsSUFBSSxHQUFHLElBQUFDLG9CQUFZLEVBQUNkLE9BQU8sQ0FBQ0ksSUFBSSxDQUFDO01BQ3JDLElBQUlXLElBQUksR0FBR2YsT0FBTyxDQUFDSyxJQUFJO01BQ3ZCLElBQUk4QyxRQUFRO01BQ1osSUFBSW5ELE9BQU8sQ0FBQ29ELFlBQVksSUFBSSxJQUFJLEVBQUU7UUFDaENELFFBQVEsR0FBRyxJQUFBekUsZUFBTyxFQUFDc0IsT0FBTyxDQUFDb0QsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDO01BQzdDLENBQUMsTUFBTTtRQUNMRCxRQUFRLEdBQUcsSUFBSTtNQUNqQjtNQUNBL0UsTUFBTSxDQUFDa0QsT0FBTyxDQUFDekMsSUFBSSxDQUFDO1FBQUU0QixJQUFJO1FBQUVFLFlBQVk7UUFBRUUsSUFBSTtRQUFFRSxJQUFJO1FBQUVvQztNQUFTLENBQUMsQ0FBQztJQUNuRSxDQUFDLENBQUM7RUFDSjtFQUVBLElBQUk5RCxNQUFNLENBQUNtRCxjQUFjLEVBQUU7SUFDekIsSUFBQTlELGVBQU8sRUFBQ1csTUFBTSxDQUFDbUQsY0FBYyxDQUFDLENBQUM3RCxPQUFPLENBQUVrRCxZQUFZLElBQUs7TUFDdkR6RCxNQUFNLENBQUNrRCxPQUFPLENBQUN6QyxJQUFJLENBQUM7UUFBRWlELE1BQU0sRUFBRSxJQUFBcEIseUJBQWlCLEVBQUMsSUFBQWhDLGVBQU8sRUFBQ21ELFlBQVksQ0FBQ0UsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFBRWhCLElBQUksRUFBRTtNQUFFLENBQUMsQ0FBQztJQUM5RixDQUFDLENBQUM7RUFDSjtFQUNBLE9BQU8zQyxNQUFNO0FBQ2YifQ==