import * as fs from "fs";
import * as path from "path";
import * as querystring from 'query-string';
import * as errors from "./errors.mjs";
import { getEncryptionHeaders, isEmpty, isEmptyObject, isNumber, isObject, isString, isValidBucketName, isValidObjectName } from "./internal/helper.mjs";
import { RETENTION_MODES } from "./internal/type.mjs";
export { ENCRYPTION_TYPES, LEGAL_HOLD_STATUS, RETENTION_MODES, RETENTION_VALIDITY_UNITS } from "./internal/type.mjs";
export const DEFAULT_REGION = 'us-east-1';
export const PRESIGN_EXPIRY_DAYS_MAX = 24 * 60 * 60 * 7; // 7 days in seconds

export class CopySourceOptions {
  constructor({
    Bucket,
    Object,
    VersionID = '',
    MatchETag = '',
    NoMatchETag = '',
    MatchModifiedSince = null,
    MatchUnmodifiedSince = null,
    MatchRange = false,
    Start = 0,
    End = 0,
    Encryption = undefined
  }) {
    this.Bucket = Bucket;
    this.Object = Object;
    this.VersionID = VersionID;
    this.MatchETag = MatchETag;
    this.NoMatchETag = NoMatchETag;
    this.MatchModifiedSince = MatchModifiedSince;
    this.MatchUnmodifiedSince = MatchUnmodifiedSince;
    this.MatchRange = MatchRange;
    this.Start = Start;
    this.End = End;
    this.Encryption = Encryption;
  }
  validate() {
    if (!isValidBucketName(this.Bucket)) {
      throw new errors.InvalidBucketNameError('Invalid Source bucket name: ' + this.Bucket);
    }
    if (!isValidObjectName(this.Object)) {
      throw new errors.InvalidObjectNameError(`Invalid Source object name: ${this.Object}`);
    }
    if (this.MatchRange && this.Start !== -1 && this.End !== -1 && this.Start > this.End || this.Start < 0) {
      throw new errors.InvalidObjectNameError('Source start must be non-negative, and start must be at most end.');
    } else if (this.MatchRange && !isNumber(this.Start) || !isNumber(this.End)) {
      throw new errors.InvalidObjectNameError('MatchRange is specified. But Invalid Start and End values are specified.');
    }
    return true;
  }
  getHeaders() {
    const headerOptions = {};
    headerOptions['x-amz-copy-source'] = encodeURI(this.Bucket + '/' + this.Object);
    if (!isEmpty(this.VersionID)) {
      headerOptions['x-amz-copy-source'] = `${encodeURI(this.Bucket + '/' + this.Object)}?versionId=${this.VersionID}`;
    }
    if (!isEmpty(this.MatchETag)) {
      headerOptions['x-amz-copy-source-if-match'] = this.MatchETag;
    }
    if (!isEmpty(this.NoMatchETag)) {
      headerOptions['x-amz-copy-source-if-none-match'] = this.NoMatchETag;
    }
    if (!isEmpty(this.MatchModifiedSince)) {
      headerOptions['x-amz-copy-source-if-modified-since'] = this.MatchModifiedSince;
    }
    if (!isEmpty(this.MatchUnmodifiedSince)) {
      headerOptions['x-amz-copy-source-if-unmodified-since'] = this.MatchUnmodifiedSince;
    }
    return headerOptions;
  }
}

/**
 * @deprecated use nodejs fs module
 */
export function removeDirAndFiles(dirPath, removeSelf = true) {
  if (removeSelf) {
    return fs.rmSync(dirPath, {
      recursive: true,
      force: true
    });
  }
  fs.readdirSync(dirPath).forEach(item => {
    fs.rmSync(path.join(dirPath, item), {
      recursive: true,
      force: true
    });
  });
}
export class CopyDestinationOptions {
  constructor({
    Bucket,
    Object,
    Encryption,
    UserMetadata,
    UserTags,
    LegalHold,
    RetainUntilDate,
    Mode,
    MetadataDirective
  }) {
    this.Bucket = Bucket;
    this.Object = Object;
    this.Encryption = Encryption ?? undefined; // null input will become undefined, easy for runtime assert
    this.UserMetadata = UserMetadata;
    this.UserTags = UserTags;
    this.LegalHold = LegalHold;
    this.Mode = Mode; // retention mode
    this.RetainUntilDate = RetainUntilDate;
    this.MetadataDirective = MetadataDirective;
  }
  getHeaders() {
    const replaceDirective = 'REPLACE';
    const headerOptions = {};
    const userTags = this.UserTags;
    if (!isEmpty(userTags)) {
      headerOptions['X-Amz-Tagging-Directive'] = replaceDirective;
      headerOptions['X-Amz-Tagging'] = isObject(userTags) ? querystring.stringify(userTags) : isString(userTags) ? userTags : '';
    }
    if (this.Mode) {
      headerOptions['X-Amz-Object-Lock-Mode'] = this.Mode; // GOVERNANCE or COMPLIANCE
    }

    if (this.RetainUntilDate) {
      headerOptions['X-Amz-Object-Lock-Retain-Until-Date'] = this.RetainUntilDate; // needs to be UTC.
    }

    if (this.LegalHold) {
      headerOptions['X-Amz-Object-Lock-Legal-Hold'] = this.LegalHold; // ON or OFF
    }

    if (this.UserMetadata) {
      for (const [key, value] of Object.entries(this.UserMetadata)) {
        headerOptions[`X-Amz-Meta-${key}`] = value.toString();
      }
    }
    if (this.MetadataDirective) {
      headerOptions[`X-Amz-Metadata-Directive`] = this.MetadataDirective;
    }
    if (this.Encryption) {
      const encryptionHeaders = getEncryptionHeaders(this.Encryption);
      for (const [key, value] of Object.entries(encryptionHeaders)) {
        headerOptions[key] = value;
      }
    }
    return headerOptions;
  }
  validate() {
    if (!isValidBucketName(this.Bucket)) {
      throw new errors.InvalidBucketNameError('Invalid Destination bucket name: ' + this.Bucket);
    }
    if (!isValidObjectName(this.Object)) {
      throw new errors.InvalidObjectNameError(`Invalid Destination object name: ${this.Object}`);
    }
    if (!isEmpty(this.UserMetadata) && !isObject(this.UserMetadata)) {
      throw new errors.InvalidObjectNameError(`Destination UserMetadata should be an object with key value pairs`);
    }
    if (!isEmpty(this.Mode) && ![RETENTION_MODES.GOVERNANCE, RETENTION_MODES.COMPLIANCE].includes(this.Mode)) {
      throw new errors.InvalidObjectNameError(`Invalid Mode specified for destination object it should be one of [GOVERNANCE,COMPLIANCE]`);
    }
    if (this.Encryption !== undefined && isEmptyObject(this.Encryption)) {
      throw new errors.InvalidObjectNameError(`Invalid Encryption configuration for destination object `);
    }
    return true;
  }
}

/**
 * maybe this should be a generic type for Records, leave it for later refactor
 */
export class SelectResults {
  constructor({
    records,
    // parsed data as stream
    response,
    // original response stream
    stats,
    // stats as xml
    progress // stats as xml
  }) {
    this.records = records;
    this.response = response;
    this.stats = stats;
    this.progress = progress;
  }
  setStats(stats) {
    this.stats = stats;
  }
  getStats() {
    return this.stats;
  }
  setProgress(progress) {
    this.progress = progress;
  }
  getProgress() {
    return this.progress;
  }
  setResponse(response) {
    this.response = response;
  }
  getResponse() {
    return this.response;
  }
  setRecords(records) {
    this.records = records;
  }
  getRecords() {
    return this.records;
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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