(function (global, factory) {
  typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
  typeof define === 'function' && define.amd ? define(['exports'], factory) :
  (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.MOJFrontend = global.MOJFrontend || {}));
})(this, (function (exports) { 'use strict';

  /**
   * Find an elements preceding sibling
   *
   * Utility function to find an elements previous sibling matching the provided
   * selector.
   *
   * @param {HTMLElement} $element - Element to find siblings for
   * @param {string} selector - selector for required sibling
   */
  function getPreviousSibling($element, selector) {
    if (!$element) return
    // Get the previous sibling element
    let $sibling = $element.previousElementSibling;

    // If the sibling matches our selector, use it
    // If not, jump to the next sibling and continue the loop
    while ($sibling) {
      if ($sibling.matches(selector)) return $sibling
      $sibling = $sibling.previousElementSibling;
    }
  }

  function findNearestMatchingElement($element, selector) {
    // If no element or selector is provided, return null
    if (!$element) return

    // Start with the current element
    let $currentElement = $element;

    while ($currentElement) {
      // First check the current element
      if ($currentElement.matches(selector)) {
        return $currentElement
      }

      // Check all previous siblings
      let $sibling = $currentElement.previousElementSibling;
      while ($sibling) {
        // Check if the sibling itself is a heading
        if ($sibling.matches(selector)) {
          return $sibling
        }
        $sibling = $sibling.previousElementSibling;
      }

      // If no match found in siblings, move up to parent
      $currentElement = $currentElement.parentElement;
    }
  }

  /**
   * Move focus to element
   *
   * Sets tabindex to -1 to make the element programmatically focusable,
   * but removes it on blur as the element doesn't need to be focused again.
   *
   * @param {HTMLElement} $element - HTML element
   * @param {object} [options] - Handler options
   * @param {function(this: HTMLElement): void} [options.onBeforeFocus] - Callback before focus
   * @param {function(this: HTMLElement): void} [options.onBlur] - Callback on blur
   */
  function setFocus($element, options = {}) {
    const isFocusable = $element.getAttribute('tabindex');

    if (!isFocusable) {
      $element.setAttribute('tabindex', '-1');
    }

    /**
     * Handle element focus
     */
    function onFocus() {
      $element.addEventListener('blur', onBlur, { once: true });
    }

    /**
     * Handle element blur
     */
    function onBlur() {
      if (options.onBlur) {
        options.onBlur.call($element);
      }

      if (!isFocusable) {
        $element.removeAttribute('tabindex');
      }
    }

    // Add listener to reset element on blur, after focus
    $element.addEventListener('focus', onFocus, { once: true });

    // Focus element
    if (options.onBeforeFocus) {
      options.onBeforeFocus.call($element);
    }
    $element.focus();
  }

  /**
   * @typedef {object} AlertConfig
   * @property {boolean} [dismissible=false] - Can the alert be dismissed by the user
   * @property {string} [dismissText=Dismiss] - the label text for the dismiss button
   * @property {boolean} [disableAutoFocus=false] - whether the alert will be autofocused
   * @property {string} [focusOnDismissSelector] - CSS Selector for element to be focused on dismiss
   */

  /**
   * @param {HTMLElement} $module - the Alert element
   * @param {AlertConfig} config - configuration options
   * @class
   */
  function Alert($module, config = {}) {
    if (!$module) {
      return this
    }

    const schema = Object.freeze({
      properties: {
        dismissible: { type: 'boolean' },
        dismissText: { type: 'string' },
        disableAutoFocus: { type: 'boolean' },
        focusOnDismissSelector: { type: 'string' }
      }
    });

    const defaults = {
      dismissible: false,
      dismissText: 'Dismiss',
      disableAutoFocus: false
    };

    // data attributes override JS config, which overrides defaults
    this.config = this.mergeConfigs(
      defaults,
      config,
      this.parseDataset(schema, $module.dataset)
    );

    this.$module = $module;
  }

  Alert.prototype.init = function () {
    /**
     * Focus the alert
     *
     * If `role="alert"` is set, focus the element to help some assistive
     * technologies prioritise announcing it.
     *
     * You can turn off the auto-focus functionality by setting
     * `data-disable-auto-focus="true"` in the component HTML. You might wish to
     * do this based on user research findings, or to avoid a clash with another
     * element which should be focused when the page loads.
     */
    if (
      this.$module.getAttribute('role') === 'alert' &&
      !this.config.disableAutoFocus
    ) {
      setFocus(this.$module);
    }

    this.$dismissButton = this.$module.querySelector('.moj-alert__dismiss');

    if (this.config.dismissible && this.$dismissButton) {
      this.$dismissButton.innerHTML = this.config.dismissText;
      this.$dismissButton.removeAttribute('hidden');

      this.$module.addEventListener('click', (event) => {
        if (this.$dismissButton.contains(event.target)) {
          this.dimiss();
        }
      });
    }
  };

  /**
   * Handle dismissing the alert
   */
  Alert.prototype.dimiss = function () {
    let $elementToRecieveFocus;

    // If a selector has been provided, attempt to find that element
    if (this.config.focusOnDismissSelector) {
      $elementToRecieveFocus = document.querySelector(
        this.config.focusOnDismissSelector
      );
    }

    // Is the next sibling another alert
    if (!$elementToRecieveFocus) {
      const $nextSibling = this.$module.nextElementSibling;
      if ($nextSibling && $nextSibling.matches('.moj-alert')) {
        $elementToRecieveFocus = $nextSibling;
      }
    }

    // Else try to find any preceding sibling alert or heading
    if (!$elementToRecieveFocus) {
      $elementToRecieveFocus = getPreviousSibling(
        this.$module,
        '.moj-alert, h1, h2, h3, h4, h5, h6'
      );
    }

    // Else find the closest ancestor heading, or fallback to main, or last resort
    // use the body element
    if (!$elementToRecieveFocus) {
      $elementToRecieveFocus = findNearestMatchingElement(
        this.$module,
        'h1, h2, h3, h4, h5, h6, main, body'
      );
    }

    // If we have an element, place focus on it
    if ($elementToRecieveFocus) {
      setFocus($elementToRecieveFocus);
    }

    // Remove the alert
    this.$module.remove();
  };

  /**
   * Normalise string
   *
   * 'If it looks like a duck, and it quacks like a duck…' 🦆
   *
   * If the passed value looks like a boolean or a number, convert it to a boolean
   * or number.
   *
   * Designed to be used to convert config passed via data attributes (which are
   * always strings) into something sensible.
   *
   * @internal
   * @param {DOMStringMap[string]} value - The value to normalise
   * @param {SchemaProperty} [property] - Component schema property
   * @returns {string | boolean | number | undefined} Normalised data
   */
  Alert.prototype.normaliseString = function (value, property) {
    const trimmedValue = value ? value.trim() : '';

    let output;
    let outputType;
    if (property && property.type) {
      outputType = property.type;
    }

    // No schema type set? Determine automatically
    if (!outputType) {
      if (['true', 'false'].includes(trimmedValue)) {
        outputType = 'boolean';
      }

      // Empty / whitespace-only strings are considered finite so we need to check
      // the length of the trimmed string as well
      if (trimmedValue.length > 0 && isFinite(Number(trimmedValue))) {
        outputType = 'number';
      }
    }

    switch (outputType) {
      case 'boolean':
        output = trimmedValue === 'true';
        break

      case 'number':
        output = Number(trimmedValue);
        break

      default:
        output = value;
    }

    return output
  };

  /**
   * Parse dataset
   *
   * Loop over an object and normalise each value using {@link normaliseString},
   * optionally expanding nested `i18n.field`
   *
   * @param {Schema} schema - component schema
   * @param {DOMStringMap} dataset - HTML element dataset
   * @returns {object} Normalised dataset
   */
  Alert.prototype.parseDataset = function (schema, dataset) {
    const parsed = {};

    for (const [field, property] of Object.entries(schema.properties)) {
      if (field in dataset) {
        if (dataset[field]) {
          parsed[field] = this.normaliseString(dataset[field], property);
        }
      }
    }

    return parsed
  };

  /**
   * Config merging function
   *
   * Takes any number of objects and combines them together, with
   * greatest priority on the LAST item passed in.
   *
   * @param {...{ [key: string]: unknown }} configObjects - Config objects to merge
   * @returns {{ [key: string]: unknown }} A merged config object
   */
  Alert.prototype.mergeConfigs = function (...configObjects) {
    const formattedConfigObject = {};

    // Loop through each of the passed objects
    for (const configObject of configObjects) {
      for (const key of Object.keys(configObject)) {
        const option = formattedConfigObject[key];
        const override = configObject[key];

        // Push their keys one-by-one into formattedConfigObject. Any duplicate
        // keys with object values will be merged, otherwise the new value will
        // override the existing value.
        if (typeof option === 'object' && typeof override === 'object') {
          // @ts-expect-error Index signature for type 'string' is missing
          formattedConfigObject[key] = this.mergeConfigs(option, override);
        } else {
          formattedConfigObject[key] = override;
        }
      }
    }

    return formattedConfigObject
  };

  /**
   * Schema for component config
   *
   * @typedef {object} Schema
   * @property {{ [field: string]: SchemaProperty | undefined }} properties - Schema properties
   */

  /**
   * Schema property for component config
   *
   * @typedef {object} SchemaProperty
   * @property {'string' | 'boolean' | 'number' | 'object'} type - Property type
   */

  exports.Alert = Alert;

}));
