function dragAndDropSupported() {
  const div = document.createElement('div');
  return typeof div.ondrop !== 'undefined'
}

function formDataSupported() {
  return typeof FormData === 'function'
}

function fileApiSupported() {
  const input = document.createElement('input');
  input.type = 'file';
  return typeof input.files !== 'undefined'
}

function nodeListForEach(nodes, callback) {
  if (window.NodeList.prototype.forEach) {
    return nodes.forEach(callback)
  }
  for (let i = 0; i < nodes.length; i++) {
    callback.call(window, nodes[i], i, nodes);
  }
}

/**
 * Find an elements preceding sibling
 *
 * Utility function to find an elements previous sibling matching the provided
 * selector.
 *
 * @param {HTMLElement} $element - Element to find siblings for
 * @param {string} selector - selector for required sibling
 */
function getPreviousSibling($element, selector) {
  if (!$element) return
  // Get the previous sibling element
  let $sibling = $element.previousElementSibling;

  // If the sibling matches our selector, use it
  // If not, jump to the next sibling and continue the loop
  while ($sibling) {
    if ($sibling.matches(selector)) return $sibling
    $sibling = $sibling.previousElementSibling;
  }
}

function findNearestMatchingElement($element, selector) {
  // If no element or selector is provided, return null
  if (!$element) return

  // Start with the current element
  let $currentElement = $element;

  while ($currentElement) {
    // First check the current element
    if ($currentElement.matches(selector)) {
      return $currentElement
    }

    // Check all previous siblings
    let $sibling = $currentElement.previousElementSibling;
    while ($sibling) {
      // Check if the sibling itself is a heading
      if ($sibling.matches(selector)) {
        return $sibling
      }
      $sibling = $sibling.previousElementSibling;
    }

    // If no match found in siblings, move up to parent
    $currentElement = $currentElement.parentElement;
  }
}

/**
 * Move focus to element
 *
 * Sets tabindex to -1 to make the element programmatically focusable,
 * but removes it on blur as the element doesn't need to be focused again.
 *
 * @param {HTMLElement} $element - HTML element
 * @param {object} [options] - Handler options
 * @param {function(this: HTMLElement): void} [options.onBeforeFocus] - Callback before focus
 * @param {function(this: HTMLElement): void} [options.onBlur] - Callback on blur
 */
function setFocus($element, options = {}) {
  const isFocusable = $element.getAttribute('tabindex');

  if (!isFocusable) {
    $element.setAttribute('tabindex', '-1');
  }

  /**
   * Handle element focus
   */
  function onFocus() {
    $element.addEventListener('blur', onBlur, { once: true });
  }

  /**
   * Handle element blur
   */
  function onBlur() {
    if (options.onBlur) {
      options.onBlur.call($element);
    }

    if (!isFocusable) {
      $element.removeAttribute('tabindex');
    }
  }

  // Add listener to reset element on blur, after focus
  $element.addEventListener('focus', onFocus, { once: true });

  // Focus element
  if (options.onBeforeFocus) {
    options.onBeforeFocus.call($element);
  }
  $element.focus();
}

export { dragAndDropSupported, fileApiSupported, findNearestMatchingElement, formDataSupported, getPreviousSibling, nodeListForEach, setFocus };
