function isInitialised($root, moduleName) {
  return $root instanceof HTMLElement && $root.hasAttribute(`data-${moduleName}-init`);
}

/**
 * Checks if GOV.UK Frontend is supported on this page
 *
 * Some browsers will load and run our JavaScript but GOV.UK Frontend
 * won't be supported.
 *
 * @param {HTMLElement | null} [$scope] - (internal) `<body>` HTML element checked for browser support
 * @returns {boolean} Whether GOV.UK Frontend is supported on this page
 */
function isSupported($scope = document.body) {
  if (!$scope) {
    return false;
  }
  return $scope.classList.contains('govuk-frontend-supported');
}
function isArray(option) {
  return Array.isArray(option);
}
function isObject(option) {
  return !!option && typeof option === 'object' && !isArray(option);
}
function formatErrorMessage(Component, message) {
  return `${Component.moduleName}: ${message}`;
}

class GOVUKFrontendError extends Error {
  constructor(...args) {
    super(...args);
    this.name = 'GOVUKFrontendError';
  }
}
class SupportError extends GOVUKFrontendError {
  /**
   * Checks if GOV.UK Frontend is supported on this page
   *
   * @param {HTMLElement | null} [$scope] - HTML element `<body>` checked for browser support
   */
  constructor($scope = document.body) {
    const supportMessage = 'noModule' in HTMLScriptElement.prototype ? 'GOV.UK Frontend initialised without `<body class="govuk-frontend-supported">` from template `<script>` snippet' : 'GOV.UK Frontend is not supported in this browser';
    super($scope ? supportMessage : 'GOV.UK Frontend initialised without `<script type="module">`');
    this.name = 'SupportError';
  }
}
class ConfigError extends GOVUKFrontendError {
  constructor(...args) {
    super(...args);
    this.name = 'ConfigError';
  }
}
class ElementError extends GOVUKFrontendError {
  constructor(messageOrOptions) {
    let message = typeof messageOrOptions === 'string' ? messageOrOptions : '';
    if (typeof messageOrOptions === 'object') {
      const {
        component,
        identifier,
        element,
        expectedType
      } = messageOrOptions;
      message = identifier;
      message += element ? ` is not of type ${expectedType != null ? expectedType : 'HTMLElement'}` : ' not found';
      message = formatErrorMessage(component, message);
    }
    super(message);
    this.name = 'ElementError';
  }
}
class InitError extends GOVUKFrontendError {
  constructor(componentOrMessage) {
    const message = typeof componentOrMessage === 'string' ? componentOrMessage : formatErrorMessage(componentOrMessage, `Root element (\`$root\`) already initialised`);
    super(message);
    this.name = 'InitError';
  }
}

class Component {
  /**
   * Returns the root element of the component
   *
   * @protected
   * @returns {RootElementType} - the root element of component
   */
  get $root() {
    return this._$root;
  }
  constructor($root) {
    this._$root = void 0;
    const childConstructor = this.constructor;
    if (typeof childConstructor.moduleName !== 'string') {
      throw new InitError(`\`moduleName\` not defined in component`);
    }
    if (!($root instanceof childConstructor.elementType)) {
      throw new ElementError({
        element: $root,
        component: childConstructor,
        identifier: 'Root element (`$root`)',
        expectedType: childConstructor.elementType.name
      });
    } else {
      this._$root = $root;
    }
    childConstructor.checkSupport();
    this.checkInitialised();
    const moduleName = childConstructor.moduleName;
    this.$root.setAttribute(`data-${moduleName}-init`, '');
  }
  checkInitialised() {
    const constructor = this.constructor;
    const moduleName = constructor.moduleName;
    if (moduleName && isInitialised(this.$root, moduleName)) {
      throw new InitError(constructor);
    }
  }
  static checkSupport() {
    if (!isSupported()) {
      throw new SupportError();
    }
  }
}

/**
 * @typedef ChildClass
 * @property {string} moduleName - The module name that'll be looked for in the DOM when initialising the component
 */

/**
 * @typedef {typeof Component & ChildClass} ChildClassConstructor
 */
Component.elementType = HTMLElement;

const configOverride = Symbol.for('configOverride');
class ConfigurableComponent extends Component {
  [configOverride](param) {
    return {};
  }

  /**
   * Returns the root element of the component
   *
   * @protected
   * @returns {ConfigurationType} - the root element of component
   */
  get config() {
    return this._config;
  }
  constructor($root, config) {
    super($root);
    this._config = void 0;
    const childConstructor = this.constructor;
    if (!isObject(childConstructor.defaults)) {
      throw new ConfigError(formatErrorMessage(childConstructor, 'Config passed as parameter into constructor but no defaults defined'));
    }
    const datasetConfig = normaliseDataset(childConstructor, this._$root.dataset);
    this._config = mergeConfigs(childConstructor.defaults, config != null ? config : {}, this[configOverride](datasetConfig), datasetConfig);
  }
}
function normaliseString(value, property) {
  const trimmedValue = value ? value.trim() : '';
  let output;
  let outputType = property == null ? void 0 : property.type;
  if (!outputType) {
    if (['true', 'false'].includes(trimmedValue)) {
      outputType = 'boolean';
    }
    if (trimmedValue.length > 0 && isFinite(Number(trimmedValue))) {
      outputType = 'number';
    }
  }
  switch (outputType) {
    case 'boolean':
      output = trimmedValue === 'true';
      break;
    case 'number':
      output = Number(trimmedValue);
      break;
    default:
      output = value;
  }
  return output;
}
function normaliseDataset(Component, dataset) {
  if (!isObject(Component.schema)) {
    throw new ConfigError(formatErrorMessage(Component, 'Config passed as parameter into constructor but no schema defined'));
  }
  const out = {};
  const entries = Object.entries(Component.schema.properties);
  for (const entry of entries) {
    const [namespace, property] = entry;
    const field = namespace.toString();
    if (field in dataset) {
      out[field] = normaliseString(dataset[field], property);
    }
    if ((property == null ? void 0 : property.type) === 'object') {
      out[field] = extractConfigByNamespace(Component.schema, dataset, namespace);
    }
  }
  return out;
}
function mergeConfigs(...configObjects) {
  const formattedConfigObject = {};
  for (const configObject of configObjects) {
    for (const key of Object.keys(configObject)) {
      const option = formattedConfigObject[key];
      const override = configObject[key];
      if (isObject(option) && isObject(override)) {
        formattedConfigObject[key] = mergeConfigs(option, override);
      } else {
        formattedConfigObject[key] = override;
      }
    }
  }
  return formattedConfigObject;
}
function extractConfigByNamespace(schema, dataset, namespace) {
  const property = schema.properties[namespace];
  if ((property == null ? void 0 : property.type) !== 'object') {
    return;
  }
  const newObject = {
    [namespace]: {}
  };
  for (const [key, value] of Object.entries(dataset)) {
    let current = newObject;
    const keyParts = key.split('.');
    for (const [index, name] of keyParts.entries()) {
      if (isObject(current)) {
        if (index < keyParts.length - 1) {
          if (!isObject(current[name])) {
            current[name] = {};
          }
          current = current[name];
        } else if (key !== namespace) {
          current[name] = normaliseString(value);
        }
      }
    }
  }
  return newObject[namespace];
}

/**
 * @augments {ConfigurableComponent<RichTextEditorConfig>}
 */
class RichTextEditor extends ConfigurableComponent {
  /**
   * @param {Element | null} $root - HTML element to use for rich text editor
   * @param {RichTextEditorConfig} config
   */
  constructor($root, config = {}) {
    super($root, config);
    if (!RichTextEditor.isSupported()) {
      return this;
    }
    const $textarea = this.$root.querySelector('.govuk-textarea');
    if (!$textarea || !($textarea instanceof HTMLTextAreaElement)) {
      return this;
    }
    this.$textarea = $textarea;
    this.createToolbar();
    this.hideDefault();
    this.configureToolbar();
    this.keys = {
      left: 37,
      right: 39,
      up: 38,
      down: 40
    };
    this.$content.addEventListener('input', this.onEditorInput.bind(this));
    this.$root.querySelector('label').addEventListener('click', this.onLabelClick.bind(this));
    this.$toolbar.addEventListener('keydown', this.onToolbarKeydown.bind(this));
  }

  /**
   * @param {KeyboardEvent} event - Click event
   */
  onToolbarKeydown(event) {
    let $focusableButton;
    switch (event.keyCode) {
      case this.keys.right:
      case this.keys.down:
        {
          $focusableButton = this.$buttons.find(button => button.getAttribute('tabindex') === '0');
          if ($focusableButton) {
            const $nextButton = $focusableButton.nextElementSibling;
            if ($nextButton && $nextButton instanceof HTMLButtonElement) {
              $nextButton.focus();
              $focusableButton.setAttribute('tabindex', '-1');
              $nextButton.setAttribute('tabindex', '0');
            }
          }
          break;
        }
      case this.keys.left:
      case this.keys.up:
        {
          $focusableButton = this.$buttons.find(button => button.getAttribute('tabindex') === '0');
          if ($focusableButton) {
            const $previousButton = $focusableButton.previousElementSibling;
            if ($previousButton && $previousButton instanceof HTMLButtonElement) {
              $previousButton.focus();
              $focusableButton.setAttribute('tabindex', '-1');
              $previousButton.setAttribute('tabindex', '0');
            }
          }
          break;
        }
    }
  }
  getToolbarHtml() {
    let html = '';
    html += '<div class="moj-rich-text-editor__toolbar" role="toolbar">';
    if (this.config.toolbar.bold) {
      html += '<button class="moj-rich-text-editor__toolbar-button moj-rich-text-editor__toolbar-button--bold" type="button" data-command="bold"><span class="govuk-visually-hidden">Bold</span></button>';
    }
    if (this.config.toolbar.italic) {
      html += '<button class="moj-rich-text-editor__toolbar-button moj-rich-text-editor__toolbar-button--italic" type="button" data-command="italic"><span class="govuk-visually-hidden">Italic</span></button>';
    }
    if (this.config.toolbar.underline) {
      html += '<button class="moj-rich-text-editor__toolbar-button moj-rich-text-editor__toolbar-button--underline" type="button" data-command="underline"><span class="govuk-visually-hidden">Underline</span></button>';
    }
    if (this.config.toolbar.bullets) {
      html += '<button class="moj-rich-text-editor__toolbar-button moj-rich-text-editor__toolbar-button--unordered-list" type="button" data-command="insertUnorderedList"><span class="govuk-visually-hidden">Unordered list</span></button>';
    }
    if (this.config.toolbar.numbers) {
      html += '<button class="moj-rich-text-editor__toolbar-button moj-rich-text-editor__toolbar-button--ordered-list" type="button" data-command="insertOrderedList"><span class="govuk-visually-hidden">Ordered list</span></button>';
    }
    html += '</div>';
    return html;
  }
  getEnhancedHtml() {
    return `${this.getToolbarHtml()}<div class="govuk-textarea moj-rich-text-editor__content" contenteditable="true" spellcheck="false"></div>`;
  }
  hideDefault() {
    this.$textarea.classList.add('govuk-visually-hidden');
    this.$textarea.setAttribute('aria-hidden', 'true');
    this.$textarea.setAttribute('tabindex', '-1');
  }
  createToolbar() {
    this.$toolbar = document.createElement('div');
    this.$toolbar.className = 'moj-rich-text-editor';
    this.$toolbar.innerHTML = this.getEnhancedHtml();
    this.$root.append(this.$toolbar);
    this.$content = /** @type {HTMLElement} */
    this.$root.querySelector('.moj-rich-text-editor__content');
    this.$content.innerHTML = this.$textarea.value;
  }
  configureToolbar() {
    this.$buttons = Array.from(/** @type {NodeListOf<HTMLButtonElement>} */
    this.$root.querySelectorAll('.moj-rich-text-editor__toolbar-button'));
    this.$buttons.forEach(($button, index) => {
      $button.setAttribute('tabindex', !index ? '0' : '-1');
      $button.addEventListener('click', this.onButtonClick.bind(this));
    });
  }

  /**
   * @param {MouseEvent} event - Click event
   */
  onButtonClick(event) {
    if (!(event.currentTarget instanceof HTMLElement)) {
      return;
    }
    document.execCommand(event.currentTarget.getAttribute('data-command'), false, undefined);
  }
  getContent() {
    return this.$content.innerHTML;
  }
  onEditorInput() {
    this.updateTextarea();
  }
  updateTextarea() {
    document.execCommand('defaultParagraphSeparator', false, 'p');
    this.$textarea.value = this.getContent();
  }

  /**
   * @param {MouseEvent} event - Click event
   */
  onLabelClick(event) {
    event.preventDefault();
    this.$content.focus();
  }
  static isSupported() {
    return 'contentEditable' in document.documentElement;
  }

  /**
   * Name for the component used when initialising using data-module attributes.
   */
}

/**
 * Rich text editor config
 *
 * @typedef {object} RichTextEditorConfig
 * @property {RichTextEditorToolbar} [toolbar] - Toolbar options
 */

/**
 * Rich text editor toolbar options
 *
 * @typedef {object} RichTextEditorToolbar
 * @property {boolean} [bold] - Show the bold button
 * @property {boolean} [italic] - Show the italic button
 * @property {boolean} [underline] - Show the underline button
 * @property {boolean} [bullets] - Show the bullets button
 * @property {boolean} [numbers] - Show the numbers button
 */

/**
 * @import { Schema } from 'govuk-frontend/dist/govuk/common/configuration.mjs'
 */
RichTextEditor.moduleName = 'moj-rich-text-editor';
/**
 * Rich text editor config
 *
 * @type {RichTextEditorConfig}
 */
RichTextEditor.defaults = Object.freeze({
  toolbar: {
    bold: false,
    italic: false,
    underline: false,
    bullets: true,
    numbers: true
  }
});
/**
 * Rich text editor config schema
 *
 * @satisfies {Schema<RichTextEditorConfig>}
 */
RichTextEditor.schema = Object.freeze(/** @type {const} */{
  properties: {
    toolbar: {
      type: 'object'
    }
  }
});

export { RichTextEditor };
//# sourceMappingURL=rich-text-editor.bundle.mjs.map
