"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var m = require("mithril");
var utils_1 = require("./utils");
var TIMEOUT_DELAY = 17;
var TransitionState;
(function (TransitionState) {
    TransitionState["UNMOUNTED"] = "unmounted";
    TransitionState["EXITED"] = "exited";
    TransitionState["ENTERING"] = "entering";
    TransitionState["ENTERED"] = "entered";
    TransitionState["EXITING"] = "exiting";
})(TransitionState = exports.TransitionState || (exports.TransitionState = {}));
var Transition = /** @class */ (function () {
    function Transition() {
        var _this = this;
        this.status = TransitionState.UNMOUNTED;
        this.nextStatus = null;
        this.node = null;
        this.timeoutStack = [];
        this.clearTimeouts = function () {
            if (_this.timeoutStack.length) {
                _this.timeoutStack.map(function (timeout) { return clearTimeout(timeout); });
                _this.timeoutStack = [];
            }
        };
    }
    Transition.prototype.oninit = function (_a) {
        var attrs = _a.attrs;
        if (attrs.isVisible) {
            this.status = TransitionState.EXITED;
            this.nextStatus = TransitionState.ENTERING;
        }
    };
    Transition.prototype.oncreate = function (_a) {
        var attrs = _a.attrs, dom = _a.dom;
        this.node = dom;
        this.updateStatus(attrs);
    };
    Transition.prototype.onupdate = function (_a) {
        var attrs = _a.attrs, dom = _a.dom;
        this.node = dom;
        this.updateStatus(attrs);
    };
    Transition.prototype.onbeforeupdate = function (vnode, old) {
        var isVisible = vnode.attrs.isVisible;
        if (isVisible && this.status === TransitionState.UNMOUNTED) {
            this.status = TransitionState.EXITED;
            this.nextStatus = TransitionState.ENTERING;
        }
        else if (isVisible && !old.attrs.isVisible) {
            this.clearTimeouts();
            this.nextStatus = TransitionState.ENTERING;
        }
        else if (!isVisible && old.attrs.isVisible) {
            this.clearTimeouts();
            this.nextStatus = TransitionState.EXITING;
        }
    };
    Transition.prototype.onbeforeremove = function () {
        this.clearTimeouts();
    };
    Transition.prototype.view = function (_a) {
        var attrs = _a.attrs;
        if (this.status === TransitionState.UNMOUNTED) {
            return null;
        }
        if (typeof (attrs.content) === 'function') {
            return attrs.content(this.status);
        }
        return attrs.content;
    };
    Transition.prototype.getTimeouts = function (attrs) {
        var timeout = attrs.timeout;
        // tslint:disable-next-line:one-variable-per-declaration
        var enter, exit;
        exit = enter = timeout;
        if (timeout !== null && typeof timeout !== 'number') {
            enter = timeout.enter;
            exit = timeout.exit;
        }
        return { enter: enter, exit: exit };
    };
    Transition.prototype.updateStatus = function (attrs, mounting) {
        if (mounting === void 0) { mounting = false; }
        if (this.nextStatus === TransitionState.ENTERING) {
            this.performEnter(attrs);
        }
        else if (this.nextStatus === TransitionState.EXITING) {
            this.performExit(attrs);
        }
        else if (this.nextStatus === TransitionState.UNMOUNTED) {
            this.performUnmount();
        }
    };
    Transition.prototype.performEnter = function (attrs) {
        var _this = this;
        var timeouts = this.getTimeouts(attrs);
        utils_1.safeCall(attrs.onEnter, this.node);
        this.setTimeout(function () {
            _this.status = TransitionState.ENTERING;
            _this.nextStatus = TransitionState.ENTERED;
            m.redraw();
            utils_1.safeCall(attrs.onEntering, _this.node);
        }, TIMEOUT_DELAY);
        this.setTimeout(function () {
            _this.status = TransitionState.ENTERED;
            _this.nextStatus = null;
            m.redraw();
            utils_1.safeCall(attrs.onEntered, _this.node);
        }, timeouts.enter + TIMEOUT_DELAY);
    };
    Transition.prototype.performExit = function (attrs) {
        var _this = this;
        var timeouts = this.getTimeouts(attrs);
        utils_1.safeCall(attrs.onExit, this.node);
        this.setTimeout(function () {
            _this.status = TransitionState.EXITING;
            _this.nextStatus = TransitionState.EXITED;
            m.redraw();
            utils_1.safeCall(attrs.onExiting, _this.node);
        }, TIMEOUT_DELAY);
        this.setTimeout(function () {
            _this.status = TransitionState.EXITED;
            _this.nextStatus = TransitionState.UNMOUNTED;
            m.redraw();
            utils_1.safeCall(attrs.onExited, _this.node);
        }, timeouts.exit + TIMEOUT_DELAY);
    };
    Transition.prototype.performUnmount = function () {
        this.status = TransitionState.UNMOUNTED;
        this.nextStatus = null;
        m.redraw();
    };
    Transition.prototype.setTimeout = function (callback, timeout) {
        var handle = window.setTimeout(callback, timeout);
        this.timeoutStack.push(handle);
        return function () { return clearTimeout(handle); };
    };
    return Transition;
}());
exports.Transition = Transition;
