"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.readMjmlConfig = readMjmlConfig;
exports.resolveComponentPath = resolveComponentPath;
exports.registerCustomComponent = registerCustomComponent;
exports.handleMjmlConfigComponents = handleMjmlConfigComponents;
exports["default"] = handleMjmlConfig;

var _path = _interopRequireDefault(require("path"));

var _fs = _interopRequireDefault(require("fs"));

var _mjmlValidator = require("mjml-validator");

var _components = require("../components");

function readMjmlConfig() {
  var configPathOrDir = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : process.cwd();
  var componentRootPath = process.cwd();
  var mjmlConfigPath = configPathOrDir;

  try {
    mjmlConfigPath = _path["default"].basename(configPathOrDir) === '.mjmlconfig' ? _path["default"].resolve(configPathOrDir) : _path["default"].resolve(configPathOrDir, '.mjmlconfig');
    componentRootPath = _path["default"].dirname(mjmlConfigPath);
    var mjmlConfig = JSON.parse(_fs["default"].readFileSync(_path["default"].resolve(mjmlConfigPath), 'utf8'));
    return {
      mjmlConfig: mjmlConfig,
      componentRootPath: componentRootPath
    };
  } catch (e) {
    if (e.code !== 'ENOENT') {
      console.error('Error reading mjmlconfig : ', e); // eslint-disable-line no-console
    }

    return {
      mjmlConfig: {
        packages: [],
        options: {}
      },
      mjmlConfigPath: mjmlConfigPath,
      componentRootPath: componentRootPath,
      error: e
    };
  }
}

function resolveComponentPath(compPath, componentRootPath) {
  if (!compPath) {
    return null;
  }

  if (!compPath.startsWith('.') && !_path["default"].isAbsolute(compPath)) {
    try {
      return require.resolve(compPath);
    } catch (e) {
      if (e.code !== 'MODULE_NOT_FOUND') {
        console.error('Error resolving custom component path : ', e); // eslint-disable-line no-console

        return null;
      } // we got a 'MODULE_NOT_FOUND' error


      try {
        // try again as relative path to node_modules: (this may be necessary if mjml is installed globally or by npm link)
        return resolveComponentPath("./node_modules/".concat(compPath), componentRootPath);
      } catch (e) {
        //  try again as a plain local path:
        return resolveComponentPath("./".concat(compPath), componentRootPath);
      }
    }
  }

  return require.resolve(_path["default"].resolve(componentRootPath, compPath));
}

function registerCustomComponent(comp) {
  var registerCompFn = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : _components.registerComponent;

  if (comp instanceof Function) {
    registerCompFn(comp);
  } else {
    var compNames = Object.keys(comp); // this approach handles both an array and an object (like the mjml-accordion default export)

    compNames.forEach(function (compName) {
      registerCustomComponent(comp[compName], registerCompFn);
    });
  }
}

function handleMjmlConfigComponents(packages, componentRootPath, registerCompFn) {
  var result = {
    success: [],
    failures: []
  };
  packages.forEach(function (compPath) {
    var resolvedPath = compPath;

    try {
      resolvedPath = resolveComponentPath(compPath, componentRootPath);

      if (resolvedPath) {
        var requiredComp = require(resolvedPath); // eslint-disable-line global-require, import/no-dynamic-require


        registerCustomComponent(requiredComp["default"] || requiredComp, registerCompFn);
        (0, _mjmlValidator.registerDependencies)((requiredComp["default"] || requiredComp).dependencies);
        result.success.push(compPath);
      }
    } catch (e) {
      result.failures.push({
        error: e,
        compPath: compPath
      });

      if (e.code === 'ENOENT' || e.code === 'MODULE_NOT_FOUND') {
        console.error('Missing or unreadable custom component : ', resolvedPath); // eslint-disable-line no-console
      } else {
        // eslint-disable-next-line no-console
        console.error('Error when registering custom component : ', resolvedPath, e);
      }
    }
  });
  return result;
}

function handleMjmlConfig() {
  var configPathOrDir = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : process.cwd();
  var registerCompFn = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : _components.registerComponent;

  var _readMjmlConfig = readMjmlConfig(configPathOrDir),
      packages = _readMjmlConfig.mjmlConfig.packages,
      componentRootPath = _readMjmlConfig.componentRootPath,
      error = _readMjmlConfig.error;

  if (error) return {
    error: error
  };
  return handleMjmlConfigComponents(packages, componentRootPath, registerCompFn);
}