import Hammer from './utils/hammer';
import {RecognizerTuple} from './types';

// This module contains constants that must be conditionally required
// due to `window`/`document` references downstream.
export const RECOGNIZERS: RecognizerTuple[] = Hammer
  ? [
      [Hammer.Pan, {event: 'tripan', pointers: 3, threshold: 0, enable: false}],
      [Hammer.Rotate, {enable: false}],
      [Hammer.Pinch, {enable: false}],
      [Hammer.Swipe, {enable: false}],
      [Hammer.Pan, {threshold: 0, enable: false}],
      [Hammer.Press, {enable: false}],
      [Hammer.Tap, {event: 'doubletap', taps: 2, enable: false}],
      // TODO - rename to 'tap' and 'singletap' in the next major release
      [Hammer.Tap, {event: 'anytap', enable: false}],
      [Hammer.Tap, {enable: false}]
    ]
  : null;

// Recognize the following gestures even if a given recognizer succeeds
export const RECOGNIZER_COMPATIBLE_MAP = {
  tripan: ['rotate', 'pinch', 'pan'],
  rotate: ['pinch'],
  pinch: ['pan'],
  pan: ['press', 'doubletap', 'anytap', 'tap'],
  doubletap: ['anytap'],
  anytap: ['tap']
} as const;

// Recognize the folling gestures only if a given recognizer fails
export const RECOGNIZER_FALLBACK_MAP = {
  doubletap: ['tap']
} as const;

/**
 * Only one set of basic input events will be fired by Hammer.js:
 * either pointer, touch, or mouse, depending on system support.
 * In order to enable an application to be agnostic of system support,
 * alias basic input events into "classes" of events: down, move, and up.
 * See `_onBasicInput()` for usage of these aliases.
 */
export const BASIC_EVENT_ALIASES = {
  pointerdown: 'pointerdown',
  pointermove: 'pointermove',
  pointerup: 'pointerup',
  touchstart: 'pointerdown',
  touchmove: 'pointermove',
  touchend: 'pointerup',
  mousedown: 'pointerdown',
  mousemove: 'pointermove',
  mouseup: 'pointerup'
} as const;

export const INPUT_EVENT_TYPES = {
  KEY_EVENTS: ['keydown', 'keyup'],
  MOUSE_EVENTS: ['mousedown', 'mousemove', 'mouseup', 'mouseover', 'mouseout', 'mouseleave'],
  WHEEL_EVENTS: [
    // Chrome, Safari
    'wheel',
    // IE
    'mousewheel'
  ]
} as const;

/**
 * "Gestural" events are those that have semantic meaning beyond the basic input event,
 * e.g. a click or tap is a sequence of `down` and `up` events with no `move` event in between.
 * Hammer.js handles these with its Recognizer system;
 * this block maps event names to the Recognizers required to detect the events.
 */
export const EVENT_RECOGNIZER_MAP = {
  tap: 'tap',
  anytap: 'anytap',
  doubletap: 'doubletap',
  press: 'press',
  pinch: 'pinch',
  pinchin: 'pinch',
  pinchout: 'pinch',
  pinchstart: 'pinch',
  pinchmove: 'pinch',
  pinchend: 'pinch',
  pinchcancel: 'pinch',
  rotate: 'rotate',
  rotatestart: 'rotate',
  rotatemove: 'rotate',
  rotateend: 'rotate',
  rotatecancel: 'rotate',
  tripan: 'tripan',
  tripanstart: 'tripan',
  tripanmove: 'tripan',
  tripanup: 'tripan',
  tripandown: 'tripan',
  tripanleft: 'tripan',
  tripanright: 'tripan',
  tripanend: 'tripan',
  tripancancel: 'tripan',
  pan: 'pan',
  panstart: 'pan',
  panmove: 'pan',
  panup: 'pan',
  pandown: 'pan',
  panleft: 'pan',
  panright: 'pan',
  panend: 'pan',
  pancancel: 'pan',
  swipe: 'swipe',
  swipeleft: 'swipe',
  swiperight: 'swipe',
  swipeup: 'swipe',
  swipedown: 'swipe'
} as const;

/**
 * Map gestural events typically provided by browsers
 * that are not reported in 'hammer.input' events
 * to corresponding Hammer.js gestures.
 */
export const GESTURE_EVENT_ALIASES = {
  click: 'tap',
  anyclick: 'anytap',
  dblclick: 'doubletap',
  mousedown: 'pointerdown',
  mousemove: 'pointermove',
  mouseup: 'pointerup',
  mouseover: 'pointerover',
  mouseout: 'pointerout',
  mouseleave: 'pointerleave'
} as const;
